import logging

logger = logging.getLogger(__name__)

class Led:
    LED_ON = '255'
    LED_OFF = '0'
    BASE_PATH = '/sys/class/leds/'

    def __init__(self, ledName: str):
        self.ledPaths = {
            'brightness': self.BASE_PATH + ledName + '/brightness',
            'trigger'   : self.BASE_PATH + ledName + '/trigger',
            'delay_on'  : self.BASE_PATH + ledName + '/delay_on',
            'delay_off' : self.BASE_PATH + ledName + '/delay_off'
        }

    def setValue(self, path:str, value:str) -> None:
        try:
            with open(self.ledPaths[path], 'w') as f:
                f.write(str(value))
        except OSError as e:
            logger.error('Error setting LEDS path {} to value {}: {}'.format(path, value, e))

    def setBrigtness(self,level:float) -> None:
        intLv   = int(level*255)
        if intLv<0:
            intLv   = 0
        elif intLv>255:
            intLv   = 255

        self.setValue('brightness',str(intLv))

    def set(self,on:bool) -> None:
        if on:
            self.setBrigtness(1.0)
        else:
            self.setBrigtness(0)


class IpamLed:
    '''
    Module to implement the led operations (only on IPAM400 for now).
    It supports the functions Red, Green, Yellow and Off. Every collor
    could also be set to blink.
    '''
    LEDS_NAME = ('led1', 'led2')
    LEDS_COLOR = ('off', 'red', 'green', 'yellow')

    def __init__(self, ledName:str):
        '''
        Instantiate a IPAM led object.

        :param str ledName: Led name. Supported values are in LEDS_NAME tuple.

        Raises:
            ValueError: if 'ledName' value is invalid
        '''

        if ledName not in self.LEDS_NAME:
            raise ValueError('Value for ledName not supported')

        self.led = {
                'red': Led('barix:' + ledName + ':red'),
                'green': Led('barix:' + ledName + ':green')
            }

    def __setBlink(self, led:Led, blink:bool, timeOn:int, timeOff:int) -> None:
        trigger = 'none'
        if blink:
            trigger = 'timer'
        led.setValue('trigger', trigger)
        if blink:
            led.setValue('delay_on', timeOn)
            led.setValue('delay_off', timeOff)

    def set(self, color:str, blink:bool=False, blinkTime:tuple=(1000, 1000)) -> None:
        '''
        Change the state of the led.

        :param str color: Led color. Supported values are in LEDS_COLOR tuple.
        :param bool blink: If true, the led will start blinking.
        :param tuple blinkTime: A tuple with the blinking times in milliseconds.
            First position in the tuple is the 'on' time. Second position is for
            the 'off' time.

        Raises:
            ValueError: if 'color' value is invalid
            TypeError: if 'blinkTime' is not a tuple
        '''
        if color not in self.LEDS_COLOR:
            raise ValueError('Leds color not supported')

        if not isinstance(blinkTime, tuple):
            raise TypeError('blinkTime is not a tuple')

        self.led['red'].setValue('brightness', Led.LED_OFF)
        self.led['green'].setValue('brightness', Led.LED_OFF)

        if color == 'yellow':
            self.__setBlink(self.led['red'], blink, blinkTime[0], blinkTime[1])
            self.__setBlink(self.led['green'], blink, blinkTime[0], blinkTime[1])
            self.led['red'].setValue('brightness', Led.LED_ON)
            self.led['green'].setValue('brightness', Led.LED_ON)
        elif color != 'off':
            self.__setBlink(self.led[color], blink, blinkTime[0], blinkTime[1])
            self.led[color].setValue('brightness', Led.LED_ON)

