import logging
import uuid
from struct import pack, unpack

from BARP.constants import FORMATS

from enum import IntEnum


class SimplePagingDevice(IntEnum):
    UNKNOWN = 0
    MASTER = 1
    ENDPOINT = 2
    # TODO: check the value type of gateway.
    GATEWAY = 3

    @staticmethod
    def is_valid_device(value):
        return True if value in [1, 2, 3] else False


class SimplePagingUtils:
    """
    Utility class for unpacking(conversion to respective datatype) and
    packing(conversion to byte array) components of BARP header for Simple Paging.
    """

    station_name_idx = slice(6, 21)
    device_type_idx = slice(21, 22)
    mac_idx = slice(28, 34)

    @staticmethod
    def pack_station_name(station_name):
        """Converts station name into a byte array.

        Args:
            station_name (str): name of station/group name.
            Only first 15 characters processed.

        Returns:
            bytes: byte array of length 15
        """
        if station_name == "":
            return pack(FORMATS.BYTE_FMT * 15, *[0] * 15)
        else:
            len_station = len(station_name)
            if len_station < 15:
                # filling the rest of buffer with zeroes.
                empty_buffer_len = 15 - len_station
                return station_name.encode() + pack(
                    FORMATS.BYTE_FMT * empty_buffer_len, *[0] * empty_buffer_len
                )
            else:
                return station_name[:15].encode()

    @staticmethod
    def unpack_station_name(station_name):
        """Converts byte array containing station name to string.
        NOTE: only ascii are converted and trailling \x00 are removed.

        Args:
            station_name (bytes): station-name/group-name bytes from BARP header.

        Returns:
            str: station_name if decoding possible else "".
        """
        try:
            station_name_ = station_name.decode("ascii", errors="ignore")
            station_name_.split("\x00")[0]
            return station_name_.split("\x00")[0]
        except Exception as e:
            logging.warning("{}".format(e))
            return ""

    @staticmethod
    def unpack_mac(mac):
        """
        Unpacks the mac in barp header to the string representation.

        Args:
            mac (bytes): Mac bytes from BARP header.

        Returns:
            str: MAC address.
        """
        return ":".join(
            ["{:02x}".format(ch) for ch in unpack(FORMATS.BYTE_FMT * 6, mac)]
        )

    @staticmethod
    def pack_mac():
        """
        Returns the mac address of the device in byte format.
        mac address: 00:08:e1:08:06:23 is returned as b'\x00\x08\xe1\x08\x06\x23'

        Returns:
            bytes: mac in byte array
        """
        mac = [(uuid.getnode() >> ele) & 0xFF for ele in range(0, 8 * 6, 8)][::-1]
        return pack(FORMATS.BYTE_FMT * 6, *mac)

    @staticmethod
    def unpack_device_type(device_type):
        """
        Unpacks and returns the device type in string format.

        Args:
            device_type (bytes): A single byte representing the device type.

        Returns:
            SimplePagingDevice: Enum of the simple paging device.
        """
        device_type = unpack(FORMATS.BYTE_FMT, device_type)[0]
        if SimplePagingDevice.is_valid_device(device_type):
            return SimplePagingDevice(device_type)
        else:
            return SimplePagingDevice.UNKNOWN

    @staticmethod
    def pack_device_type(device_type):
        """Packs the device type in BARP compatible bytes.

        Args:
            device_type (Union[SimplePagingDevice, int]): device type valid (0,1,2)

        Returns:
            bytes: Device type as a byte.
        """
        if SimplePagingDevice.is_valid_device(device_type):
            return pack(FORMATS.BYTE_FMT, SimplePagingDevice(device_type).value)
        else:
            return pack(FORMATS.BYTE_FMT, SimplePagingDevice.UNKNOWN.value)

    @staticmethod
    def is_simple_paging_header(barp_header):
        """Detects whether the header is potentially from simple paging BARP or
        not based on presense of mac address

        Args:
            barp_header (bytes): Complete BARP header.

        Returns:
            bool: True if header belongs to simple paging else False.
        """
        mac = barp_header[SimplePagingUtils.mac_idx]
        mac = unpack(FORMATS.BYTE_FMT * 6, mac)
        if max(mac) == min(mac) and max(mac) == 0:
            # mac address is "00:00:00:00:00:00", this belongs to xerox corporation, NY
            # therefore it can't be mac of any barix device.
            return False
        else:
            return True
