#!/bin/sh

# Barix Log utilities.
# Copyright (C) 2018 Barix AG (http://www.barix.com). All rights reserved.
#
# Dependencies from 'qiba-exstreamerl.sh' main script:
#  Includes
#   NONE
#  Variables
#   NONE

# The log messages prefix.
# Allowed to be specified externally.
[ -z "${LOG_PFX}" ] && LOG_PFX="qiba-log"

# Log a message to the console and syslog.
# It supports multiple line message, when lines are separated by the 'newline'
# character. Saves current 'IFS' context and restores it on return.
#
# Parameters:
#  $1: The severity level.
#  $2: The function name.
#  $@: The string to log.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
log_msg() {
	local level="${1}" && shift
	local func="${1}" && shift
	local str="$@"

	local prev_ifs

	# Save 'IFS' to restore later. Change it to the newline character to support
	# multiple lines in the message.
	prev_ifs=$IFS
	IFS="
	"

	# Log an entry for each line received.
	for line in ${str}; do
		echo "${LOG_PFX}: ${level}: ${func}: ${line}"
		logger -t "${LOG_PFX}" -p "${level}" "${func}: ${line}"
	done

	# Restore 'IFS'.
	IFS=${prev_ifs}

	return 0
}

# Log an error message.
#
# Parameters:
#  $1: The function name.
#  $@: The string to log.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
log_err() {
	local func="${1}" && shift
	local str="$@"

	log_msg "err" "${func}" "${str}" 2>/dev/null || return $?

	return 0
}

# Log an info message.
#
# Parameters:
#  $1: The function name.
#  $@: The string to log.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
log_inf() {
	local func="${1}" && shift
	local str="$@"

	log_msg "info" "${func}" "${str}" 2>/dev/null || return $?

	return 0
}
