import json
import logging
import os
from secrets import token_urlsafe
from restart_services import restartApp

USERS_FILE_PATH = "/barix/apps/icpgw/users_backup.json"

logger = logging.getLogger('flask-backend')

ALL_USER_IDS = set([str(i) for i in range(1, 51)])


def get_users():
    # read list of users from file
    if not os.path.isfile(USERS_FILE_PATH):
        return {}

    with open(USERS_FILE_PATH, 'r', encoding='utf-8') as file:
        return json.load(file)


def save_users(users):
    with open(USERS_FILE_PATH, 'w', encoding='utf-8') as file:
        json.dump(users, file)


def save_users_and_restart(users):
    save_users(users)
    restartApp()


def new_secret() -> str:
    return token_urlsafe(16)


def get_available_id(users) -> int | None:
    available_str_ids = ALL_USER_IDS - set(users.keys())
    if len(available_str_ids) != 0:
        return min(int(x) for x in available_str_ids)
    else:
        return None


def new_user(user) -> str | None:
    users = get_users()
    new_id = get_available_id(users)
    if new_id is not None:
        users[str(new_id)] = {"name": user["name"], "email": user["email"], "secret": new_secret(), "audio_files": user["audio_files"]}
        save_users_and_restart(users)
    return str(new_id)


def update_user(user) -> bool:
    users = get_users()
    user_id = str(user["id"])
    if user_id in users:
        users[user_id]["name"] = user["name"]
        users[user_id]["email"] = user["email"]
        # assign audio files to users
        if user_id == "0":
            users[user_id]["audio_files"] = []
        else:
            users[user_id]["audio_files"] = user["audio_files"]
        save_users_and_restart(users)
        return True
    else:
        logger.error(f"User id {user['id']} doesn't exist!")
        return False


def remove_users(user_ids) -> list[str]:
    users = get_users()
    users_not_removed = []
    for uid in user_ids:
        user_id = str(uid)
        if user_id in users and user_id != "0":
            del users[user_id]
            logger.info(f"User {user_id} deletion successful!")
        if user_id == "0":
            logger.info(f"Admin user can't be deleted!")
        else:
            users_not_removed.append(user_id)
            logger.warning(f"User id {user_id} doesn't exist!")
    save_users_and_restart(users)
    return users_not_removed
