import logging
import json

from flask import Blueprint, Response, request
from pathlib import Path

from barix.web.auth import login_required
from barix.web.certificates import getCACertificates, uploadCACertificate, addCACertificates, removeCACertificates
from barix.web.constants import CA_CERTIFICATES_DIR
from barix.web.exceptions import InvalidUploadedFileError

bp = Blueprint('ca_certificates_API', __name__)

logger = logging.getLogger('flask-backend')

p = Path(CA_CERTIFICATES_DIR)
if not p.exists():
    try:
        p.mkdir()
        #logger.debug("Directory {} successfully created.".format(CA_CERTIFICATES_DIR))
    except Exception as e:
        logger.error("Error creating directory {}: {}".format(CA_CERTIFICATES_DIR, e))

@bp.route('/api/rest/v1/cacerts', methods=['GET'])
@login_required
def handleGetCACertificates():
    try:
        certificateList = getCACertificates()
        return_dict = {"custom": certificateList}
        return Response(json.dumps(return_dict), status=200)
    except Exception as e:
        logger.error(e, exc_info=True)
        return Response(status=500)


@bp.route('/api/rest/v1/cacerts', methods=['POST'])
@login_required
def handleUploadCACertificate():
    try:
        res = uploadCACertificate(request.files)
    except InvalidUploadedFileError as e:
        logger.error(e, exc_info=True)
        return Response(str(e), status=400)
    except Exception as e:
        logger.error(e, exc_info=True)
        if "No space left on device" in str(e):
            return Response(status=507)
        return Response(status=500)
    else:
        return Response(res, status=200)


@bp.route('/api/rest/v1/cacerts', methods=['DELETE'])
@login_required
def deleteCACertificates():
    try:
        data = json.loads(request.data)
    except Exception as e:
        logger.error(e, exc_info=True)
        return Response(status=400)
    else:
        try:
            res = removeCACertificates(data['payload'])
            return Response(json.dumps(res), status=200)
        except Exception as e:
            logger.error(e, exc_info=True)
            return Response(status=500)
