import logging
from datetime import datetime
# import pathlib
import json
import socket
import os
from os import system
from subprocess import Popen, PIPE, TimeoutExpired
# from inspect import currentframe

from barix.web.uci import getValueOfUci, setUciConfigs

import logging

logger = logging.getLogger('flask-backend')

IF_CONFIG_PATH = '/var/tmp/'
IF_CONFIG_FILE = 'ifconfig.settings'


def get_active_devices(barp, listen_status_socket, pmaster, active_devices):
    while not pmaster.exit_flag:
        try:
            data, address = listen_status_socket.recvfrom(4096)
        except socket.timeout:
            continue
        msg = barp.decode(data)
        if msg[0]["message_type"] == 128:
            active_devices[msg[0]["sender_id"]] = {
                "state": msg[1]["state"],
                "time_stamp": datetime.now(),
            }

        # Removal of devices with stale status.
        current_time = datetime.now()
        for dev_id in list(active_devices.keys()):
            if (current_time - active_devices[dev_id]["time_stamp"]).seconds > 5:
                del active_devices[dev_id]
    logger.warning("Status listening stopped!")


def read_json(file_name):
    with open(file_name, "r") as f:
        return json.load(f)


def is_multicast(ip):
    """
    Performs check on ipv4 ip address to determine if it is multicast or not.
    Note: it is assumed that the address is valid.
    Args:
        ip (str)
    Returns:
        (bool) : True if address is multicast and false if it isn't.
    """
    ip_address = [int(i) for i in ip.split(".")]
    if 224 <= ip_address[0] <= 239:
        return True
    else:
        return False


def is_broadcast(multicast_ip, broadcast_ip):
    """
    Performs check on ipv4 ip address to determine if it is broadcast or not.
    Note: it is assumed that the address is valid.
    Args:
        multicast_ip (str)
        broadcast_ip (str)
    Returns:
        (bool) : True if address is broadcast and false if it isn't.
    """
    if multicast_ip == "0.0.0.0":
        return True
    elif multicast_ip == "255.255.255.255":
        return True
    elif multicast_ip == broadcast_ip:
        return True
    else:
        return False


def get_uci(package, section, option):
    command = ["uci", "get", f"{package}.{section}.{option}"]
    output, error = Popen(command, stdout=PIPE, stderr=PIPE).communicate()
    if len(error) == 0:
        return output.decode().strip("\n")
    else:
        raise f"Unable to get {package}.{section}.{option} uci setting : {error}"


def get_pregong_filename():
    directory = "/mnt/data/pregong/"
    file_list = os.listdir(directory)
    for f in file_list:
        if os.path.isfile(os.path.join(directory, f)):
            return os.path.join(directory, f)
    return ""

def get_ip():
    try:
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.connect(("8.8.8.8", 80))
        ip = s.getsockname()[0]
        s.close()
        return ip
    except Exception as e:
        logger.error(f"Couldn't get the ip - {e}")
        return "Not found"


def load_alsa_settings(system_func: bool = True):
    logger.info("Loading alsa settings")
    command = [
        'alsactl',
        'restore',
    ]
    if system_func:
        system(" ".join(command))
        return
    return execute_command(command)


def store_alsa_settings(system_func: bool = True):
    logger.info("Storing alsa settings")
    command = [
        'alsactl',
        'store',
    ]
    if system_func:
        system(" ".join(command))
        return
    return execute_command(command)


def execute_command(command: list):
    """
        Open a process to execute the given command
    @param command: List of strings
    @return: True if successful, False otherwise
    """

    process = Popen(
        command,
        stdout=PIPE,
        stderr=PIPE,
    )
    try:
        # logger.info("Executing command: %s", " ".join(command))
        outs, errs = process.communicate(timeout=5)

        if process.returncode != 0:
            logger.error(f"Error occurred: {errs}")
            return False
        # logger.info("Success")
        return True

    except Exception as e:
        logger.warning(f"While trying to execute command: [{command}], the following exception occurred: {e}")
        return False


def check_uci_icpgw():
    ret_val = True

    try:
        try:
            config = getValueOfUci('icpgw')
        except Exception as e:
            config = {}
            logger.error(f"error {e}")

        new_config = {}

        if config is None or "barp" not in config:
            new_config['icpgw.barp.ip'] = '0.0.0.0'
            new_config['icpgw.barp.master_id'] = '1005'
            new_config['icpgw.barp.paging_priority'] = '127'
            new_config['icpgw.barp.audio_multicast_group_address'] = '225.1.2.3'
            new_config['icpgw.barp.audio_port'] = '5555'
            new_config['icpgw.barp.control_port'] = '5556'
            new_config['icpgw.barp.status_port'] = '5557'
            new_config['icpgw.barp.sample_rate'] = '8000'
            new_config['icpgw.barp.format'] = 'ulaw'
            new_config['icpgw.barp.flags'] = '12'
            new_config['icpgw.barp.sequence_number'] = '24'
            ret_val = False
        else:
            if "ip" not in config['barp']:
                new_config['icpgw.barp.ip'] = '0.0.0.0'
                ret_val = False
            if "master_id" not in config['barp']:
                new_config['icpgw.barp.master_id'] = '1005'
                ret_val = False
            if "paging_priority" not in config['barp']:
                new_config['icpgw.barp.paging_priority'] = '127'
                ret_val = False
            if "audio_multicast_group_address" not in config['barp']:
                new_config['icpgw.barp.audio_multicast_group_address'] = '225.1.2.3'
                ret_val = False
            if "audio_port" not in config['barp']:
                new_config['icpgw.barp.audio_port'] = '5555'
                ret_val = False
            if "control_port" not in config['barp']:
                new_config['icpgw.barp.control_port'] = '5556'
                ret_val = False
            if "status_port" not in config['barp']:
                new_config['icpgw.barp.status_port'] = '5557'
                ret_val = False
            if "sample_rate" not in config['barp']:
                new_config['icpgw.barp.sample_rate'] = '8000'
                ret_val = False
            if "format" not in config['barp']:
                new_config['icpgw.barp.format'] = 'ulaw'
                ret_val = False
            if "flags" not in config['barp']:
                new_config['icpgw.barp.flags'] = '12'
                ret_val = False
            if "sequence_number" not in config['barp']:
                new_config['icpgw.barp.sequence_number'] = '24'
                ret_val = False

        if config is None or "server" not in config:
            new_config['icpgw.server.playback_device'] = 'plug:master1'
            new_config['icpgw.server.timeout'] = '300'
            ret_val = False
        else:
            if "playback_device" not in config['server']:
                new_config['icpgw.server.playback_device'] = 'plug:master1'
                ret_val = False
            if "timeout" not in config['server']:
                new_config['icpgw.server.timeout'] = '300'
                ret_val = False

        if config is None or "rtp" not in config:
            new_config['icpgw.rtp.ip'] = '0.0.0.0'
            new_config['icpgw.rtp.port'] = '1234'
            new_config['icpgw.rtp.format'] = 'PCM16BE'
            new_config['icpgw.rtp.enabled'] = 'False'
            ret_val = False
        else:
            if "ip" not in config['rtp']:
                new_config['icpgw.rtp.ip'] = '0.0.0.0'
                ret_val = False
            if "port" not in config['rtp']:
                new_config['icpgw.rtp.port'] = '1234'
                ret_val = False
            if "format" not in config['rtp']:
                new_config['icpgw.rtp.format'] = 'PCM16BE'
                ret_val = False
            if "enabled" not in config['rtp']:
                new_config['icpgw.rtp.enabled'] = 'False'
                ret_val = False

        if config is None or "modbus" not in config:
            new_config['icpgw.modbus.port'] = '1234'
            new_config['icpgw.modbus.enabled'] = 'False'
            ret_val = False
        else:
            if "port" not in config['modbus']:
                new_config['icpgw.modbus.port'] = '502'
                ret_val = False
            if "enabled" not in config['modbus']:
                new_config['icpgw.modbus.enabled'] = 'False'
                ret_val = False

        if config is None or "pregong" not in config:
            new_config['icpgw.pregong.enabled'] = 'False'
            ret_val = False
        else:
            if "enabled" not in config['pregong']:
                new_config['icpgw.pregong.enabled'] = 'False'
                ret_val = False

        if config is None or "relay" not in config:
            new_config['icpgw.relay.enabled'] = 'False'
            ret_val = False
        else:
            if "enabled" not in config['relay']:
                new_config['icpgw.relay.enabled'] = 'False'
                ret_val = False

        if not ret_val:
            logger.warning(f"setting default uci: {new_config}")
            try:
                setUciConfigs(new_config, True, False, None)
            except Exception as e:
                logger.error(f"error {e}")
    except Exception as e:
        logger.error(f"error {e}")
        ret_val = False

    return ret_val


def uci_to_config_dict():
    config_dict = {}

    try:
        config = getValueOfUci('icpgw')

        config_dict["ports"] = {
            "audio": int(config['barp']['audio_port']),
            "control": int(config['barp']['control_port']),
            "status": int(config['barp']['status_port']),
        }

        config_dict["sample_rate"] = int(config['barp']['sample_rate'])
        config_dict["multicast_group_address"] = config['barp']['audio_multicast_group_address']
        config_dict["audio_device"] = {
            "playback": config['server']['playback_device'],
            "record": "plug:analog_out",
        }

        config_dict["encoding_type"] = 0
        config_dict["paging_master_id"] = int(config['barp']['master_id'])
        config_dict["barp_ip"] = config['barp']['ip']
        config_dict["barp_header_kwargs"] = {
            "flags": int(config['barp']['flags']),
            "sender_id": int(config['barp']['master_id']),
            "receiver_id": 0,
            "sequence_number": int(config['barp']['sequence_number'])
        }
        config_dict["audio_format"] = config['barp']['format']
        config_dict["browser_timeout"] = int(config['server']['timeout'])
        config_dict["paging_priority"] = config['barp']['paging_priority']

        # Returns this kind of dict:
        # {'inet': '192.168.10.65', 'netmask': '255.255.255.0', 'broadcast': '192.168.10.255'}
        config_dict["device_ip_settings"] = load_ip_config()

        config_dict["rtp"] = {}
        config_dict["rtp"]["enabled"] = True if config['rtp']['enabled'] == "True" else False
        config_dict["rtp"]["port"] = int(config['rtp']['port'])
        config_dict["rtp"]["ip"] = config['rtp']['ip']
        config_dict["rtp"]["format"] = config['rtp']['format']

        config_dict["modbus"] = {}
        config_dict["modbus"]["enabled"] = True if config['modbus']['enabled'] == "True" else False
        config_dict["modbus"]["port"] = int(config['modbus']['port'])

        config_dict["server"] = {}

        config_dict["pregong_enabled"] = True if config['pregong']['enabled'] == "True" else False

        config_dict["relay_enabled"] = True if config['relay']['enabled'] == "True" else False
    except Exception as e:
        logger.error(f"error {e}")
    return config_dict


def load_ip_config():
    """
        Creates a file inside the RAM of the device, containing its IP settings.
    Returns:

    """
    # Only used in this function, so no need to have it available for the whole file
    import re

    file_with_path = IF_CONFIG_PATH + IF_CONFIG_FILE
    ip_settings = dict()
    # declaring the regex pattern for IP addresses
    pattern = re.compile('''((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)''')

    # First create the ifconfig file in the IF_CONFIG_PATH
    settings_file = open(file_with_path, "w")
    command = ["ifconfig", "eth0"]
    process = Popen(command, stdout=settings_file, stderr=PIPE, )
    try:
        outs, errs = process.communicate(timeout=5)

        settings_file.close()  # Close the fd after the data has been added to it

        # logger.info(f"Sent command: {command}")
        # logger.info(f"Result from the command: {outs}")

        # Second open the file and extract the ip, netmask and broadcast
        with open(file_with_path, 'r') as f:
            file = f.readlines()
            valid = []
            # Find lines with IPs
            for line in file:
                line = line.rstrip()
                result = pattern.search(line)

                # valid IP addresses
                if result:
                    valid.append(line)
            valid = valid[0].strip().replace("  ", " ").split(" ")  # Clean up the data from unnecessary spaces

            # Store the values to key-value pairs
            for index in range(len(valid)):
                if index % 2 == 0:
                    ip_settings[valid[index]] = valid[index + 1]

            # logger.info(f"Valid IP addresses: {valid}")

    except TimeoutExpired:
        process.kill()
        outs, errs = process.communicate()
        logger.error(f"Timed out when loading IP config. Errs :{errs}")

    except Exception as e:
        logger.error(f"error {e}")

    # logger.info(f"Returning following settings: {ip_settings}")
    return ip_settings
