from secrets import token_urlsafe
import json

import os
import subprocess

import logging
from logging.handlers import RotatingFileHandler


rotHandler = RotatingFileHandler('/var/log/RestoreDefaults.log', maxBytes=2*1024*1024, backupCount=1)
logging.basicConfig(
    level=logging.DEBUG,
    handlers=[rotHandler], 
    format='%(asctime)s [%(levelname)-7s] %(filename)26s:%(lineno)3s %(threadName)18s | %(message)s'
)
logger = logging.getLogger(__name__)


FILE_PATH = '/mnt/data/prerecorded_messages/'
PROCESSED_FILES_PATH = FILE_PATH + 'processed/'

DEFAULT_FILES_PATH = FILE_PATH + 'default/'
DEFAULT_FILE_NAME = "default.mp3"  # This file name will be used to create the button



def reset_groups():
    with open("groups_backup.json", "w") as f:
        # groups = {str(i): {"name": f"Group {i}", "clients": [i]} for i in range(1, 16)}
        # json.dump(groups, f)
        json.dump({}, f)


def reset_users():
    with open("users_backup.json", "w") as f:
        users = {"0": {"name": "admin", "email": "", "secret": token_urlsafe(16)}}
        json.dump(users, f)


def delete_existing_files():
    """
        When restoring defaults, delete all existing files as they will not be accessible later
    Returns:

    """

    for file in os.listdir(FILE_PATH):
        if os.path.isfile(os.path.join(FILE_PATH, file)):
            logger.info(f"Deleting : {file}")
            try:
                base_name = file.split('.')
                wav_file = base_name[0] + '.wav'

                raw_file = os.path.join(FILE_PATH, file)
                processed_file = os.path.join(PROCESSED_FILES_PATH, wav_file)

                os.remove(raw_file)

                if os.path.exists(processed_file):
                    os.remove(processed_file)

            except Exception as e:
                logger.warning(f"Failed deletion: {e}")


def copy_default_file():
    """
        A default audio file is stored inside /mnt/data/prerecorded_messages/default to be used as an example.
        The filename in that directory must be default.mp3 in order for it to be processed.
    Returns:

    """

    copy_from = os.path.join(DEFAULT_FILES_PATH, DEFAULT_FILE_NAME)
    copy_to = os.path.join(FILE_PATH, DEFAULT_FILE_NAME)

    logger.info(f"Copy default file from: {copy_from}")
    logger.info(f"Copy default file to: {copy_to}")

    command = [
        "cp",
        copy_from,
        copy_to,
    ]

    process = subprocess.Popen(
        command,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
    )
    try:
        outs, errs = process.communicate(timeout=3)  # Longer timeout so larger files can be processed

        logger.debug(f"output: {outs}")
        logger.debug(f"error :{errs}")
        logger.info("Successfull!")
    except subprocess.TimeoutExpired:
        process.kill()
        outs, errs = process.communicate()
        logger.warning(f"Timed out, killing processing of {DEFAULT_FILE_NAME} : errs :{errs}")


def reset_audio_files():
    with open("audio_files_backup.json", "w") as f:
        audio_files = {
            "1": {"btn_name": "Default", "file_name": DEFAULT_FILE_NAME,
                  "timestamp": "1970-01-01 00:00:00", "sample_rate": "8000"}
        }
        json.dump(audio_files, f)


def main():
    reset_groups()
    reset_users()
    delete_existing_files()
    copy_default_file()
    reset_audio_files()


if __name__ == '__main__':
    main()
