import logging
import json
import base64
import os

from typing import Tuple,  Union
import subprocess

from barix.system.barix_enums import BarixHWType

from AudioController import AudioController

from lib.utils import get_pregong_filename
from lib.constants import Constants, AudioControllerState
from lib.InOut import Relay
from lib.utils import uci_to_config_dict

import logging
logger = logging.getLogger(__name__)

def get_filename_from_modbus_idx(idx, list_audio_files):
    logger.debug(f"try find file with modbus_idx {idx}")
    for i in list_audio_files:
        if "modbus_idx" in list_audio_files[i]:
            if list_audio_files[i]["modbus_idx"] == idx:
                logger.debug(f"file found {list_audio_files[i]['file_name']}")
                return os.path.join(Constants.FILE_PATH, list_audio_files[i]['file_name'])
    logger.debug("file not found")
    return ""

def getHwType():
    p = subprocess.Popen(["/usr/bin/qiba-spi-get-production-info.sh", "-w"], stdout=subprocess.PIPE, shell=False)
    return int(p.communicate()[0])

class MainController:
    def __init__(self):
        logger.info("MainController initialize...")

        self._config = uci_to_config_dict()

        with open("/barix/apps/icpgw/audio_files_backup.json", 'r') as f:
            self.audio_files = json.load(f)
        with open("/barix/apps/icpgw/groups_backup.json", 'r') as f:
            self.groups_data = json.load(f)
        with open("/barix/apps/icpgw/users_backup.json", 'r') as f:
            self.users_data = json.load(f)

        self.pregong_enabled = self._config['pregong_enabled']

        self.relay_enabled = False
        try:
            self.hw_type = getHwType()
            if self.hw_type == BarixHWType.Barix_MPI400.value:
                self.relay_enabled = self._config['relay_enabled']

                logger.info(f"relay: {self.relay_enabled}")

                self.relay = Relay(518)
        except Exception as e:
            logging.error(f"Error setting relay, {e}")


        self._audio_controller = AudioController(self._config, self.audio_controller_state_callback)
        # self._audio_controller = AudioController(self._config)

        self._file_being_played = None, None

        # flags to "lock" playing audio
        self._status = Constants.IDLE
        self._modbus_status = Constants.IDLE

        self.active_user = None
        self.active_user_uuid = None
        self._paging = False

        self.pregong_page_status = False


    @property
    def status(self):
        return self._status

    @status.setter
    def status(self, new_status):
        if self._status != new_status:
            self._status = new_status

###############################################################################
#
# 
#
###############################################################################

    def _set_active_user(self, data):
        self.active_user = self.get_user_from_token(data['token'])
        self.active_user_uuid = data['uuid']

    def _check_user(self, secret):
        if self.active_user is None:
            return True if self.get_user_from_token(secret) is not None else False
        elif self.users_data[self.active_user]["secret"] != secret:
            return False
        return True


    def stop_audio(self):
        self._paging = False
        self._audio_controller.stop_audio()

        self.active_user = None
        self.active_user_uuid = None

    def get_status(self):
        if self._audio_controller.status == AudioControllerState.IDLE:
            return "IDLE"
        if self._audio_controller.status == AudioControllerState.PAGING:
            return "PAGING"
        if self._audio_controller.status == AudioControllerState.PLAYING_FILE:
            return "PLAYING_FILE"
        if self._audio_controller.status == AudioControllerState.RECORDING:
            return "RECORDING"
        if self._audio_controller.status == AudioControllerState.STARTING:
            return "STARTING"
        return ""
    
    def get_clients(self, group_ids):
        client_ids = set([])
        
        if group_ids is None:
            return []
        for group_id in group_ids:
            try:
                client_ids.update( self.groups_data[str(group_id)]["clients"])
            except Exception as e:
                logger.warning(f"Group id {group_id} don't exist")
        return list(client_ids)

    def get_user_from_token(self, secret):
        for user_id in self.users_data.keys():
            if self.users_data[user_id]["secret"] == secret:
                return user_id

    def check_user_permissions(self, data) -> Tuple[bool, str]:
        try:
            uuid = data['uuid']
        except Exception as e:
            logger.error(f"Following occurred: {e}")
            return "uuid not defined", 409

        if self._modbus_status == Constants.BUSY:
            return "Modbus running", 409

        if not self._check_user(data["token"]):
            new_user = self.get_user_from_token(data["token"])
            logger.info(f"Received request from {new_user}")
            logger.warning(f"Another user is paging: {self.active_user}")
            return "Another user is paging!", 409

        if self.active_user_uuid is not None and uuid != self.active_user_uuid:
            logger.warning(f"Same user trying to page from a different UUID not allowed!"
                           f"\nRequest -> {data},\nCurrent UUID -> {self.active_user_uuid}")
            return "Same user trying to page from a different device not allowed!", 409

        return "", 200

    def check_play_permissions(self, data) -> Tuple[bool, str]:

        logger.debug(f"check_play_permissions {self.get_status()}")

        if self._audio_controller.status != AudioControllerState.IDLE or self._modbus_status == Constants.BUSY:
            msg, retval = self.check_user_permissions(data)
            if retval != 200:
                return msg, retval
            return "Resource busy!\nTry again in a few seconds.", 409
        return "", 200
    

    def audio_controller_state_callback(self, new_state: AudioControllerState):
        logger.debug(f"audio_controller_state_callback {new_state}")
        if new_state == AudioControllerState.IDLE:
            self._try_release_relay()
            

            # logger.debug("Clear active user")
            # self.active_user = ""
            # self.active_user_uuid = ""

###############################################################################
#
# Paging
#
###############################################################################
    
    def play_file(self, data: dict):
        clients = self.get_clients(data["groups"])
        
        filename = self.audio_files[data['file_ID']]['file_name']
        filepath = os.path.join(Constants.FILE_PATH, filename)

        self._set_active_user(data)
        self._relay_start()

        # "lock"
        self._audio_controller.status = AudioControllerState.STARTING
        self._audio_controller.add_task_start({"clients": clients})

        if self.pregong_enabled:
            self._audio_controller.add_task_play_file({"filepath": get_pregong_filename()})

        self._audio_controller.add_task_play_file({"filepath": filepath})
        
        self._audio_controller.add_task_stop()

        self._file_being_played = data['file_ID'], filename




    def play_recorded(self, data: dict):
        clients = self.get_clients(data["groups"])

        self._set_active_user(data)
        self._relay_start()

        self._audio_controller.status = AudioControllerState.STARTING
        self._audio_controller.add_task_start({"clients": clients})

        if self.pregong_enabled:
            self._audio_controller.add_task_play_file({"filepath": get_pregong_filename()})

        self._audio_controller.add_task_play_recorded({"audio": base64.b64decode(data["audio_encoded"])})
        self._audio_controller.add_task_stop()
        self._file_being_played = None, None


    def start_paging(self, data: dict):
        clients = self.get_clients(data["groups"])

        self._set_active_user(data)
        self._relay_start()
        self._paging = True

        self._audio_controller.status = AudioControllerState.STARTING
        self._audio_controller.add_task_start({"clients": clients})

        if self.pregong_enabled:
            self.pregong_page_status = True
            self._audio_controller.add_task_play_file({"filepath": get_pregong_filename()})
            self._audio_controller.add_task_stop()
        
        self._file_being_played = None, None


    def page_audio(self, data: dict) -> Tuple[str, int]:
        # Discard page audio while pregong is being played
        if self.pregong_page_status:
            if self._audio_controller.status == AudioControllerState.IDLE and self.active_user is not None:
                self._audio_controller.status = AudioControllerState.STARTING
                clients = self.get_clients(data["groups"])
                self._audio_controller.add_task_start({"clients": clients})
                self.pregong_page_status = False
            else:
                return

        self._audio_controller.add_task_page_audio({"audio": base64.b64decode(data["audio_encoded"])})



###############################################################################
#
# Modbus
#
###############################################################################

    def modbus_lock(self):
        self._modbus_status = Constants.BUSY

    def modbus_unlock(self):
        self._modbus_status = Constants.IDLE

    def modbus_play_file(self, json_data):
        logger.debug(f"modbus_play_file: {json_data}")
               
        if json_data['pregong']:
            filepath = get_pregong_filename()
        else:
            filepath = get_filename_from_modbus_idx(json_data['file_idx'], self.audio_files)

        if filepath is None:
            logger.error("file doesn't exist")
            return "file doesn't exist", 400

        clients = self.get_clients([json_data['client_idx']])

        # data = {
        #     "filepath" : filepath,
        #     "clients" : clients
        # }

        self._relay_start()

        self._audio_controller.status = AudioControllerState.STARTING
        self._audio_controller.add_task_start({"clients": clients})
        self._audio_controller.add_task_play_file({"filepath": filepath})
        self._audio_controller.add_task_stop()
        
        return "", 200
        # self._audio_controller.start_playing_file(data)

###############################################################################
#
#
#
###############################################################################

    def get_playfile_state(self, data) -> Tuple[str, int]:
        """
            While a file is being played, the front end can get the status by sending a /playfile PUT request
        """

        try:
            uuid = data['uuid']
        except Exception as e:
            logger.error(f"Following occurred: {e}")
            return "uuid not defined", 401


        if self._modbus_status == Constants.BUSY:
            return "Modbus running", 409

        if not self._check_user(data["token"]):
            new_user = self.get_user_from_token(data["token"])
            logger.info(f"Received request from {new_user}")
            logger.warning(f"Another user is paging: {self.active_user}")
            return "Another user is paging!", 409

        if self.active_user_uuid is not None and uuid != self.active_user_uuid:
            logger.warning(f"Same user trying to page from a different UUID not allowed!"
                           f"\nRequest -> {data},\nCurrent UUID -> {self.active_user_uuid}")
            return "Same user trying to page from a different device not allowed!", 409

        if self._audio_controller.status == AudioControllerState.PLAYING_FILE:
            return str(self._file_being_played), 206

        if self._audio_controller.status == AudioControllerState.RECORDING:
            return "playing recorded audio", 206

        # if self.active_user != None:
        #     return "", 206

        if self._audio_controller.status == AudioControllerState.STARTING:
            return "", 206
        
        if self._audio_controller.status == AudioControllerState.IDLE:
            return "", 200


        msg = f"playfile state {self._audio_controller.status}"
        logger.warning(msg)
        return msg, 500


    def set_modbus_status(self, caller, status):
        if caller == "modbus":
            self._modbus_status = status
        else:
            logger.warning(f"Only modbus can set this status")
        if self._modbus_status == Constants.IDLE:
            self._relay_stop()

    def _relay_start(self):
        if self.relay_enabled:
            self.relay.write(True)
            logger.debug("# Relay active")

    def _try_release_relay(self):
        if self._modbus_status == Constants.IDLE:
            self._relay_stop()

    def _relay_stop(self):
        if self.relay_enabled and not self._paging:
            self.relay.write(False)
            logger.debug("# Relay inactive")


    def set_active_audio_file(self, filename: Union[str, None]):
        try:
            if filename is not None:
                filename_base = filename.split(".")[0]
                # logger.info(f"filename_base: {filename_base}")
                for file_idx in self.audio_files:
                    # logger.info(f"Current file being checked: {self.audio_files[file_idx]}")
                    if self.audio_files[file_idx]["file_name"].split(".")[0] == filename_base:
                        # logger.info(f"Setting new file being played: {file_idx}")
                        self._file_being_played = file_idx, self.audio_files[file_idx]
                        break
            else:
                self._file_being_played = None, None
        except Exception as e:
            logger.warning(f"Error while trying to set active_audio_file: {e}")
            self._file_being_played = None, None


