import json
import logging
import os
import time
import threading
import urllib.error

from pathlib import Path

from barix import HTTPRequestsAPI

from .uci import getValueOfUci
from .AES67_rest import AES67DaemonREST, BacoAudioLoopREST
from .utils import readFromJSONFile

class SourceNotFoundException(Exception):
    pass

class SourceNotAliveException(Exception):
    pass

class SourceNotReadyException(Exception):
    pass

AES67_DIR = "/mnt/data/aes67"
SINK_CONFIG_FILE_PATH = os.path.join(AES67_DIR, "sink.json")
SOURCES_LIST_FILE_PATH = os.path.join(AES67_DIR, "sourcesList.json")

log = logging.getLogger('flask-backend')

listOfAES67Sources = None
lock = threading.Lock()

def getRemoteSources():
    aes67 = AES67DaemonREST()
    try:
        result = aes67.getRemoteSources()
        sourcesList = result['remote_sources']
    except Exception as e:
        log.error(e)
        raise e
    else:
        return sourcesList

def getSourcesFromFile():
    sourcesList = []
    try:
        if os.path.isfile(SOURCES_LIST_FILE_PATH):
            sourcesList = readFromJSONFile(SOURCES_LIST_FILE_PATH)
    except Exception as e:
        log.error(e)
        raise e
    else:
        return sourcesList

def storeSourcesInFile(sourcesList):
    try:
        log.debug("saveAES67Configs is trying to acquire lock...")
        lock.acquire()
        log.debug("saveAES67Configs acquired lock...")
        writeSourcesInJSONFile(sourcesList)
        lock.release()
        log.debug("saveAES67Configs released lock...")
    except Exception as e:
        log.error(e)
        raise e

def getSourceID(sourceSDP):
    try:
        originator = None
        RTPAddr = None
        RTPPort = None
        lines = sourceSDP.split('\r\n')
        for line in lines:
            if line.startswith('o='):
                originator = line.split(' ')[1]
            if line.startswith('c='):
                RTPAddr = line.split(' ')[2].split('/')[0]
            if line.startswith('m='):
                RTPPort = line.split(' ')[1]
    except Exception as e:
        log.error(e)
        raise e
    else:
        return originator, RTPAddr, RTPPort

def getSourceSessionID(sourceSDP):
    try:
        sessionID = None
        lines = sourceSDP.split('\r\n')
        for line in lines:
            if line.startswith('o='):
                sessionID = line.split(' ')[2]
    except Exception as e:
        log.error(e)
        raise e
    else:
        return sessionID

def sourcesMatch(source1, source2):
    try:
        source1Originator, source1RTPAddr, source1RTPPort = getSourceID(source1['sdp'])
        source2Originator, source2RTPAddr, source2RTPPort = getSourceID(source2['sdp'])
        if 'protocol' in source1.keys():
            source1Protocol = source1['protocol']
        else:
            source1Protocol = source1['source']
        if 'protocol' in source2.keys():
            source2Protocol = source2['protocol']
        else:
            source2Protocol = source2['source']
    except Exception as e:
        log.error(e)
        raise e
    else:
        return source1Originator == source2Originator and source1RTPAddr == source2RTPAddr and source1RTPPort == source2RTPPort and source1Protocol == source2Protocol

def removeDuplicateSources(sources):
    try:
        cleanedSourceList = []
        for source in sources:  # remote sources are always ordered from the oldest to the newest
            sourceFound = False
            for i in range(len(cleanedSourceList)):
                cleanedSource = cleanedSourceList[i]
                if sourcesMatch(cleanedSource, source):
                    sourceFound = True
                    if source["last_seen"] > cleanedSource["last_seen"]:  # the most recent sources have a bigger last_seen
                        cleanedSourceList[i] = source
            if not sourceFound:
                cleanedSourceList.append(source)
    except Exception as e:
        log.error(e)
        raise e
    return cleanedSourceList

def checkSourcesActivity():
    try:
        while True:
            lock.acquire()
            fileSources = getSourcesFromFile()
            try:
                remoteSources = getRemoteSources()
            except urllib.error.URLError as e:
                log.error("AES67 daemon rest returned {}".format(e))
                lock.release()
                time.sleep(10)
            else:
                cleanedRemoteSources = removeDuplicateSources(remoteSources)

                newSources = []
                for remoteSource in cleanedRemoteSources:
                    sourceFound = False
                    for i in range(len(fileSources)):
                        fileSource = fileSources[i]
                        if sourcesMatch(remoteSource, fileSource):
                            sourceFound = True
                            if remoteSource["last_seen"] != fileSource["last_seen"]:
                                timestamp = time.time()
                                # update source
                                newSourceDict = updateSourceConfig(fileSource, remoteSource, timestamp)
                                fileSources[i] = newSourceDict
                            break
                    if not sourceFound:
                        newSourceDict = createNewSourceConfig(remoteSource)
                        newSources.append(newSourceDict)

                updatedFileSourcesList = []
                # if it is a remote source (SAP), and it is deleted and not found in daemon sources list announced, remove it from file
                # if it is a mDNS source and mDNS is disabled, remove it from file
                mdnsDiscovery = getValueOfUci('aes67', 'network', 'mdns_enabled')
                for fileSource in fileSources:
                    if fileSource["protocol"] == "SAP":
                        if fileSource["deleted"]:
                            for remoteSource in cleanedRemoteSources:
                                if sourcesMatch(fileSource, remoteSource):
                                    updatedFileSourcesList.append(fileSource)
                                    break
                        else:
                            updatedFileSourcesList.append(fileSource)
                    elif fileSource['protocol'] == "mDNS":
                        if mdnsDiscovery == "true":
                            updatedFileSourcesList.append(fileSource)
                        # if mdnsDiscovery if disabled, don't append
                    elif fileSource['protocol'] == "Manual":
                        if not fileSource['deleted']:
                            updatedFileSourcesList.append(fileSource)
                    else: # default
                        updatedFileSourcesList.append(fileSource)
                updatedFileSourcesList.extend(newSources)
                writeSourcesInJSONFile(updatedFileSourcesList)
                lock.release()
                time.sleep(10)
    except Exception as e:
        log.error(e)
        lock.release()
        raise e

def createNewSourceConfig(config):
    try:
        source = {}
        if {"address", "announce_period", "domain", "id", "last_seen", "name", "sdp", "source"} <= config.keys():
            source['address'] = config['address']
            source['period'] = config['announce_period']
            source['domain'] = config['domain']
            source['id'] = config['id']
            source['name'] = config['name']
            source['protocol'] = config['source']
            sdp = config['sdp']
            lines = sdp.split('\r\n')
            # source['sdp'] = lines[0:-1]
            source['sdp'] = config['sdp']
            for line in lines:
                if line.startswith('c='):
                    source['rtp_addr'] = line.split(' ')[2].split('/')[0]
                if line.startswith('m='):
                    source['rtp_port'] = line.split(' ')[1]
            source['last_seen'] = config['last_seen']
            source["last_time_checked"] = time.time()
            source["alias"] = ""
            source["deleted"] = False
    except Exception as e:
        log.error(e)
        raise e
    else:
        return source

def updateSourceConfig(oldConfig, newConfig, timestamp):
    try:
        updatedSource = {}
        if {"address", "announce_period", "domain", "id", "last_seen", "name", "sdp", "source"} <= newConfig.keys():
            updatedSource['address'] = newConfig['address']
            updatedSource['period'] = newConfig['announce_period']
            updatedSource['domain'] = newConfig['domain']
            updatedSource['id'] = newConfig['id']
            updatedSource['name'] = newConfig['name']
            updatedSource['protocol'] = newConfig['source']
            sdp = newConfig['sdp']
            lines = sdp.split('\r\n')
            # updatedSource['sdp'] = lines[0:-1]
            updatedSource['sdp'] = newConfig['sdp']
            for line in lines:
                if line.startswith('c='):
                    updatedSource['rtp_addr'] = line.split(' ')[2].split('/')[0]
                if line.startswith('m='):
                    updatedSource['rtp_port'] = line.split(' ')[1]
            updatedSource["last_seen"] = newConfig["last_seen"]
            updatedSource["last_time_checked"] = timestamp
            # last activity can be replaced by last_time_checked
            updatedSource["alias"] = oldConfig["alias"]
            updatedSource["deleted"] = False
    except Exception as e:
        log.error(e)
        raise e
    else:
        return updatedSource

def writeSourcesInJSONFile(json_data):
    try:
        with open(SOURCES_LIST_FILE_PATH, 'w') as json_file:
            log.debug("Storing AES67 sources {} configuration in file {}".format(json_data, SOURCES_LIST_FILE_PATH))
            json.dump(json_data, json_file)
    except Exception as e:
        log.error(e)
        raise e

def checkShouldPlayOnInit():
    try:
        if os.path.isfile(SINK_CONFIG_FILE_PATH):
            sinkConfig = readFromJSONFile(SINK_CONFIG_FILE_PATH)
            log.debug("Sink configuration: {}".format(sinkConfig))
            time.sleep(5)
            currSourceProtocol = sinkConfig['source_protocol']
            if currSourceProtocol == "SAP" or currSourceProtocol == "mDNS":
                currSourceOrigin, currSourceRTPAddr, currSourceRTPPort = getSourceID(sinkConfig['sdp'])
                remoteSources = getRemoteSources()
                cleanedRemoteSources = removeDuplicateSources(remoteSources)
                sourceFound = False
                for remoteSource in cleanedRemoteSources:
                    remoteSourceProtocol = remoteSource['source']
                    remoteSourceOrigin, remoteSourceRTPAddr, remoteSourceRTPPort = getSourceID(remoteSource['sdp'])
                    if remoteSourceProtocol == currSourceProtocol and remoteSourceOrigin == currSourceOrigin and remoteSourceRTPAddr == currSourceRTPAddr and remoteSourceRTPPort == currSourceRTPPort:
                        sourceFound = True
                        sourceToPlay = {}
                        sourceToPlay['sdp'] = remoteSource['sdp']
                        sourceToPlay['protocol'] = remoteSource['source']
                        sourceToPlay['name'] = remoteSource['name']
                        sourceToPlay['id'] = remoteSource['id']
                        playSource(sourceToPlay)
                        break
                if not sourceFound:
                    sourceToPlay = {}
                    sourceToPlay['sdp'] = sinkConfig['sdp']
                    sourceToPlay['protocol'] = sinkConfig['source_protocol']
                    sourceToPlay['name'] = sinkConfig['source_name']
                    sourceToPlay['id'] = sinkConfig['name'].replace('Sink ', '')
                    playSource(sourceToPlay)
            else: # manual sources
                fileSources = getSourcesFromFile()
                for fileSource in fileSources:
                    if fileSource['protocol'] == "Manual":
                        #if ids match
                        currSourceID = sinkConfig['name'].replace('Sink ', '')
                        if currSourceID == fileSource['id']:
                            playSource(fileSource)
                            break
        else:
            log.info("No sink is configured, nothing to play.")
    except Exception as e:
        log.error(e)

def checkIfCanPlaySource(sourceJson):
    sourceToPlay = None
    try:
        fileSources = getSourcesFromFile()
        for fileSource in fileSources:
            if sourcesMatch(sourceJson, fileSource):
                sourceToPlay = fileSource
        if sourceToPlay is None:
            raise SourceNotFoundException
        if (time.time() - sourceToPlay['last_time_checked']) > (sourceToPlay['period'] + 10):
            raise SourceNotAliveException
        audio_loop = BacoAudioLoopREST(8088)
        statsJson = audio_loop.getStats()
        if statsJson["mode"] == "LOOP" and statsJson["loopMode"] != 0:
            return sourceToPlay
        else:
            raise SourceNotReadyException
    except SourceNotFoundException:
        raise SourceNotFoundException
    except SourceNotAliveException:
        raise SourceNotAliveException
    except SourceNotReadyException:
        raise SourceNotReadyException
    except Exception as e:
        log.error(e)
        raise

def getSdpNumChannels(sdp):
    """
    Get number of channels presented in SDP
    :param sdp: sdp info string
    """
    try:
        sdpFieldList = sdp.split("\r\n")
        sdpRtpAudioFormatField = [s for s in sdpFieldList if "a=rtpmap" in s]
        splitRtpField = sdpRtpAudioFormatField[0].split("/")
        return splitRtpField.pop()
    except Exception as e:
        log.error(e)
        raise

def playSource(sourceToPlay):
    try:
        sinkConfig = {}
        sinkConfig['sdp'] = sourceToPlay['sdp']
        # default values
        sinkConfig['name'] = "Sink " + sourceToPlay['id']
        sinkConfig['io'] = 'Audio Device'
        sinkConfig['use_sdp'] = True
        sinkConfig['source'] = "http://127.0.0.1:8080/api/source/sdp/0"
        nChannels = int(getSdpNumChannels(sinkConfig['sdp']))
        sinkConfig['map'] = [0, 1]
        if nChannels == 2:
            log.info("Add Sink --> Stereo")
        elif nChannels == 1:
            sinkConfig['map'] = [0]
            log.info("Add Sink --> Mono")
        else:
            log.error(f"Add Sink -- invalid number of channels ({nChannels}) in SDP")
        sinkConfig['delay'] = 192  # sourceToPlay['delay']
        sinkConfig['ignore_refclk_gmid'] = True  # sourceToPlay['ignore_refclk_gmid']
        sinkID = 0  # so far we only play one source, sink pos 0
        log.debug("Sink to play: {}".format(sinkConfig))
        aes67 = AES67DaemonREST()
        result = aes67.addRTPSink(sinkID, sinkConfig)
        response = None
        if result == 200:
            # store sink config to play automatically after reset
            with open(SINK_CONFIG_FILE_PATH, 'w') as json_file:
                sinkConfig['source_name'] = sourceToPlay['name']
                sinkConfig['source_protocol'] = sourceToPlay['protocol']
                log.debug("Sink configuration: {}".format(sinkConfig))
                json.dump(sinkConfig, json_file)
            response = sourceToPlay['id']
            # start monitoring stream connection
            t = threading.Thread(target=monitorStreamConnection, args=())
            t.setDaemon(True)
            t.start()
    except Exception as e:
        log.error(e)
        raise e
    else:
        return response, result

def monitorStreamConnection():
    aes67 = AES67DaemonREST()
    streamLost = None
    keepMonitoring = True
    if os.path.isfile(SINK_CONFIG_FILE_PATH):
        try:
            result = aes67.getRTPSinkStatus(0)
            #log.debug(result)
            if type(result) is dict:
                log.debug("Receiving RTP packets: {}".format(result['sink_flags']['receiving_rtp_packet']))
                streamLost = not result['sink_flags']['receiving_rtp_packet']
            else:  # no sink? o.O
                log.error("Unexpected sink status response: {}".format(result))
        except Exception as e:
            log.error(e)
            raise e
        while os.path.isfile(SINK_CONFIG_FILE_PATH) and keepMonitoring:
            log.debug("Stream Lost? {}".format(streamLost))
            # get Sink Status
            try:
                sinkStatus = aes67.getRTPSinkStatus(0)
                #log.debug(sinkStatus)
                if type(sinkStatus) is dict:
                    log.debug("Receiving RTP packets: {}".format(sinkStatus['sink_flags']['receiving_rtp_packet']))
                    if sinkStatus['sink_flags']['receiving_rtp_packet']:
                        if streamLost:
                            log.debug("Stream is alive again! Play stream!")
                            try:
                                content = readFromJSONFile(SINK_CONFIG_FILE_PATH)
                                currOriginator, currRTPAddr, currRTPPort = getSourceID(content['sdp'])
                                #time.sleep(10)
                                # sink new source
                                remoteSources = getRemoteSources()
                                #log.debug("Remote Sources")
                                #log.debug(remoteSources)
                                cleanedRemoteSources = removeDuplicateSources(remoteSources)
                                #log.debug("Cleaned Remote Sources")
                                #log.debug(cleanedRemoteSources)
                                for remoteSource in cleanedRemoteSources:
                                    remoteOriginator, remoteRTPAddr, remoteRTPPort = getSourceID(remoteSource['sdp'])
                                    if remoteOriginator == currOriginator and remoteRTPAddr == currRTPAddr and remoteRTPPort == currRTPPort:
                                        sourceToPlay = {}
                                        sourceToPlay['sdp'] = remoteSource['sdp']
                                        sourceToPlay['protocol'] = remoteSource['source']
                                        sourceToPlay['name'] = remoteSource['name']
                                        sourceToPlay['id'] = remoteSource['id']
                                        playSource(sourceToPlay)
                                        keepMonitoring = False
                                        break
                            except Exception as e:
                                log.error(e)
                                raise e
                        else:
                            # check if a new version of this source showed up
                            try:
                                content = readFromJSONFile(SINK_CONFIG_FILE_PATH)
                                currOriginator, currRTPAddr, currRTPPort = getSourceID(content['sdp'])
                                # sink new source
                                remoteSources = getRemoteSources()
                                #log.debug("Remote Sources")
                                #log.debug(remoteSources)
                                cleanedRemoteSources = removeDuplicateSources(remoteSources)
                                #log.debug("Cleaned Remote Sources")
                                #log.debug(cleanedRemoteSources)
                                for remoteSource in cleanedRemoteSources:
                                    remoteOriginator, remoteRTPAddr, remoteRTPPort = getSourceID(remoteSource['sdp'])
                                    if remoteOriginator == currOriginator and remoteRTPAddr == currRTPAddr and remoteRTPPort == currRTPPort:
                                        if getSourceSessionID(content['sdp']) != getSourceSessionID(remoteSource['sdp']):
                                            log.debug("Stream was updated! Play new stream!")
                                            sourceToPlay = {}
                                            sourceToPlay['sdp'] = remoteSource['sdp']
                                            sourceToPlay['protocol'] = remoteSource['source']
                                            sourceToPlay['name'] = remoteSource['name']
                                            sourceToPlay['id'] = remoteSource['id']
                                            playSource(sourceToPlay)
                                            keepMonitoring = False
                                            break
                            except Exception as e:
                                log.error(e)
                                raise e
                    streamLost = not sinkStatus['sink_flags']['receiving_rtp_packet']
                elif sinkStatus == 400: # no sink? o.O For example when update fails
                    log.debug("No sink configured, stop monitoring")
                    keepMonitoring = False
                else:
                    log.error("Unexpected sink status response: {}".format(sinkStatus))
            except Exception as e:
                log.error(e)
                raise e
            time.sleep(1)

def stopBacoAudioLoop():
    # stop Baco audio loop to set new configs
    r = HTTPRequestsAPI.post('http://{}:{}{}'.format('127.0.0.1', 50500, '/api/baco/stop'), {})
    return r.status

def startBacoAudioLoop():
    r = HTTPRequestsAPI.post('http://{}:{}{}'.format('127.0.0.1', 50500, '/api/baco/start'), {})
    return r.status

def stopPlayingAES67():
    try:
        aes67 = AES67DaemonREST()
        result = aes67.remRTPSink(0) #so far we only play one source, sink pos 0
        if result == 200:
            #remove file with sink config
            p = Path(SINK_CONFIG_FILE_PATH)
            p.unlink()
    except Exception as e:
        log.error(e)
        raise e

def getSinkStatusAndErrors(sinkID):
    try:
        aes67 = AES67DaemonREST()
        result = aes67.getRTPSinkStatus(sinkID)
        if type(result) is dict:
            statusDict = result
            # get error counters as well
            result = aes67.getRTPSinkErrorCounters(sinkID)
            if type(result) is dict:
                statusDict.update(result)
                return statusDict
            elif result == 400:  # no sink for now
                return {}
            # else raises exception
        elif result == 400: # no sink for now
            return {}
        # else raises exception
    except Exception as e:
        log.error(e) # <urlopen error [Errno 111] Connection refused>
        raise e