#  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#  [21/7/2023] Created by Anton Angov for Barix AG.
#  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

import logging
import logging.config
from typing import Literal

from concurrent_log_handler import ConcurrentRotatingFileHandler
import os
from os import environ
import sys, traceback
import pathlib


# Random numbers for rotation temp file names, using secrets module if available (Python 3.6).
# Otherwise use `random.SystemRandom` if available, then fall back on `random.Random`.
try:
    from secrets import randbits
except ImportError:
    import random

    if hasattr(random, "SystemRandom"):  # May not be present in all Python editions
        # Should be safe to reuse `SystemRandom` - not software state dependant
        randbits = random.SystemRandom().getrandbits
    else:
        def randbits(nb):
            return random.Random().getrandbits(nb)


# ANSI color codes
COLOR_CODES = {
    "red": "\033[31m",
    "green": "\033[32m",
    "yellow": "\033[33m",
    "blue": "\033[34m",
    "magenta": "\033[35m",
    "cyan": "\033[36m",
    "white": "\033[37m",
    "reset": "\033[0m",
}


# @deprecated ("Use get_logger() which will ColorLogger class instead")
def color_print(message, logger, level, color="white"):
    """
        Log a message with the specified log level and color.

    @param message: The message to log.
    @param level: The log level (e.g., "debug", "info", "warning", "error", "critical").
    @param color: The text color (e.g., "red", "green", "yellow").
    @param logger: Logger instance
    """
    color_code = COLOR_CODES.get(color.lower(), COLOR_CODES["white"])  # Default to white if color is invalid
    reset_code = COLOR_CODES["reset"]

    # Format the message with the color
    colored_message = f"{color_code}{message}{reset_code}"

    # Use the logging module to log the message
    if level.lower() == "debug":
        logger.debug(colored_message)
    elif level.lower() == "info":
        logger.info(colored_message)
    elif level.lower() == "warning":
        logger.warning(colored_message)
    elif level.lower() == "error":
        logger.error(colored_message)
    elif level.lower() == "critical":
        logger.critical(colored_message)
    else:
        raise ValueError("Invalid log level. Use one of: debug, info, warning, error, critical.")


ColorsLiteral = Literal["red", "green", "yellow", "blue", "magenta", "cyan", "white", "reset"]


class ColorLogger(logging.Logger):
    """
    Custom logger that supports colored output via the 'color' parameter.
    """

    def _log(self, level, msg, args, exc_info=None, extra=None, stack_info=False, color: ColorsLiteral = None):
        """
        Overrides the _log method to add color functionality.
        """
        if color:
            color_code = COLOR_CODES.get(color.lower(), COLOR_CODES["white"])
            reset_code = COLOR_CODES["reset"]
            msg = f"{color_code}{msg}{reset_code}"
        # stacklevel=2 allows to use the caller lineno
        super()._log(level, msg, args, exc_info, extra, stack_info, stacklevel=2)


class BarixRotatingFileHandler(ConcurrentRotatingFileHandler):

    """
        This class tries to overcome a bug in the file rotation, where some files are reported missing
        even though they are not, and this gives a FileNotFoundError
    """

    def __init__(self, filename, mode='a', maxBytes=0, backupCount=0, encoding=None, delay=False):
        super().__init__(filename=filename, mode=mode, maxBytes=maxBytes,
                         backupCount=backupCount, encoding=encoding, delay=delay)
        self._count_do_rollover_calls = 0
        self._count_do_rollover_success = 0

    def doRollover(self):  # noqa: C901
        """
        Do a rollover, as described in __init__().
        """
        self._close()
        if self.backupCount <= 0:
            # Don't keep any backups, just overwrite the existing backup file
            # Locking doesn't much matter here; since we are overwriting it anyway
            self.stream = self.do_open("w")
            self._close()
            return

        # Determine if we can rename the log file or not. Windows refuses to
        # rename an open file, Unix is inode based, so it doesn't care.

        # Attempt to rename logfile to tempname:
        # There is a slight race-condition here, but it seems unavoidable
        tmpname = None
        while not tmpname or os.path.exists(tmpname):
            tmpname = f"{self.baseFilename}.rotate.{randbits(64):08}"
        try:
            # Do a rename test to determine if we can successfully rename the log file
            os.rename(self.baseFilename, tmpname)

            if self.use_gzip:
                self.do_gzip(tmpname)
        except OSError as e:
            self._console_log(f"rename failed.  File in use? e={e}", stack = True)
            stack_str = ":\n" + "".join(traceback.format_stack())
            print(f"rename failed.  File in use? e={e}. Stack: {stack_str}")
            return

        gzip_ext = ".gz" if self.use_gzip else ""

        def do_rename(source_fn, dest_fn):
            self._console_log(f"Rename {source_fn} -> {dest_fn + gzip_ext}")
            print(f"Rename {source_fn} -> {dest_fn + gzip_ext}")
            if os.path.exists(dest_fn):
                os.remove(dest_fn)
            if os.path.exists(dest_fn + gzip_ext):
                os.remove(dest_fn + gzip_ext)
            source_gzip = source_fn + gzip_ext
            if os.path.exists(source_gzip):
                os.rename(source_gzip, dest_fn + gzip_ext)
            elif os.path.exists(source_fn):
                os.rename(source_fn, dest_fn)

        # Q: Is there some way to protect this code from a KeyboardInterrupt?
        # This isn't necessarily a data loss issue, but it certainly does
        # break the rotation process during stress testing.

        # There is currently no mechanism in place to handle the situation
        # where one of these log files cannot be renamed. (Example, user
        # opens "logfile.3" in notepad); we could test rename each file, but
        # nobody's complained about this being an issue; so the additional
        # code complexity isn't warranted.

        do_renames = []
        for i in range(1, self.backupCount):
            sfn = self.rotation_filename(f"{self.baseFilename}.{i}")
            dfn = self.rotation_filename(f"{self.baseFilename}.{i + 1}")
            if os.path.exists(sfn + gzip_ext):
                do_renames.append((sfn, dfn))
            else:
                # Break looking for more rollover files as soon as we can't find one
                # at the expected name.
                break

        for sfn, dfn in reversed(do_renames):
            do_rename(sfn, dfn)

        dfn = self.rotation_filename(self.baseFilename + ".1")
        do_rename(tmpname, dfn)

        if self.use_gzip:
            logFilename = self.baseFilename + ".1.gz"
            self._do_chown_and_chmod(logFilename)

        self.num_rollovers += 1
        self._console_log("Rotation completed (on size)")

    def doRollover_orig(self):
        """
            This function overwrites the original one from RotatingFileHandler.
            Currently, the only changes are added print messages for debug
            Do a rollover, as described in __init__().
        """
        self._count_do_rollover_calls += 1
        print(f"Using BarixRotatingFileHandler for doRollover(). Called [{self._count_do_rollover_calls}] times")
        if self.stream:
            self.stream.close()
            self.stream = None

        sfn = ''
        dfn = ''
        try:
            if self.backupCount > 0:
                print(os.listdir("/var/log/"))
                for i in range(self.backupCount - 1, 0, -1):
                    sfn = self.rotation_filename("%s.%d" % (self.baseFilename, i))
                    dfn = self.rotation_filename("%s.%d" % (self.baseFilename, i + 1))
                    if os.path.exists(sfn):
                        if os.path.exists(dfn):
                            print(f"Removing dfn: {dfn}")
                            os.remove(dfn)
                        print(f"Renaming sfn to dfn: {sfn} --> {dfn}")
                        os.rename(sfn, dfn)
                dfn = self.rotation_filename(self.baseFilename + ".1")
                if os.path.exists(dfn):
                    os.remove(dfn)
                self.rotate(self.baseFilename, dfn)
            # Here we are already successful
            self._count_do_rollover_success += 1
            print(f"Success doRollover [{self._count_do_rollover_success}] "
                  f"out of [{self._count_do_rollover_calls}] calls")

        except FileNotFoundError as e:
            print(f"sfn exists: {os.path.exists(sfn)}")
            print(f"dfn exists: {os.path.exists(dfn)}")
            print(f"While trying to rotate logs with BarixRotatingFileHandler, following occurred: {e}")
            with open("../traceback.log", 'a') as trace_back:
                print("Writing to package/traceback.log")

                trace_back.write("\nError:")
                trace_back.write(str(e))
                stack = traceback.extract_stack()

                trace_back.write("\nTraceback stack:")
                trace_back.write(str(stack))

                trace_back.write("\nTraceback stack with traceback.print_exc:")
                traceback.print_exc(file = trace_back)

                print("Finished writing to package/traceback.log")

            traceback.print_exc(file = sys.stdout)

        if not self.delay:
            self.stream = self._open()


def get_logger(name="", log_file="/var/log/babroad_client.log", max_file_size=2*1024*1024, config=""):
    """
        Create a logger for the caller. Uses BarixRotatingFileHandler (fixes file rotation bugs)
        and ColorLogger (color option added to logger)
    @param name: How the log messages would be named in the log
    @param log_file: Where should the log be stored
    @param max_file_size: Max size of the log file
    @param config: Config for the logger, if available
    @return: New logger handle
    """
    # Set the color logger as the default logger class
    logging.setLoggerClass(ColorLogger)

    if config == "":
        print(f"Creating default logger for {name}")

        if environ.get('BABROAD_LOGLEVEL') is None:
            environ['BABROAD_LOGLEVEL'] = 'development'
        LOGLEVEL = environ.get('BABROAD_LOGLEVEL', 'INFO').upper()
        # logging.setLoggerClass(ColorLogger)  # This way we can have a parameter color
        new_logger = logging.getLogger(name)
        if LOGLEVEL.upper() != 'development'.upper():
            # new_logger.setLevel(level = logging.INFO)
            new_logger.setLevel(level = logging.WARNING)
        else:
            # new_logger.setLevel(level = logging.INFO)
            new_logger.setLevel(level = logging.INFO)

        # Initialize directory structure
        log_file = pathlib.Path(log_file)
        log_file.touch(exist_ok=True)
        file_handler = BarixRotatingFileHandler(log_file, maxBytes=max_file_size, backupCount=10)
        file_handler.setLevel(logging.INFO)

        # create formatter, to get path: - [%(pathname)s]
        formatter = logging.Formatter(
            "%(asctime)s " + name + "[%(lineno)d] %(levelname)s | %(message)s [%(threadName)s]",
            datefmt="%b %d %H:%M:%S",
        )

        # add formatter to file_handler
        file_handler.setFormatter(formatter)

        # add file_handler to logger
        new_logger.addHandler(file_handler)
    else:
        print("About to load .conf file: %s", config)
        logging.config.fileConfig(config)
        new_logger = logging.getLogger(name)
        print("Loaded the .conf file")
    return new_logger
