import json
import logging

from flask import Blueprint, Response, request

from barix.web.auth import login_required

from users import new_secret, get_users, new_user, update_user, remove_users, save_users_and_restart

bp = Blueprint('users_API', __name__)

logger = logging.getLogger('flask-backend')


@bp.route('/api/rest/v1/users', methods=['GET'])
@login_required
def get_all_users():
    try:
        data = get_users()

        # add audio_files key to user admin dict
        data["0"]["audio_files"] = []

        # transform dict into array
        usersList = [
            {"id": key, **value} for key, value in data.items()
        ]

        # send list of users in response
        response_json = {"users": usersList}

        return Response(json.dumps(response_json), status=200, headers={"Content-Type": "application/json"})

    except Exception as e:
        logger.error(e, exc_info=True)
        return Response(status=500)


@bp.route('/api/rest/v1/users/<user_id>/token', methods=['POST'])
@login_required
def renew_user_token(user_id: str):
    try:
        data = get_users()

        if user_id not in data:
            logger.error(f"User id {user_id} doesn't exist!")
            return Response(status=404)

        data[user_id]["secret"] = new_secret()

        save_users_and_restart(data)

        return Response(status=200)

    except Exception as e:
        logger.error(e, exc_info=True)
        return Response(status=500)


@bp.route('/api/rest/v1/users', methods=['POST'])
@login_required
def add_user():
    try:
        user_id = new_user(request.json)
        if user_id is not None:
            return Response(json.dumps({"id": user_id}), status=201, headers={"Content-Type": "application/json"})
        else:
            return Response("The maximum user limit has been reached", status=409)
    except Exception as e:
        logger.error(e, exc_info=True)
        return Response(status=500)


@bp.route('/api/rest/v1/users', methods=['PUT'])
@login_required
def edit_user():
    try:
        update_successful = update_user(request.json)
        if update_successful:
            return Response(status=200)
        else:
            return Response(status=404)
    except Exception as e:
        logger.error(e, exc_info=True)
        return Response(status=500)


@bp.route('/api/rest/v1/users', methods=['DELETE'])
@login_required
def delete_users():
    try:
        users_not_removed = remove_users(request.json)
        return Response(json.dumps({"users_not_removed": users_not_removed}), status=200)
    except Exception as e:
        logger.error(e, exc_info=True)
        return Response(status=500)