# standard library imports
import logging
from logging.handlers import RotatingFileHandler
import os
import sys
import threading
from datetime import timedelta
import signal
from threading import Thread

import time

from flask import Flask, request, redirect
from pathlib import Path
# related third party imports


from barix.web.restart_services import restartServicesFromFile
from barix.web.constants import SSL_CERTIFICATES_DIR
from barix.web.uci import getValueOfUci
from barix.web.certificates import generateSSLCerts, certificateIsValid, keyIsValid
from barix.web.auth import generateNewSessionKey

from io import BytesIO
from tempfile import TemporaryFile, SpooledTemporaryFile
from flask.wrappers import Request

# # Get current directory
# current = os.path.dirname(os.path.realpath(__file__))
# # Get parent directory
# parent = os.path.dirname(current)
# # Append parent directory
# sys.path.append(parent)

# from application.MainController import MainController
from lib.utils import uci_to_config_dict, check_uci_icpgw
# from application.Modbus import Modbus

from lib.backup_settings import BackupSettingsWD

UPLOAD_FILES_DIR = "/mnt/data/uploads"
CONFIG_DIR = "/mnt/data/config"
PREGONG_DIR = "/mnt/data/pregong"
PROCESSED_PREGONG_DIR = "/mnt/data/pregong/processed"
MAX_FILE_SIZE_UPLOAD = 256 * 1024 * 1024  # 256Mb


"""
Init logger
"""

rotHandler = RotatingFileHandler('/var/log/webui-backend.log', maxBytes=2*1024*1024, backupCount=1)
logging.basicConfig(
    level=logging.DEBUG,
    handlers=[rotHandler], 
    format='%(asctime)s [%(levelname)-7s] %(filename)26s:%(lineno)3s %(threadName)18s | %(message)s'
)
logger = logging.getLogger('flask-backend')
logger.info("Starting flask backend v2.0...")


def show_routes(app):
    logger.debug("Available Endpoints:")
    for rule in app.url_map.iter_rules():
        methods = ','.join(rule.methods)
        logger.debug(f"{rule.rule:40s} [{methods+']':35} - {rule.endpoint}")


def create_non_existing_directories():
    """
    Create non-existing directories
    """
    dir_list = [UPLOAD_FILES_DIR, CONFIG_DIR, PREGONG_DIR, PROCESSED_PREGONG_DIR]
    for dir in dir_list:
        p = Path(dir)
        if not p.exists():
            try:
                p.mkdir()
            except Exception as e:
                logger.error("Error creating directory {}: {}".format(dir, e))


class MemoryRequest(Request):
    def _get_file_stream(self, total_content_length, content_type, filename, content_length):
        max_size = 1024 * 500
        if SpooledTemporaryFile is not None:
            return SpooledTemporaryFile(max_size=max_size, mode="rb+", dir=UPLOAD_FILES_DIR)
        elif total_content_length is None or total_content_length > max_size:
            return TemporaryFile(mode="rb+", dir=UPLOAD_FILES_DIR)
        return BytesIO()


# def checkIfWebProtocolChanged(ucisChanged):
#     try:
#         # check if web protocol changed to HTTPS
#         for uciTuple in ucisChanged:
#             if ('httpd', 'webserver', 'protocol') == uciTuple:
#                 generateNewSessionKey()
#                 if getValueOfUci('httpd', 'webserver', 'protocol') == 'https':
#                     generateSSLCerts()
#                 break
#     except Exception as e:
#         raise e



"""
Check if has services to restart
"""
restartServicesFromFile()

create_non_existing_directories()

# p = Path(UPLOAD_FILES_DIR)
# if not p.exists():
#     try:
#         p.mkdir()
#         #logger.debug("Directory {} successfully created".format(UPLOAD_FILES_DIR))
#     except Exception as e:
#         logger.error("Error creating directory {}: {}".format(UPLOAD_FILES_DIR, e))
#
# p = Path(CONFIG_DIR)
# if not p.exists():
#     try:
#         p.mkdir()
#     except Exception as e:
#         logger.error("Error creating directory {}: {}".format(CONFIG_DIR, e))


class Webui:

    @staticmethod
    def init(app):
        # config = uci_to_config_dict()
        # _main_controller = MainController()
        # _main_controller.load_configs(config)

        # _modbus = Modbus(main_controller=_main_controller)
        # _modbus.load_configs(config)
        # _modbus.run()

        """
        Init Flask application
        """
        # static_url_path='' removes any preceding path from the URL (i.e. the default /static).
        # static_folder='web/static' to serve any files found in the folder web/static as static files.
        # template_folder='web/templates' similarly, this changes the templates folder.
        
        # secret key used to sign session cookies for protection against cookie data tampering.
        # TODO: get secret key from anywhere else, like env variable, for security (?)
        app.secret_key = b'\xb3\xa482C\x98\x81\xa9\x8e5\xe9\x7f\xe2\xf9y/'
        # UPLOAD_FOLDER is where uploaded files are stored
        app.config['UPLOAD_FOLDER'] = UPLOAD_FILES_DIR
        # maximum size allowed for the files to upload
        app.config['MAX_CONTENT_LENGTH'] = MAX_FILE_SIZE_UPLOAD
        # override default werkzeug formparser to allow to update files to the UPLOAD_FOLDER instead of default
        app.request_class = MemoryRequest

        """
        Implement session login timeout is login is required
        """
        if getValueOfUci('httpd', 'webserver', 'password_set') == "true":
            session_timeout = getValueOfUci('httpd', 'webserver', 'session_timeout')
            if session_timeout == "":
                session_timeout = "30"
            logger.info("Using session timeout value: {}".format(session_timeout))
            app.config['PERMANENT_SESSION_LIFETIME'] = timedelta(minutes=int(session_timeout))
        else:
            logger.info("Login is not required => no session timeout.")
        return app

    @staticmethod
    def include_routes_to_frontend(app):
        try:
            from frontendRoutes import frontendRoutes
            app.register_blueprint(frontendRoutes)
            # logger.debug("Imported frontend routes successfully")
        except Exception as e:
            logger.debug("It was not possible to import frontend routes: {}".format(e))
        return app

    @staticmethod
    def include_standard_routes(app):
        from SettingsAPI import SettingAPI
        settings_api = SettingAPI()
        app.register_blueprint(settings_api.blueprint)

        import auth_API
        app.register_blueprint(auth_API.bp)
        import logs_API
        app.register_blueprint(logs_API.bp)
        import reboot_API
        app.register_blueprint(reboot_API.bp)
        import reset_to_defaults_API as defaults_API
        app.register_blueprint(defaults_API.bp)
        import system_info_API
        app.register_blueprint(system_info_API.bp)
        import uci_API
        app.register_blueprint(uci_API.bp)
        import fw_update_API
        app.register_blueprint(fw_update_API.bp)
        import ca_certificates_API
        app.register_blueprint(ca_certificates_API.bp)

        import users_API
        app.register_blueprint(users_API.bp)
        import groups_API
        app.register_blueprint(groups_API.bp)
        import audio_files_API
        app.register_blueprint(audio_files_API.bp)

        
        return app



app = Flask(__name__, static_url_path='', static_folder='static', template_folder='static')

ret_val = check_uci_icpgw()
if not ret_val:
    logger.warning("uci was not configured correctly, setting default values")

backup_files = BackupSettingsWD(["*_backup.json"])

logger.info("Starting BackupFiles WD...")
backup_files_thread = Thread(
    target=backup_files.run,
    daemon=True,
    name="BackupFiles"
)
backup_files_thread.start()

logger.info("Initializing Webui...")

app = Webui.init(app)
app = Webui.include_routes_to_frontend(app)
app = Webui.include_standard_routes(app)


show_routes(app)

if __name__ == "__main__":
    app.run(host='127.0.0.1', port=5000, threaded=True)

