import json
import logging
import os
from restart_services import restartApp

GROUPS_FILE_PATH = "/barix/apps/icpgw/groups_backup.json"

logger = logging.getLogger('flask-backend')

ALL_GROUP_IDS = set([str(i) for i in range(1, 129)])


def get_groups():
    # read list of users from file
    if not os.path.isfile(GROUPS_FILE_PATH):
        return {}

    with open(GROUPS_FILE_PATH, 'r', encoding='utf-8') as file:
        return json.load(file)


def save_groups_and_restart(users):
    with open(GROUPS_FILE_PATH, 'w', encoding='utf-8') as file:
        json.dump(users, file)

    # restart service
    restartApp()


def validate_clients(clients) -> tuple[bool, str]:
    if not isinstance(clients, list):
        logger.error(f"{clients} is not a list")
        return False, "List of clients is not a list"
    elif len(clients) == 0:
        logger.error("Empty client list")
        return False, "List of clients is empty"
    else:
        return True, ""


def get_available_id(groups) -> int | None:
    available_str_ids = ALL_GROUP_IDS - set(groups.keys())
    if len(available_str_ids) != 0:
        return min(int(x) for x in available_str_ids)
    else:
        return None


def new_group(group) -> tuple[str | None, str]:
    groups = get_groups()
    new_id = get_available_id(groups)
    valid_clients, msg = validate_clients(group["clients"])
    if new_id is not None and valid_clients:
        groups[str(new_id)] = {"name": group["name"], "clients": group["clients"]}
        save_groups_and_restart(groups)
    return str(new_id), msg


def update_group(group):
    groups = get_groups()
    group_id = str(group["id"])
    if group_id in groups:
        valid_clients, msg = validate_clients(group["clients"])
        if valid_clients:
            groups[group_id]["name"] = group["name"]
            groups[group_id]["clients"] = group["clients"]
            save_groups_and_restart(groups)
        return valid_clients, msg
    else:
        logger.error(f"Group id {group_id} doesn't exist!")
        return False, f"Group id {group_id} not found"


def remove_groups(group_ids):
    groups = get_groups()
    groups_not_removed = []
    for gid in group_ids:
        group_id = str(gid)
        if group_id in groups:
            del groups[group_id]
            logger.info(f"Group {group_id} deletion successful!")
        else:
            groups_not_removed.append(group_id)
            logger.warning(f"Group id {group_id} doesn't exist!")
    save_groups_and_restart(groups)
    return groups_not_removed
