import json
import logging
import os
import shutil
import subprocess
import uuid
from datetime import datetime
from typing import Any

from restart_services import restartApp, restartModbus
from users import get_users, save_users

from barix.web.system_info import getStorageSize
from barix.web.uci import setUciConfigs

AUDIO_FILES_PATH = "/barix/apps/icpgw/audio_files_backup.json"
PRE_RECORDED_MESSAGES_DIR = "/mnt/data/prerecorded_messages"
PROCESSED_PRE_RECORDED_MESSAGES_DIR = os.path.join(PRE_RECORDED_MESSAGES_DIR, "processed")
PREGONG_FILES_DIR = "/mnt/data/pregong"
PROCESSED_PREGONG_FILES_DIR = os.path.join(PREGONG_FILES_DIR, "processed")

logger = logging.getLogger('flask-backend')

ALL_AUDIO_FILES_IDS = set([str(i) for i in range(1, 101)])


def get_available_id(audio_files) -> int | None:
    available_str_ids : set[str] = ALL_AUDIO_FILES_IDS - set(audio_files.keys())
    if len(available_str_ids) != 0:
        return min(int(x) for x in available_str_ids)
    else:
        return None


def get_audio_files_info():

    if not os.path.isfile(AUDIO_FILES_PATH):
        return {}

    with open(AUDIO_FILES_PATH, 'r', encoding='utf-8') as file:
        return json.load(file)


def save_audio_files_info(data):
    with open(AUDIO_FILES_PATH, 'w', encoding='utf-8') as file:
        json.dump(data, file)


def get_modbus_idx_in_use(audio_files) -> list[int]:
    return [audio_file["modbus_idx"] for audio_file in audio_files.values() if "modbus_idx" in audio_file]


def handleFileUploaded(fileUploaded, destination, filename=None) -> str:
    # if user does not select file, browser also
    # submit an empty part without filename
    if fileUploaded.filename == '':
        logger.error("Invalid file")
        raise Exception("Invalid file")

    if filename is None:
        # generate random file name
        filename = str(uuid.uuid4())

    # Get size of file in memory to check if it has enough space
    fileUploaded.seek(0, os.SEEK_END)
    file_length = fileUploaded.tell()
    fileUploaded.seek(0)  # Reset pointer to the beginning
    if not has_enough_space(file_length, path=destination):
        logger.error("Insufficient Storage")
        raise Exception("Insufficient Storage")

    # Save the file
    final_filepath = os.path.join(destination, filename)
    fileUploaded.save(final_filepath)

    return filename


def has_enough_space(required_space_bytes, path='/', min_free_bytes=50 * 1024 * 1024):
    _, _, free = shutil.disk_usage(path)
    return (free - required_space_bytes) > min_free_bytes


def validate_modbus_idx(audio_files, modbus_idx: int) -> bool:
    if modbus_idx is None:
        return True
    if 0 < int(modbus_idx) < 1000:
        if int(modbus_idx) not in get_modbus_idx_in_use(audio_files):
            return True
        else:
            logger.error("Modbus index already in use: {}".format(modbus_idx))
            return False
    else:
        logger.error("Modbus index invalid: {}".format(modbus_idx))
        return False


def create_timestamp() -> str:
    timestamp_date = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    return str(timestamp_date)


def new_audio_file(filesUploaded, file_data) -> tuple[str, int]:
    final_filename = ""

    audio_files = get_audio_files_info()

    new_audio_file_id = get_available_id(audio_files)

    if new_audio_file_id is not None:

        new_audio_file_id = str(new_audio_file_id)

        try:
            if 'audio_file' not in filesUploaded:
                logger.error("Invalid file uploaded")
                return "Invalid file uploaded", 400

            uploaded_file = filesUploaded['audio_file']
            try:
                final_filename = handleFileUploaded(uploaded_file, PRE_RECORDED_MESSAGES_DIR)
            except Exception as e:
                if str(e) == "Invalid file":
                    return "Invalid file", 400
                elif str(e) == "Insufficient Storage":
                    return 'Not enough disk space', 507
                else:
                    return '', 500

            audio_file_button_name = "Audio File " + new_audio_file_id if file_data["name"].strip() == "" else file_data["name"]
            modbus_idx: int = int(file_data["modbus_idx"]) if ("modbus_idx" in file_data and file_data["modbus_idx"] is not None) else None
            filename = final_filename

            try:
                valid_modbus_idx = validate_modbus_idx(audio_files, modbus_idx)
            except Exception as e:
                logger.error("Error validating modbus index: {}".format(e))
                return "Modbus index is not valid", 400

            if valid_modbus_idx:
                timestamp: str = create_timestamp()
                logger.info("Finished adding new file")

                # The json entry is generated with the acquired parameters
                audio_files[new_audio_file_id] = {
                    "btn_name": audio_file_button_name,
                    "file_name": filename,
                    "timestamp": timestamp,
                }

                if modbus_idx is not None:
                    audio_files[new_audio_file_id]["modbus_idx"] = modbus_idx

                save_audio_files_info(audio_files)

                # update users with audio file if audio_file upload was successful
                if len(file_data["users"]) > 0:

                    users_dict = get_users()
                    changed = False

                    # for each user
                    for user_id in users_dict:
                        # if user id is in list from request
                        if user_id in file_data["users"]:
                            if "audio_files" in users_dict[user_id].keys():
                                users_dict[user_id]['audio_files'].append(new_audio_file_id)
                            else:
                                users_dict[user_id]['audio_files'] = [new_audio_file_id]
                            changed = True

                    if changed:
                        # update users
                        save_users(users_dict)

                restartApp()

                logger.info(f"Returning {new_audio_file_id}, {201}")

                return new_audio_file_id, 201
            else:
                return "Modbus index is not valid", 400
        except Exception as e:
            logger.error(f"New device message uploaded failed: {e}", exc_info=True)
            # logger.warning("New device message uploaded failed! Removing physical audio files (if any)")
            # if final_filename != "":
            #     raw_file_path = os.path.join(PRE_RECORDED_MESSAGES_DIR, final_filename)
            #     if os.path.isfile(raw_file_path):
            #         logger.info(f"Removing {raw_file_path}")
            #         os.remove(raw_file_path)
            return '', 500
    else:
        return "The maximum audio files limit has been reached", 409


def update_audio_file(audio_file_info, uploaded_file) -> tuple[str, int]:
    try:
        audio_file_button_id: str = audio_file_info["id"]
        audio_files = get_audio_files_info()

        if audio_file_button_id in audio_files:

            if uploaded_file is not None:

                try:
                    filename = handleFileUploaded(uploaded_file, PRE_RECORDED_MESSAGES_DIR)
                except Exception as e:
                    if str(e) == "Invalid file":
                        return "Invalid file", 400
                    elif str(e) == "Insufficient Storage":
                        return 'Not enough disk space', 507
                    else:
                        return '', 500

                timestamp: str = create_timestamp()
            else:
                # if file did not change
                # filename is the previous one
                filename: str = audio_files[audio_file_button_id]["file_name"]
                # timestamp is the previous one
                timestamp: str = audio_files[audio_file_button_id]["timestamp"]

            audio_file_button_name: str = audio_file_info["name"]
            modbus_idx: int = None if audio_file_info["modbus_idx"] == "" else audio_file_info["modbus_idx"]

            try:
                valid_modbus_idx = validate_modbus_idx(audio_files, modbus_idx)
            except Exception as e:
                logger.error("Error validating modbus index: {}".format(e))
                return "Modbus index is not valid", 400

            if valid_modbus_idx:

                # Check if the file name has changed. If so, encode the new file and delete the old one
                if filename != audio_files[audio_file_button_id]['file_name']:
                    logger.info("Updating associated audio file for \"%s\"",
                                   audio_files[audio_file_button_id]['btn_name'])
                    logger.info("Removing the old file: %s", audio_files[audio_file_button_id]['file_name'])
                    file = os.path.join(PRE_RECORDED_MESSAGES_DIR, audio_files[audio_file_button_id]['file_name'])
                    if os.path.isfile(file):
                        os.remove(file)
                        logger.info(f"Remove success: {audio_files[audio_file_button_id]['file_name']}")
                        logger.info("New filename: %s", filename)
                    else:
                        logger.warning(f"{file} doesn't exist")

                audio_files[audio_file_button_id] = {
                    "btn_name": audio_file_button_name,
                    "file_name": filename,
                    "timestamp": timestamp
                }

                if modbus_idx is not None:
                    audio_files[audio_file_button_id]["modbus_idx"] = modbus_idx

                save_audio_files_info(audio_files)

                # update users with audio file if audio_file upload was successful
                users_dict = get_users()
                changed = False

                # for each user
                for user_id in users_dict:
                    # if user_id in audio_file list of users
                    if user_id in audio_file_info["users"]:
                        if "audio_files" in users_dict[user_id].keys():
                            if audio_file_button_id not in users_dict[user_id]['audio_files']:
                                users_dict[user_id]['audio_files'].append(audio_file_button_id)
                                changed = True
                        else:
                            users_dict[user_id]['audio_files'] = audio_file_button_id
                            changed = True

                    else:
                        if "audio_files" in users_dict[user_id].keys() and audio_file_button_id in users_dict[user_id]['audio_files']:
                            # remove it
                            users_dict[user_id]['audio_files'] = [x for x in users_dict[user_id]['audio_files'] if x != audio_file_button_id]
                            changed = True

                if changed:
                    # update users
                    save_users(users_dict)

                restartApp()

                return '', 200
            else:
                return "Modbus index is not valid", 400
        else:
            return "Audio file doesn't exist!", 404

    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


def remove_audio_files(audio_file_ids) -> list[str]:
    audio_files_info = get_audio_files_info()
    audio_files_not_removed = []
    users = get_users()
    for audio_file_id in audio_file_ids:
        audio_file_id = str(audio_file_id)
        if audio_file_id in audio_files_info:
            logger.info(f'removing {audio_file_id}')
            for user_id in users:
                if "audio_files" in users[user_id].keys() and audio_file_id in users[user_id]['audio_files']:
                    logger.warning(f"Audio file {audio_file_id} is assigned to user {user_id}. Unassigning it.")
                    users[user_id]['audio_files'].remove(audio_file_id)

            file_to_delete = audio_files_info[audio_file_id][str("file_name")]

            logger.debug("Deleting physical audio files")
            del audio_files_info[audio_file_id]
            file = os.path.join(PRE_RECORDED_MESSAGES_DIR, file_to_delete)
            if os.path.isfile(file):
                os.remove(file)
                logger.info(f"Remove success: {audio_file_id}")
            else:
                logger.warning(f"{file} doesn't exist, but audio file {audio_file_id} deleted anyway")

    save_users(users)
    save_audio_files_info(audio_files_info)
    restartApp()

    return audio_files_not_removed

# def remove_audio_files(audio_file_ids) -> tuple[list[str], dict[str, list[Any]]]:
#     audio_files_info = get_audio_files_info()
#     audio_files_not_found = []
#     audio_files_in_use = {}
#     for audio_file_id in audio_file_ids:
#         audio_file_id = str(audio_file_id)
#         if audio_file_id not in audio_files_info:
#             audio_files_not_found.append(audio_file_id)
#             logger.warning(f"Audio file deletion unsuccessful, {audio_file_id} doesn't exist")
#
#         # check if file is in use (assigned to users)
#         users = get_users()
#         for user_id in users:
#             if "audio_files" in users[user_id].keys() and audio_file_id in users[user_id]['audio_files']:
#                 logger.warning(f"Audio file deletion unsuccessful, {audio_file_id} is assigned to user {user_id}")
#                 if audio_file_id in audio_files_in_use:
#                     audio_files_in_use[audio_file_id].append(user_id)
#                 else:
#                     audio_files_in_use[audio_file_id] = [user_id]
#
#     if len(audio_files_not_found) == 0 and len(audio_files_in_use) == 0:
#         for audio_file_id in audio_file_ids:
#             audio_file_id = str(audio_file_id)
#
#             file_to_delete = audio_files_info[audio_file_id][str("file_name")]
#
#             logger.debug("Deleting physical audio files")
#             del audio_files_info[audio_file_id]
#             file = os.path.join(PRE_RECORDED_MESSAGES_DIR, file_to_delete)
#             if os.path.isfile(file):
#                 os.remove(file)
#                 logger.info(f"Remove success: {audio_file_id}")
#             else:
#                 logger.warning(f"{file} doesn't exist, but audio file {audio_file_id} deleted anyway")
#
#
#         save_audio_files_info(audio_files_info)
#         restartApp()
#
#     return audio_files_not_found, audio_files_in_use


def store_pregong_file(filesUploaded) -> tuple[str, int]:
    filename = ""
    try:
        if 'file' not in filesUploaded:
            logger.error("Invalid file uploaded")
            return "Invalid file uploaded", 400

        uploaded_file = filesUploaded.get('file')

        try:
            filename = handleFileUploaded(uploaded_file, PREGONG_FILES_DIR, uploaded_file.filename)
        except Exception as e:
            if str(e) == "Invalid file":
                return "Invalid file", 400
            elif str(e) == "Insufficient Storage":
                return 'Not enough disk space', 507
            else:
                return '', 500

        # remove previous file (if any)
        for file in os.listdir(PREGONG_FILES_DIR):
            complete_path = os.path.join(PREGONG_FILES_DIR, file)
            if os.path.isfile(complete_path) and file != filename:
                logger.info(f"Deleting previous pregong file {file}")
                os.remove(complete_path)

        restartModbus()

        return '', 200
    except Exception as e:
        logger.error(e, exc_info=True)
        logger.info("New device message uploaded failed! Removing physical audio files")
        raw_file_path = os.path.join(PREGONG_FILES_DIR, filename)
        if os.path.isfile(raw_file_path):
            os.remove(raw_file_path)
        return '', 500


def get_pregong_file() -> tuple[str, int]:
    try:
        response_json = {"filename": ""}
        for file in os.listdir(PREGONG_FILES_DIR):
            if os.path.isfile(os.path.join(PREGONG_FILES_DIR, file)):
                response_json["filename"] = file

        return json.dumps(response_json), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


def remove_pregong_file() -> tuple[str, int]:
    try:
        for file in os.listdir(PREGONG_FILES_DIR):
            complete_path = os.path.join(PREGONG_FILES_DIR, file)
            if os.path.isfile(complete_path):
                logger.info(f"Deleting previous pregong file {file}")
                os.remove(complete_path)

        setUciConfigs(
            {"icpgw.pregong.enabled": False},
            commit=True,
            restartServices=False
        )

        restartApp()
        restartModbus()

        return '', 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500

