import asyncio

from pymodbus.datastore import (
    ModbusSequentialDataBlock,
    ModbusServerContext,
    ModbusSlaveContext,
)

# from pymodbus.server import StartAsyncTcpServer, ServerAsyncStop
from pymodbus.server import StartTcpServer, ServerStop
from pymodbus.transaction import ModbusSocketFramer
from pymodbus.device import ModbusDeviceIdentification

import time
from threading import Thread

from lib.constants import Constants

import json

import logging
from logging.handlers import RotatingFileHandler

rotHandler = RotatingFileHandler('/var/log/modbus.log', maxBytes=2*1024*1024, backupCount=1)
logging.basicConfig(
    level=logging.DEBUG,
    handlers=[rotHandler], 
    format='%(asctime)s [%(levelname)-7s] %(filename)26s:%(lineno)3s %(threadName)18s | %(message)s'
)
logger = logging.getLogger(__name__)


from lib.utils import uci_to_config_dict

import requests

class ICPGW():
    def __init__(self):
        self.url = "http://127.0.0.1:4000/api/modbus/v1"

    def _get(self, path):
        url = self.url + path
        try:
            response = requests.get(url)
            return response.content, response.status_code
        except Exception as e:
            # TODO add path
            logger.error(e, exc_info=True)
            return '', 500

    def _post(self, path, data=None, headers=None):
        url = self.url + path
        try:
            response = requests.post(url, data=data, headers=headers)
            return response.content, response.status_code
        except Exception as e:
            # TODO add path
            logger.error(e, exc_info=True)
            return '', 500

    def lock(self):
        return self._post("/lock")
        
    def unlock(self):
        return self._post("/unlock")
        
    def stop(self):
        return self._post("/stop")
        
    def play_file(self, data):
        headers = {
            'Content-Type': 'application/json'
        }
        return self._post("/play-file", json.dumps(data), headers)

    def get_status(self):
        msg, status_code = self._get("/status")
        if status_code == 200:
            data = json.loads(msg)
            logger.debug(f"status {data['status']}")
            return data['status']
        return None



class CallbackDataBlock(ModbusSequentialDataBlock):
    """
    A data block that stores the new value in memory,
    and passes the operation to a message queue for further processing.
    """

    def __init__(self, queue, addr, values, modbus):
        """Initialize."""
        logger.info("init CallbackDataBlock")
        self.queue = queue
        self._modbus = modbus

        super().__init__(addr, values)

    def setValues(self, address, value, callback_enabled=True):
        """Set the requested values of the datastore."""
        super().setValues(address, value)
        txt = f"setValues with address {address}, value {value}"
        logger.debug(txt)

        if address == 2 and callback_enabled:
            self._modbus.process_request()

    def getValues(self, address, count=1):
        """Return the requested values from the datastore."""
        result = super().getValues(address, count=count)
        txt = f"getValues with address {address}, count {count}, data {result}"
        logger.debug(txt)

        return result

    def validate(self, address, count=1):
        """Check to see if the request is in range."""
        result = super().validate(address, count=count)
        txt = f"validate with address {address}, count {count}, data {result}"
        logger.debug(txt)
        return result



class Modbus:
    """

    """

    REG_01_IDX = 2
    REG_02_IDX = 3
    REG_03_IDX = 4

    def __init__(self):
        logger.info("initializing Modbus...")

        self._config = uci_to_config_dict()

        self.pregong_enabled = self._config["pregong_enabled"]
        self.port = self._config["modbus"]["port"]

        self.status = Constants.IDLE

        self._reg01 = 0
        self._reg02 = 0
        self._reg03 = 0

        self.filepath = ""

        self._queue = asyncio.Queue()
        self._block = CallbackDataBlock(self._queue, 0x00, [0] * 5, self)

        self.icpgw = ICPGW()

        self._kill_worker = False
        self._server_running = False

        # Threads
        self._worker_thread = None


    def stop(self):
        self._kill_worker = True
        self._wait_worker_thread()
        logger.debug("stopping modbus")

        if self._server_running:
            # asyncio.run(_stop_modbus_server())
            logger.debug("send ServerStop")
            ServerStop()
            while self._server_running:
                time.sleep(0.001)

        logger.debug("modbus stopped")

    def run_modbus_server(self):
        # async def _run_server(config: dict):
        """Define datastore callback for server and do setup."""

        # store = ModbusSlaveContext(di=None, co=None, hr=config['block'], ir=None)
        store = ModbusSlaveContext(di=None, co=None, hr=self._block, ir=None)
        context = ModbusServerContext(slaves=store, single=True)

        """Run server."""
        address = ("", self.port)

        logger.info(f"Starting Modbus server, listening on {address}")

        identity = ModbusDeviceIdentification(
            info_name={
                "VendorName": "Pymodbus",
                "ProductCode": "PM",
                "VendorUrl": "https://github.com/pymodbus-dev/pymodbus/",
                "ProductName": "Pymodbus Server",
                "ModelName": "Pymodbus Server",
                "MajorMinorRevision": "3.4.1",  # pymodbus_version,
            }
        )

        self._server_running = True
        StartTcpServer(
            context=context,  # Data storage
            identity=identity,  # server identify
            # TBD host=
            # TBD port=
            address=address,  # listen address
            # custom_functions=[],  # allow custom handling
            framer=ModbusSocketFramer,  # The framer strategy to use
            # ignore_missing_slaves=True,  # ignore request to a missing slave
            # broadcast_enable=False,  # treat slave_id 0 as broadcast address,
            # timeout=1,  # waiting time for request to complete
            # TBD strict=True,  # use strict timing, t1.5 for Modbus RTU
        )
        logger.info("exit StartTcpServer")
        self._server_running = False


###############################################################################
#
# 
#
###############################################################################

    def _wait_idle_state(self):
        while self.icpgw.get_status() != "IDLE":
            time.sleep(0.05)

    def process_request(self):
        logger.info("received modbus write to address 1 register")
        pass

        # add modbus logic here???
        data = self._block.getValues(2, 3)
        self._reg01 = data[0]
        self._reg02 = data[1]
        self._reg03 = data[2]
        logger.debug(f"data: {data}")

        # Modbus already running
        if self.status == Constants.BUSY:
            if self._reg01 == 0:  # received stop command
                self._kill_worker = True
                
                self.icpgw.stop()
                self._worker_thread.join()
                self._wait_idle_state()

                self._block.setValues(2, [0, 0, 0], False)
                self.status = Constants.IDLE
            else:
                logger.warning("command not allowed, modbus is already running")
            return

        # modbus already stopped and receive 0 in register 1
        if self.status == Constants.IDLE and self._reg01 == 0:
            return  # do nothing

        # TODO . improve?
        # verify if file exists
        # self.filepath = get_file_name_from_modbus_idx(self._reg01)
        # if not self.filepath:
        #     return

        # Able to start modbus
        self.status = Constants.BUSY
        self.icpgw.lock()
        self._kill_worker = False
        self._worker_thread = Thread(target=self._worker, daemon=True, name="worker_Modbus")
        self._worker_thread.start()

    def _worker(self):
        logger.info("starting worker")

        # if audio is running stop
        if self.icpgw.get_status()!="IDLE":
            self.icpgw.stop()
            self._wait_idle_state()

        if self.pregong_enabled and not self._kill_worker:
            logger.debug("play pregong")
            # play pregong
            data_play = {
                'pregong': True,
                'client_idx': self._reg02,
            }
            
            self.icpgw.play_file(data_play)
            self._wait_idle_state()

        # play file in loop
        data_play = {
            'pregong': False,
            'file_idx': self._reg01,
            'client_idx': self._reg02,
        }
        counter = self._reg03
        while (counter or self._reg03 == 0) and not self._kill_worker:
            logger.debug("play file")
            # http request
            self.icpgw.play_file(data_play)
            self._wait_idle_state()

            # update registers
            self._block.setValues(self.REG_03_IDX, counter, False)
            counter = counter - 1

        self.icpgw.unlock()
        logger.info("worker complete")
        self.status = Constants.IDLE

    def _wait_worker_thread(self):
        if self._worker_thread is not None:
            self._worker_thread.join()
            self._worker_thread = None


if __name__ == "__main__":
    modbus = Modbus()
    modbus.run_modbus_server()
