import time
import socket
from datetime import datetime
import struct
import logging

from BARP.barp import BARP
from barix.system.barix_enums import BARPProtocol

from lib.utils import is_multicast, is_broadcast

import logging
logger = logging.getLogger(__name__)


class PagingMaster():

    PAGING_REQUEST_TIME_INTERVAL = 3
    STATUS_REPORT_TIME_INTERVAL = 5


    def __init__(self, config: dict):

        # Load all config.
        self._all_config = config
        master_id = int(config["paging_master_id"])
        self.barp = BARP(master_id)

        self._paging_priority = int(config["paging_priority"])
        self._device_ip = config["device_ip_settings"]["inet"]
        self._device_netmask = config["device_ip_settings"]["netmask"]
        self._broadcast = config["device_ip_settings"]["broadcast"]
        self.header_kwargs = config["barp_header_kwargs"]
        self.header_kwargs["sender_id"] = master_id
        self._ports = config["ports"]
        self._audio_device = config["audio_device"]
        self._encoding_type = config["encoding_type"]
        self._multicast_group_address = config["multicast_group_address"]
        self._page_group_request_address = "0.0.0.0"
        self._page_group_request_port = 0
        self.__state = BARPProtocol.NOT_USED  # starts with IDLE state.
        self._barp_ip = config["barp_ip"]
        self.active_devices = {}


        # Flags
        self.capture_disabled = False
        self._exit = False
        self._current_clients_list = None  # If the app crashes during operation, try to signal users
        # Address and port to be used for paging. They are changed depending on the IPs
        self._page_to_address = self._barp_ip
        self._page_to_port = self._ports["audio"]

        self._kill_page_group = False

        # Socket initializations.
        self.listen_status_socket = None
        self.status_socket = None
        self.control_socket = None
        self.call_socket = None
        self.audio_socket = None
        try:
            self.initialize_sockets()
        except Exception as e:
            logger.exception(f"While trying to initialize_sockets, following occurred: {e}")
            exit()

    @property
    def page_to_address(self):
        return self._page_to_address

    @property
    def page_to_port(self):
        return self._page_to_port

    def addr_tuple(self) -> tuple:
        return self.page_to_address, self.page_to_port

    @property
    def exit_flag(self):
        # TODO: To be changed to running and return not self._exit
        return self._exit

    @property
    def state(self):
        """
        Possibilites:
        0(IDLE) and 130(Sending the message)
        """
        return self.__state

    @state.setter
    def state(self, new_state: BARPProtocol):
        if new_state != self.__state:
            logger.warning(f"State change: {self.__state} -> {new_state}")
            self.__state = new_state

    @property
    def control_port(self):
        """
        Port used for sending the control messages.
        """
        return self._ports["control"]

    @property
    def audio_port(self):
        """
        Port used for sending the audio message/stream/call
        """
        return self._ports["audio"]

    @property
    def status_port(self):
        """
        Port used for sending the status message
        """
        return self._ports["status"]

    @property
    def encoding_type(self):
        """
        Encoding type to be used for the audio message.
        """
        return self._encoding_type

    @property
    def master_audio_port(self):
        # TODO: Currently there is no master_audio port available
        # return self._ports["master_audio"]
        return self.audio_port  # return the audio port instead for the moment

    @property
    def multicast_group(self):
        """
        Multicast group address. To be used while sending the paging message.
        """
        if is_multicast(self._multicast_group_address):
            logger.warning(f"Multicast address: {self._multicast_group_address}")
            return self._multicast_group_address
        else:
            return self.barp_ip

    @property
    def page_group_request_address(self):
        return self._page_group_request_address

    @property
    def page_group_request_port(self):
        return self._page_group_request_port

    # TODO: Implement logic to check what the given IP is
    @property
    def barp_ip(self):
        if is_broadcast(self._barp_ip, self._broadcast):
            return self._broadcast
        else:
            return self._barp_ip

    @property
    def record_device(self):
        return self._audio_device["record"]

    @property
    def playback_device(self):
        return self._audio_device["playback"]

    @property
    def paging_priority(self):
        return self._paging_priority

    def send_data_audio_socket(self, data):
        self.audio_socket.sendto(
            data,
            (
                self._page_to_address,
                self._page_to_port,
            ),
        )

    def initialize_sockets(self):
        """
        Initializes the socket for sending the status of master (status_socket),
        receiving status message from other devices (listen_status_socket),
        sending control message (control_socket) and sending audio (audio socket).
        """
        self.listen_status_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)

        self.status_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM, socket.IPPROTO_UDP)
        self.control_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM, socket.IPPROTO_UDP)
        # Avoid having Errno 98
        self.control_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.call_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM, socket.IPPROTO_UDP)

        if is_multicast(self.barp_ip):
            self.control_socket.setsockopt(socket.SOL_SOCKET, socket.IP_MULTICAST_TTL, 1)
            self.status_socket.setsockopt(socket.SOL_SOCKET, socket.IP_MULTICAST_TTL, 1)
            # self.listen_status_socket.setsockopt(socket.SOL_SOCKET, socket.IP_MULTICAST_TTL, 1)

            # logger.info("Binding socket")
            # self.listen_status_socket.bind((self.barp_ip, self.status_port))
            # logger.info("Done")
        else:
            self.control_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            self.status_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            # self.listen_status_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            #
            # logger.info("Binding socket")
            # self.listen_status_socket.bind(("", self.status_port))
            # logger.info("Done")

        self.control_socket.bind(("", self.control_port))

        # Audio socket
        self.audio_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM, socket.IPPROTO_UDP)
        # Avoid having Errno 98
        self.audio_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

        if is_multicast(self._multicast_group_address):
            logger.info("Audio Multicast Group IP is multicast")
            self.audio_socket.setsockopt(socket.IPPROTO_IP, socket.IP_MULTICAST_TTL, 1)
            self._page_group_request_address = self._multicast_group_address
            self._page_group_request_port = self.master_audio_port

            self._page_group_request_address = self._multicast_group_address
            self._page_group_request_port = self.master_audio_port

            self.audio_socket.bind((self.page_group_request_address, self.page_group_request_port))

        else:
            logger.info("Audio Multicast Group IP is NOT multicast.")

            self._page_group_request_address = "0.0.0.0"
            self._page_group_request_port = self.audio_port

            if is_multicast(self.barp_ip):
                logger.info("Barp IP is multicast")
                self.audio_socket.setsockopt(socket.IPPROTO_IP, socket.IP_MULTICAST_TTL, 1)
                self.audio_socket.bind((self.barp_ip, self.audio_port))

            elif is_broadcast(self.barp_ip, self._broadcast):
                logger.info("Barp IP is broadcast")
                self.audio_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)

                self.audio_socket.bind((self.barp_ip, self.audio_port))

            else:
                logger.warning("{} is NOT multicast OR broadcast. Defaulting to broadcast".format(self.barp_ip))
                # TODO: Needs to be changed to reflect the actual barp_ip. Find a replacement for socket.SO_BROADCAST
                self.audio_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
                self.audio_socket.bind(("", self.audio_port))

    def _close_sockets(self):
        if self.exit_flag:
            count = 0
            if self.listen_status_socket is not None:
                self.listen_status_socket.close()
                self.listen_status_socket = None
                count += 1
            if self.status_socket is not None:
                self.status_socket.close()
                self.status_socket = None
                count += 1
            if self.control_socket is not None:
                self.control_socket.close()
                self.control_socket = None
                count += 1
            if self.call_socket is not None:
                self.call_socket.close()
                self.call_socket = None
                count += 1
            if self.audio_socket is not None:
                self.audio_socket.close()
                self.audio_socket = None
                count += 1
            logger.info(f"All {count} sockets closed")
        else:
            logger.warning("Paging master still running, cannot close the sockets yet")

    def exit(self):
        self._exit = True
        logger.info("Exiting from PagingMaster")
        self._force_state_change_on_exit(BARPProtocol.NOT_USED)
        self._close_sockets()
        logger.info("Finished")

    def _force_state_change_on_exit(self, state: BARPProtocol = BARPProtocol.NOT_USED):
        self.state = state
        logger.warning(f"State at exit: {state}, current clients: {self._current_clients_list}")
        if self._current_clients_list is not None:
            self.send_end_group_request()

    def listen_status_message(self):
        """
            To be started as a parallel thread process for
            listening to active devices.
        """

        try:
            if is_multicast(self.barp_ip):
                self.listen_status_socket.bind(("", self.status_port))
                mreq = struct.pack("4sL", self.barp_ip, socket.INADDR_ANY)
                self.listen_status_socket.setsockopt(
                    socket.IPPROTO_IP, socket.IP_ADD_MEMBERSHIP, mreq
                )
            elif is_broadcast(self.barp_ip, self._broadcast):
                # Switching to listening on broadcast port if ip is not multicast.
                logger.info(f"Broadcast: {self._broadcast}")
                logger.info(f"Status port: {self.status_port}")

                self.listen_status_socket.bind((self._broadcast, self.status_port))
            else:
                # Switching to listening on broadcast port if ip is not multicast.
                logger.warning("[listen_status_socket] This is not yet implemented")
            self.listen_status_socket.settimeout(1)
        except Exception as e:
            logger.warning(f"While trying to start listen_status_message, following occurred: {e}")
            exit()

        while not self.exit_flag:
            try:
                data, address = self.listen_status_socket.recvfrom(4096)
            except socket.timeout:
                continue
            msg = self.barp.decode(data)
            if msg[0]["message_type"] == BARPProtocol.STATUS_REPORT:
                self.active_devices[msg[0]["sender_id"]] = {
                    "state":      msg[1]["state"],
                    "time_stamp": datetime.now(),
                }

            # Removal of devices with stale status.
            current_time = datetime.now()
            for dev_id in list(self.active_devices.keys()):
                if (
                        current_time - self.active_devices[dev_id]["time_stamp"]
                ).seconds > 5:
                    del self.active_devices[dev_id]
        logging.info("Status listening stopped!")

    def send_status_message(self):
        """
        To be started as a parallel thread process.
        Prepares a state message for the paging master.
        In this POC, only state 0 (Idle) and 130 (paging message)
        for other possible states please refer to the barp.pdf doc
        """
        while not self.exit_flag:
            message_kwargs = {
                "state":              self.state,
                "priority":           0,
                "sender_id":          self.barp.my_id,
                "volume":             50,
                "config_bgm_channel": 0,
            }
            status = self.barp.encode(
                message_type=BARPProtocol.STATUS_REPORT, message_kwargs=message_kwargs, **self.header_kwargs
            )
            self.status_socket.sendto(status, (self.barp_ip, self.status_port))
            time.sleep(self.STATUS_REPORT_TIME_INTERVAL)
        logging.info("Status message sending finished")

    def send_page_group_request(self, clients):  #  priority=0 is now a class variable
        """
        Prepares and sends state message 3 to be sent to the client
        before starting the stream.
        Args:
            clients (List[int])
            priority (int)
        """
        self._kill_page_group = False
        self.state = BARPProtocol.SEND_CUSTOM_COMMAND

        if is_multicast(self._multicast_group_address):
            self._page_to_address = self.page_group_request_address
            self._page_to_port = self.page_group_request_port
        else:
            self._page_to_address = self.barp_ip
            self._page_to_port = self.audio_port

        self._current_clients_list = clients.copy()

        logger.info("BARP IP: {}, clients: {}, page_group_request_port: {}, page_group_request_address: {}"
                    .format(self.barp_ip, self._current_clients_list, self.page_group_request_port,
                            self.page_group_request_address))

        # logger.info("Waiting for file to be downloaded")
        # while self.playfile_state == AudioControllerState.DOWNLOADING_RECORDED_MESSAGE:
        #     time.sleep(0.01)
        # logger.info("File downloaded!")

        while not self._kill_page_group:
            counter = 0
            message_kwargs = {
                "port":                   self.page_group_request_port,
                "priority":               self.paging_priority,
                "audio_encdoing_type":    self.encoding_type,
                "multicast_stream_group": self.page_group_request_address,
                "receiver_id":            self._current_clients_list,
            }
            request_msg = self.barp.encode(
                message_type=BARPProtocol.PAGE_GROUP_REQUEST.value, message_kwargs=message_kwargs, **self.header_kwargs
            )

            logger.info("[control_socket] Sending to ip: {}, port: {}, priority: {}"
                        .format(self.barp_ip, self.control_port, self.paging_priority))

            self.control_socket.sendto(request_msg, (self.barp_ip, self.control_port))
            logger.debug(f"sending data to client {self._current_clients_list}")

            while counter < self.PAGING_REQUEST_TIME_INTERVAL and not self._kill_page_group:
                time.sleep(0.01)
                counter += 0.01

        self._current_clients_list = None
        self.state = BARPProtocol.NOT_USED

    def send_end_group_request(self):
        """
        Prepares and sends the end group page request, MSG 8.
        The stream should stop after this message is sent.
        Args:
            clients (List[int])
        """

        clients = self._current_clients_list
        logger.debug(f"send_end_group_request {clients}")

        if clients:
            message_kwargs = {"port": self.audio_port, "receiver_id": clients}
            end_page_request = self.barp.encode(
                message_type=BARPProtocol.END_GROUP_REQUEST.value, message_kwargs=message_kwargs, **self.header_kwargs
            )
            logger.debug(f"stopping sending data to client {clients}")
            self.control_socket.sendto(end_page_request, (self.barp_ip, self.control_port))
        else:
            logger.debug("group request already empty!")

        self._kill_page_group = True
