import logging
import hashlib

from .uci import getValueOfUci, setUciConfigs
from .constants import DEFAULT_WEBUI_LOGIN_USERNAME, DEFAULT_WEBUI_LOGIN_REALM, WEBUI_PASSWD_FILE_PATH
from .auth import generateNewSessionKey
from .exceptions import InvalidRequestBodyContentError, InvalidOldPasswordError

log = logging.getLogger('flask-backend')

'''
Sets a new password for the webUI.
@param pwdSettings: dictionary containing the old password (pwd_old) and the new one (pwd_set). The password values are in plain text(!).
The old password may not be provided if the login with password is disabled for the webUI.
If required but not provided, or provided but not required, it will raise an InvalidRequestBodyContentError exception.
If required and provided but invalid, it will raise an InvalidOldPasswordError exception.
For the new password, if not provided or empty, the webUI's login with password will be disabled. Otherwise, the new password will be set. 
'''
def setWebUiPassword(pwdSettings):
    webUiPwdRequired = getValueOfUci('httpd', 'webserver', 'password_set')
    if 'pwd_old' in pwdSettings.keys():
        if webUiPwdRequired == 'false':
            log.error("pwd_old is in pwdSettings but password_set is disabled")
            raise InvalidRequestBodyContentError()
        else:
            try:
                isValid = currPasswordIsValid(pwdSettings['pwd_old'])
            except Exception as e:
                raise e
            else:
                if not isValid:
                    log.error("Old password is wrong")
                    raise InvalidOldPasswordError()
    else:
        if webUiPwdRequired == 'true':
            log.error("pwd_old not in pwdSettings but password_set is enabled")
            raise InvalidRequestBodyContentError()
    # at this stage, either pwd_old is valid or not required
    if 'pwd_set' not in pwdSettings.keys() or pwdSettings['pwd_set'] == '':
        # disable webui password
        try:
            enableWebUiPassword('false')
        except Exception as e:
            raise e
    else:
        # change or enable webui password
        try:
            storeWebuiPassword(pwdSettings['pwd_set'])
            enableWebUiPassword('true')
        except Exception as e:
            raise e

'''
Checks if the password provided matches the current one.
@param passwd: password to test
@param passwdFile: filepath of the file where current password is stored. If not provided, default path is WEBUI_PASSWD_FILE_PATH
@param username: string containing the username of the corresponding password. If not provided, default path is DEFAULT_WEBUI_LOGIN_USERNAME
@param realm: string containing the realm used to calculate the digest. If not provided, default path is DEFAULT_WEBUI_LOGIN_REALM
Calculates the md5 digest of the given username, realm and password. If it matched the one stored in the file, then the password provided is correct: returns True. Otherwise, returns False.
'''
def currPasswordIsValid(passwd, passwdFile=WEBUI_PASSWD_FILE_PATH, username=DEFAULT_WEBUI_LOGIN_USERNAME, realm=DEFAULT_WEBUI_LOGIN_REALM):
    strToDigest = username+":"+realm+":"+passwd
    try:
        digest = hashlib.md5(strToDigest.encode('utf-8')).hexdigest()
    except Exception as e:
        log.error("Can not calculate digest for the provided password: {}".format(e))
        raise
    try:
        f = open(passwdFile, 'r')
        content = f.read()
        f.close()
    except Exception as e:
        log.error("Error while reading passwd file contents: {}".format(e))
        raise e
    try:
        md5sum = ''
        fileLines = content.split('\n')
        for line in fileLines:
            tmpStr = username+':'+realm+':'
            if line.startswith(tmpStr):
                md5sum = line.split(':')[2]
                break
        if md5sum != '':
            return digest == md5sum
        else:
            return False
    except Exception as e:
        raise e

'''
Store a webUI Password.
@param passwd: password to store
@param passwdFile: filepath of the file where to store the password. If not provided, default path is WEBUI_PASSWD_FILE_PATH
@param username: string containing the username of the corresponding password. If not provided, default path is DEFAULT_WEBUI_LOGIN_USERNAME
@param realm: string containing the realm used to calculate the digest. If not provided, default path is DEFAULT_WEBUI_LOGIN_REALM
Calculates the md5 digest of the given username, realm and password: md5digest(username:realm:passwd)
Then, writes it in filepath with the structure username:realm:md5digest.
Finally, since the password was changed, current logins must expire and users should login again with the new password. So, generate a new session key.
'''
def storeWebuiPassword(passwd, passwdFile=WEBUI_PASSWD_FILE_PATH, username=DEFAULT_WEBUI_LOGIN_USERNAME, realm=DEFAULT_WEBUI_LOGIN_REALM):
    strToDigest = username + ":" + realm + ":" + passwd
    try:
        digest = hashlib.md5(strToDigest.encode('utf-8')).hexdigest()
    except Exception as e:
        log.error("Can not calculate digest for the provided password : {}".format(e))
        raise e
    try:
        f = open(passwdFile, 'w')
        strToWrite = username+":"+realm+":"+digest
        f.write(strToWrite)
        f.close()
        # generate new session Key
        generateNewSessionKey()
    except Exception as e:
        log.error("Error while writting new passwd in file: {}".format(e))
        raise e

'''
Enable or disabled WebUI login with password.
@param flag_value: string that indicates if login with password should be enabled or disabled. 
If "true", enables the login with password. If "false", disables it.
Sets the value in the corresponding uci and commits. No services to restart services. 
'''
def enableWebUiPassword(flag_value):
    try:
        # set webUI password flag to true
        setUciConfigs({'httpd.webserver.password_set': flag_value}, restartServices=False)
    except Exception as e:
        raise e