import alsaaudio
import logging
import shlex
import subprocess

from .system_info import getHWTypeID
from .uci import setUciConfigs, getValueOfUci
from .exceptions import SettingDateAndTimeError

log = logging.getLogger('flask-backend')

'''
Set device's system Date and Time.
@param stringDate: string containing the date and time to set. Format must be: "YYYY-MM-DD HH:mm:ss"
Note: When device reboots, date and time previously set may be lost.
'''
def setDateAndTime(stringDate):
    try:
        cmd = "/bin/date -s \"" + stringDate + "\""
        log.info("Executing: {}".format(cmd))
        proc = subprocess.Popen(shlex.split(cmd), shell=False)
        proc.wait()
        if proc.returncode != 0:
            raise SettingDateAndTimeError
    except SettingDateAndTimeError:
        raise SettingDateAndTimeError
    except Exception as e:
        log.error(e)

'''
Set volume of the provided alsamixer
@param mixer: string containing the mixer to set
@param volume: string or int containing the new volume to set.
'''
def setMixerVolume(mixer, volume):
    try:
        mainVolumeCtl = alsaaudio.Mixer(mixer)
        mainVolumeCtl.setvolume(int(volume))
    except Exception as e:
        log.error("Error setting volume {} for {} : {}".format(volume, mixer, e))
        raise e

"""
Set the volume of a device configuration (main volume)
@param uci_vol: string containing the uci associated with the volume to set. Format should be config.section.option
@param volume: string or integer containing the new volume
@param uci_muted : optional mute uci (only used to mute main volume)
@param set_vol_func: optional method to set vol (to use without using mixer)
It will retrieve the proper mixer from the uci provided and set the volume in alsamixer.
If successful, it stores the value in the uci as well.
"""
def setVolume(uci_vol, volume, uci_muted=None, set_vol_func=None):
    try:
        muted = 'false'
        if uci_muted is not None:
            uciMuted = uci_muted.split('.')
            if (len(uciMuted) != 3):
                log.error(f"Invalid uci_muted: {uci_muted}")
                raise TypeError(f"Invalid uci_muted: {uci_muted}")
            muted = getValueOfUci(uciMuted[0], uciMuted[1], uciMuted[2])

        if set_vol_func is None:
		    # get Mixer
            mixer = getMixerChannelFromUci(uci_vol)
            log.debug(mixer)
        
        if muted == 'false':
            # set Main Volume
            if set_vol_func is None:
                setMixerVolume(mixer, volume)
                if getHWTypeID() == "90":  # AHE board
                    setMixerVolume("INTERCOM_MAIN", volume)
            else:
                set_vol_func(volume)
        else:
            log.warning("Main volume is muted")

		# store value in uci
        setUciConfigs({uci_vol : volume}, commit=True, restartServices=False,selectServicesToRestartCallback=None)
    except Exception as e:
        log.error("Error setting volume {} for {} : {}".format(volume, uci_vol, e))
        raise e

"""
Mute/unmute main volume
@param uci_muted : mute uci
@param enable: bool (enable/disable mute)
@param uci_vol: uci associated with the volume to set
@param mute_vol_func: option method to mute vol (to use without using mixer)
"""
def mute(uci_muted, enable, uci_vol, mute_vol_func=None):
    try:
        uciValue = 'false'
        if mute_vol_func is None:
            mixer = getMixerChannelFromUci(uci_vol)
        
        if enable:
            if mute_vol_func is None:
                setMixerVolume(mixer, 0)
                if getHWTypeID() == "90":  # AHE board
                    setMixerVolume("INTERCOM_MAIN", 0)
            else:
                mute_vol_func(0)
            uciValue = 'true'
        else:
            uciVol = uci_vol.split('.')
            if (len(uciVol) != 3):
                log.error(f"Invalid uci_vol: {uci_vol}")
                raise TypeError(f"Invalid uci_vol: {uci_vol}")

            vol = getValueOfUci(uciVol[0], uciVol[1], uciVol[2])
            if mute_vol_func is None:
                setMixerVolume(mixer, vol)
                if getHWTypeID() == "90":  # AHE board
                    setMixerVolume("INTERCOM_MAIN", vol)
            else:
                mute_vol_func(vol)
		
        setUciConfigs({uci_muted : uciValue}, commit=True, restartServices=False, selectServicesToRestartCallback=None)
    except Exception as e:
        log.error(f"Error trying to mute main volume : {e}")
        raise e


'''
Get the corresponding mixer for the provided uci.
@param uci: string containing the uci. Format should be config.section.option.
If uci is related to the device's main volume, the mixer is retrieved from /etc/barix/soundcard.conf file.
If uci is related to a flexa_ip_former source, the mixer will be the concatenation of "chan_" and the id of the source (1, 2, 3...).
'''
def getMixerChannelFromUci(uci):
    try:
        mixer = "Line Out"
        if uci == "epic.main.volume" or uci == "application.audio.volume":
            try:
                f = open("/etc/barix/soundcard.conf", 'r')
                lines = f.read().split('\n')
                for line in lines:
                    if line.startswith('master.control'):
                        mixer = ((line.split('='))[1]).strip('\n')
                f.close()
            except Exception as e:
                log.error("An error occurred while getting mixer: {}. Going to use mixer {}.".format(e, mixer))
        elif uci.startswith("flexa_ip_former.source"):
            sourceID = uci.split('.')[1].replace('source', '')
            mixer = "chan_" + sourceID
        else:
            log.error("Unknown volume uci. Going to use default mixer: {}".format(mixer))
        return mixer
    except Exception as e:
        log.error("Error getting mixer: {}.".format(e))
        raise e