import logging
import os
import subprocess
import shlex
import time
import threading
import uuid

from pathlib import Path
from werkzeug.utils import secure_filename

from .constants import *
from .exceptions import *


log = logging.getLogger('flask-backend')

updateStatus = "clear"      # clear | uploading | installing | rebooting | error
errorStatus = None

p = Path(FW_UPDATES_DIR)
if not p.exists():
    try:
        p.mkdir()
    except Exception as e:
        log.error("Error creating directory {}: {}".format(FW_UPDATES_DIR, e))


'''
Starts a firmware update.
@param fileUploaded: firmware update file uploaded
Raises an InvalidUploadedFileError if Form Data didn't upload the firmware update file using the "fwImage" key, or if filename is empty.
If file uploaded is too large, it will raise a FileTooLargeError Exception.
Checks if there's a FW update already running. If not, a uuid is generated and stored in a file to indicate that an FW update process has started (FW_UPDATE_FILE_PATH). This file is deleted when the process ends. Next time someone starts a FW update, if the file exists, it means that there's an update already running.
If there's a FW update already running, a FwUpdateRunningError Exception is raised.
Then file is stored in the flask configured UPLOAD_FILES_DIR and installed using the device's installer framework.
When an Exception occurs and the process is interrupted (except for the FwUpdateRunningError), the flag that indicates that a firmware update is running is resetted (FW_UPDATE_FILE_PATH is deleted).
'''
def updateFirmware(fileUploaded):
    global updateStatus
    log.info("Uploading new fw image...")
    try:
        if 'fwImage' not in fileUploaded:
            log.error('fwImage is not in request files uploaded')
            # remove file if error occurs
            removeUpdateFileChecker()
            raise InvalidUploadedFileError('Invalid file')
        uploadedFile = fileUploaded['fwImage']
        # if user does not select file, browser also
        # submit an empty part without filename
        if uploadedFile.filename == '':
            log.error('filename from uploaded file is invalid')
            # remove file if error occurs
            removeUpdateFileChecker()
            raise InvalidUploadedFileError('Invalid filename')
        # TODO: validate file extension
        # generate uuid and store it in file
        # if uuid exists, stop update request because there's already an update running
        # file is automatically removed after reboot
        if not os.path.isfile(FW_UPDATE_FILE_PATH):
            uid = uuid.uuid4().hex
            try:
                f = open(FW_UPDATE_FILE_PATH, 'w')
                f.write(uid)
                f.close()
            except Exception as e:
                log.error("Error while writing in file {}: {}".format(FW_UPDATE_FILE_PATH, e))
                raise e
        else:
            log.error("ABORTING UPDATE! There is a firmware update already running.")
            raise FwUpdateRunningError()
        filename = secure_filename(uploadedFile.filename)
        try:
            log.debug("Storing fw update file in {}".format(FW_UPDATES_DIR))
            uploadedFile.save(os.path.join(FW_UPDATES_DIR, filename))
        except Exception as e:
            log.error(e)
            updateStatus = "error"
            # remove file if error occurs
            removeUpdateFileChecker()
            raise e
        t = threading.Thread(target=installNewFW,
                             args=(os.path.join(FW_UPDATES_DIR, secure_filename(uploadedFile.filename)),))
        t.start()
    except FwUpdateRunningError:
        raise FwUpdateRunningError()
    except InvalidUploadedFileError:
        raise InvalidUploadedFileError()
    except Exception as e:
        log.error("Error occurred while uploading fw file: {}".format(e))
        updateStatus = "error"
        # remove file if error occurs
        removeUpdateFileChecker()
        if str(e).startswith('413 Request Entity Too Large'):
            raise FileTooLargeError('REQUEST ENTITY TOO LARGE')
        else:
            raise e


'''
Install the fw update image.
@param filePath: string containing the fullpath of the firmware update image
If an exception occurs and the process is interrupted, the flag that indicates that an update is running is resetted (FW_UPDATE_FILE_PATH is deleted) and the fw update image is removed.
If the firmware update finishes successfully, the fw update image is removed and the device reboots.
'''
def installNewFW(filePath):
    global updateStatus, errorStatus
    try:
        updateStatus = "installing"
        cmd = "qiba-update-client -f "+filePath
        log.info("Installing new firmware...")
        proc = None
        try:
            proc = subprocess.Popen(shlex.split(cmd), shell=False)
            proc.wait()
        except Exception as e:
            log.error(e)
            updateStatus = "error"
            # remove file if error occurs
            removeUpdateFileChecker()
        #log.debug("qiba-update-client returned {}".format(proc.returncode))
        if proc.returncode != 0:
            log.error("qiba-update-client failed installation")
            updateStatus = "error"
            errorStatus = proc.returncode
            removeUpdateFileChecker()
            # remove file
            try:
                os.remove(filePath)
            except Exception as e:
                log.error("Can't remove file {}: {}".format(filePath, e))
                # remove file if error occurs
                removeUpdateFileChecker()
        else:
            # remove file after installing it
            try:
                os.remove(filePath)
            except Exception as e:
                log.error("Can't remove file {}: {}".format(filePath, e))
                updateStatus = "error"
                # remove file if error occurs
                removeUpdateFileChecker()
            updateStatus = "rebooting"
            time.sleep(3)
            cmd = "/sbin/reboot"
            log.info("Executing: {}".format(cmd))
            subprocess.Popen(shlex.split(cmd), shell=False)
            # no need to removeUpdateFileChecker because device will reboot and /tmp will be cleaned
    except Exception as e:
        log.error("An error occurred: {}".format(e))


'''
Resets the flag that indicates that there's a FW update already running. Which means remove the FW_UPDATE_FILE_PATH file.
If an error occurs, raise the Exception.
'''
def removeUpdateFileChecker():
    try:
        if os.path.isfile(FW_UPDATE_FILE_PATH):
            os.remove(FW_UPDATE_FILE_PATH)
    except Exception as e:
        log.error("Can't remove file {}: {}".format(FW_UPDATE_FILE_PATH, e))
        raise e


'''
Gets a string containing the status of the FW update. 
It can be:
    - "clear" when nothing is happening or process has finished already;
    - "uploading" when the user is uploading a new FW update file image;
    - "installing" when the current state of the process is installing the FW update image;
    - "rebooting" when the firmware update is finished successfully and the device will reboot;
    - "error" if an error has occurred during the FW update process;
'''
def getFwUpdateStatus():
    # log.debug("Firmware Update status: {}".format(updateStatus))
    return updateStatus


'''
Starts the FW update process. Current status if updated to "uploading".
'''
def startFwUpdate():
    global updateStatus
    log.info("Starting Firmware Update")
    updateStatus = "uploading"
    # TODO: stop audio if it is playing


'''
Abort the FW update process. 
Current status if updated to "error".
Resets the flag that indicates that there's a FW update already running.
'''
def stopFwImageUpload():
    global updateStatus
    try:
        log.info("Aborting Firmware Update")
        updateStatus = "error"
        removeUpdateFileChecker()
    except Exception as e:
        log.error(e)
        raise e


'''
Gets a string containing the reason why the firmware update failed (if it failed). 
It can be:
    - 1	    Unable to get system information.
    - 5	    Failed extrating image.
    - 6	    Failed parsing image header file.
    - 8	    Version already installed.
    - 9	    Image checksum mismatch.
    - 10	Update does not match system machine.
    - 11	Update does not match system image.
    - 12	Signature validation failed.
    - 13	Update does not match system image.
    - 14	A newer version is in use.
    - 21	Installation failed.
'''
def getErrorStatusMessage():
    if errorStatus == 1:
        return "Unable to get system information."
    elif errorStatus == 5:
        return "Failed extrating image."
    elif errorStatus == 6:
        return "Failed parsing image header file."
    elif errorStatus == 8:
        return "Version already installed."
    elif errorStatus == 9:
        return "Image checksum mismatch."
    elif errorStatus == 10:
        return "Update does not match system machine."
    elif errorStatus == 11:
        return "Update does not match system image."
    elif errorStatus == 12:
        return "Signature validation failed."
    elif errorStatus == 13:
        return "Update does not match system image."
    elif errorStatus == 14:
        return "A newer version is in use."
    elif errorStatus == 21:
        return "Installation failed."
    else:
        return errorStatus #None