import logging
import time
import re
from threading import *
from typing import *

import baco

from . AudioSource import *


class RtpAudioSource(AudioSource):
    def __init__(self, sourceId: int, name: str, volume:int, parameters:Dict[str,Any], audioDev:str,
            sampleRate:int=44100):
        AudioSource.__init__(self, sourceId, name, volume, parameters, audioDev)

        self.audioDevice = AudioSource.getAudioDev(self)
        self.url = AudioSource.getParameter(self, "url")
        match = re.search("^(rtp:\/\/@:)(\d+)$", self.url)
        if match:
            self.url="rtp://0.0.0.0:{}".format(match.group(2))
        self.lock               = Lock()

        self.monitoringThread   = None
        self.isMonitoring       = False

        self.logger             = logging.getLogger('rtp')

        self.state              = SourceState.STOPPED
        self.alive              = None
        self.counter            = 0
        self.inputBuffer        = 0
        self.outputBuffer       = AudioSource.getParameter(self,"output_buffer_size")
        if self.outputBuffer is None:
            self.outputBuffer   = 1000
            
        self.inputBuffer        =  AudioSource.getParameter(self,"input_buffer_size")
        if self.inputBuffer is None:
            self.inputBuffer    = 0

        self.playbackMode = AudioSource.getParameter(self, "playback_mode")
        if self.playbackMode == 'full_buffer':
            self.logger.info("RTP Full Buffer")
            self.player = baco.SimplePlayer(self.url, self.audioDevice, sampleRate, baco.FULLBUFFER)
            self.player.setInputFifoDuration(self.inputBuffer)
            self.player.setOutputFifoDuration(self.outputBuffer)
            self.player.setReadyThreshold(int(self.outputBuffer/2))
        elif self.playbackMode == 'const_delay':
            self.logger.info("RTP Constant Delay")
            self.player = baco.SimplePlayer(self.url, self.audioDevice, sampleRate, baco.CONSTDELAY)
            self.player.setInputFifoDuration(0)
            self.player.setOutputFifoDuration(self.outputBuffer+500)
            self.player.setReadyThreshold(self.outputBuffer)
            self.player.setMaxDelta(30)
        elif self.playbackMode == 'low_latency':
            self.logger.info("RTP Low Latency Mode")
            self.player = baco.SimplePlayer(self.url, self.audioDevice, sampleRate, baco.GENERIC)
            self.player.setInputFifoDuration(0)
            self.player.setOutputFifoDuration(0)
        else:
            self.logger.error("RTP Playback Mode ERROR!")

        self.player.setInputTimeout(1000)
        self.player.setCallback(self.checkStateCallback)

    def playAudio(self):
        self.lock.acquire()
        try:
            if self.state == SourceState.STOPPED:
                self.player.connectSource()

            if self.state != SourceState.PLAYING:
                self.logger.info("RTP source id={}, state: PLAY, \"{}\"".format(self.id, self.name))
                self.player.connectAudio()
                self.counter += 1

            self.state = SourceState.PLAYING
        finally:
            self.lock.release()

    def stopAudio(self):
        self.lock.acquire()
        self.logger.info("RTP source id={}, state: STOP, \"{}\"".format(self.id, self.name))
        try:
            if self.playbackMode == 'regular':
                self.player.disconnectAudio()
            else:
                self.player.disconnect()
            self.state = SourceState.STOPPED
        finally:
            self.lock.release()

    def checkAudio(self):
        self.lock.acquire()
        self.logger.info("RTP source id={}, state: MONITOR, \"{}\"".format(self.id, self.name))
        try:
            if self.state != SourceState.STOPPED:
                self.player.disconnectAudio()

            self.state = SourceState.MONITORING
            self.startMonitoring()

        finally:
            self.lock.release()

    def startMonitoring(self):
        if not self.isMonitoring:
            self.monitoringThread = Thread(target=self.reconnectSourceLoop)
            self.monitoringThread.start()
            self.isMonitoring = True

    def reconnectSourceLoop(self):
        while self.state == SourceState.MONITORING or self.state == SourceState.PLAYING:

            isConnected = self.player.isConnected()
            if not isConnected:
                res = self.player.connectSource()
                if res == 0:
                    self.logger.info("RTP source id={}, connection succeed!".format(self.id))
                    if self.state == SourceState.PLAYING:
                        self.player.connectAudio()
                #else:
                #    self.logger.info("RTP source id={}, connection failed.".format(self.id))

            time.sleep(1)
        self.isMonitoring = False

    def reportStatus(self):
        sourceDict = {}
        sourceDict["curState"] = "playing" if self.state == SourceState.PLAYING else (
            "stopped" if self.state == SourceState.STOPPED else "monitoring")
        sourceDict["state"] = "alive" if self.alive else "dead"
        sourceDict["counter"] = self.counter
        return sourceDict

    def getState(self):
        return self.state, self.alive

    def checkStateCallback(self, alive):
        self.alive = alive
        if alive:
            print("IT IS ALIVE!")
        else:
            print("STREAM IS DEAD")
        AudioSource.sendSourceStateUpdate(self, self.state, alive)

    def disconnect(self):
        print("Disconnect from RTP called")
        self.lock.acquire()
        try:
            self.state = SourceState.STOPPED
            self.isMonitoring = False
            self.monitoringThread.join()
            if self.player.isConnected():
                self.player.disconnect()
            self.player = None
        finally:
            self.lock.release()
