import logging
from struct import pack, unpack

from BARP.constants import FORMATS


def unpack_fields(format, field):
    if "s" in format:
        try:
            return b"".join(unpack(format, field))
        except Exception as e:
            logging.error("Couldn't unpack the field, {}".format(e))
            return "UNPACK_FAILURE:{}".format(field)
    else:
        try:
            return unpack(format, field)[0]
        except Exception as e:
            logging.error("Couldn't unpack the field, {}".format(e))
            return "UNPACK_FAILURE:{}".format(field)


def get_zone_from_bitmap(bitmap):
    """Gets the active zone and the address of peer in the zone the message is directed
    to.

    Args:
        bitmap (str): 127 elem long string

    Returns:
        List[int]: Peer the bitmap addresses.
    """
    bitmap_flag = unpack(FORMATS.BYTE_FMT * 128, bitmap)
    peers = []
    for zone, peer_flag in enumerate(bitmap_flag):
        if peer_flag != 0:
            binary_peer_flag = "{:08b}".format(peer_flag)
            for i in range(len(binary_peer_flag)):
                if binary_peer_flag[i] == "1":
                    peers.append((i + 1) + zone * 8)
    return peers


def check_receiver_id_from_bitmap(bitmap, receiver_id):
    """
    Checks whether the bitmap addresses the given peer id or not.
    Args:
        bitmap (str): 127 elem long string
        receiver_id (int): Realistically it is between 1 and 100.

    Returns:
        bool: True if the bitmap addrresses the receiver_id else 0
    """
    zone = (receiver_id - 1) // 8
    peer_flag_index = (receiver_id - 1) % 8

    peer_flag = unpack(FORMATS.BYTE_FMT, bitmap[zone : zone + 1])[0]
    peer_flag = "{:08b}".format(peer_flag)
    return peer_flag[peer_flag_index] == "1"


def id_to_bitmap(receiver_id):
    """Converts id to 128 byte array bitnmap.

    Args:
        id (int): Realistically between 1 and 100.
    Returns:
        128 byte array
    """
    if type(receiver_id)==int:
        zone = (receiver_id - 1) // 8
        peer_flag_idx = (receiver_id - 1) % 8 + 1
        # converting integer binary flag to 1 byte binary flag
        binary_peer_flag = "0" * (peer_flag_idx - 1) + "1" + "0" * (8 - peer_flag_idx)
        inactive_zone = pack(FORMATS.BYTE_FMT, 0)
        active_zone = pack(FORMATS.BYTE_FMT, int(binary_peer_flag, base=2))
        bitmap = inactive_zone * zone + active_zone + inactive_zone * (128 - zone - 1)
        return bitmap
    else:
        # list of receiver ids.
        bitmap_bit=["0"]*1028
        for idx in receiver_id:
            bitmap_bit[idx-1]="1"
        bitmap_byte = []
        for i in range(128):
            bitmap_byte.append(
                pack(FORMATS.BYTE_FMT,
                    max(int(
                        "".join(bitmap_bit[i*8:(i+1)*8]),2
                        ), 0)
                    )
                )
        return b"".join(bitmap_byte)



def format_multicast_stream_group(multicast_stream_group):
    processed_multicast_stream_group = []
    for idx in range(len(multicast_stream_group)):
        processed_multicast_stream_group += [str(
            unpack(FORMATS.BYTE_FMT, multicast_stream_group[idx : idx + 1])[0]
        )]
    return ".".join(processed_multicast_stream_group)
