from BARP.constants import FORMATS
from BARP.utils import (
    check_receiver_id_from_bitmap,
    format_multicast_stream_group,
    get_zone_from_bitmap,
    id_to_bitmap,
    pack,
)
from BARP.utils import unpack_fields as unpack


# TODO: Remove usage of function get_zone_from_bitmap in
# favour of check_receiver_id_from_bitmap.
class MSG_BASE(object):
    """
    Parent class for all barp messages
    """

    def __init__(self, my_id):
        self.my_id = my_id

    def encode(self, *args, **kwargs):
        return None

    def decode(self, *args, **kwargsI):
        return {}


class MSG_0(MSG_BASE):
    """
    class for BARP Control message 0 - Not assined
    """


class MSG_1(MSG_BASE):
    """
    Class for BARP contol message 1.
    Info:
    This message is sent by the master in a call-session and tells the client to stop
    sending audio to the master. This message can be sent only after a
    page group request. If the client is not in a call-session,
    the message is ignored.
    """


class MSG_2(MSG_BASE):
    """
    Class for BARP contol message 2.

    Info:
    This message is sent by the master in a call-session and tells the client to
    start sending audio to the master. Audio should be sent to the master's IP port
    defined in the message.
    The client MUST check the source ID of the message with the ID of the current stream
    played to avoid conflicts if higher priority message arrives. If the IDs do not
    match the message is ignored. This message can be sent only after a
    page group request. If the client is not in a call-session,
    the message is ignored. If the client is already sending audio this message can be
    used to change the audio port.

    The master resends the start streamback message every second.
    """

    def __init__(self, my_id):
        super(MSG_2, self).__init__(my_id)
        self.port_idx = slice(2)

    def decode(self, message):
        """
        Returns the audio port for stream playback.
        """
        return {"port": unpack(FORMATS.LITTLE_ENDIAN_16b_FMT, message[self.port_idx])}

    def encode(self, audio_port):
        """
        Encodes the audio port for stream playback to be sent to client.
        Args:
            audio_port (int): Port less tha 65535
        Returns:
            byte array of size 2 bytes.
        """
        return pack(FORMATS.LITTLE_ENDIAN_16b_FMT, audio_port)


class MSG_3(MSG_BASE):
    """
    Class for BARP contol message 3 - play audio.
    Info:
    This message is a request to play audio. This request must be sent before the master
    starts sending an audio stream.
    """

    def __init__(self, my_id):
        super(MSG_3, self).__init__(my_id)
        self.port_idx = slice(2)
        self.priorty_idx = slice(2, 3)
        self.audio_encdoing_type_idx = slice(3, 4)
        self.multicast_stream_group_idx = slice(4, 8)
        self.bitmap_idx = slice(8, 128 + 8)

    def decode(self, message):
        """
        Returns the params for audio playback
        Args:
            message (str): Message part of the BARP.
        Returns:
            dict : port, priority, audio_encdoing_type, multicast_stream_group, bitmap
                and message_to_me
        """
        port = unpack(FORMATS.LITTLE_ENDIAN_16b_FMT, message[self.port_idx])
        priority = unpack(FORMATS.BYTE_FMT, message[self.priorty_idx])
        audio_encdoing_type = unpack(
            FORMATS.BYTE_FMT, message[self.audio_encdoing_type_idx]
        )
        multicast_stream_group = unpack(
            FORMATS.STRING_FMT * 4, message[self.multicast_stream_group_idx]
        )
        bitmap = get_zone_from_bitmap(message[self.bitmap_idx])
        message_to_me = check_receiver_id_from_bitmap(
            message[self.bitmap_idx], self.my_id
        )
        return {
            "port": port,
            "priority": priority,
            "audio_encdoing_type": audio_encdoing_type,
            "multicast_stream_group": format_multicast_stream_group(
                multicast_stream_group
            ),
            "bitmap": bitmap,
            "message_to_me": message_to_me,
        }

    def encode(
        self, port, priority, audio_encdoing_type, multicast_stream_group, receiver_id
    ):
        """
        Encodes the audio port for stream playback to be sent to client.
        Args:
            port (int): port in range 1, 65535.
            priority (int) : range(1,255)
            audio_encdoing_type (int): example 97.
            multicast_stream_group (str): "213.145.154.168" , "0.0.0.0" etc.
            receiver_id (int): Realistically in range of 1 to  100.
        Returns:
            139 byte array message.

        """
        multicast_stream_group_parsed = [
            int(adr) for adr in multicast_stream_group.split(".")
        ]
        return (
            pack(FORMATS.LITTLE_ENDIAN_16b_FMT, port)
            + pack(FORMATS.BYTE_FMT, priority)
            + pack(FORMATS.BYTE_FMT, audio_encdoing_type)
            + pack(FORMATS.BYTE_FMT * 4, *multicast_stream_group_parsed)
            + id_to_bitmap(receiver_id)
        )


class MSG_4(MSG_BASE):
    """
    Class for BARP contol message 4 -Activate Relay.
    Info:
    This message is sent by the master to activate a specific relay on the client(s).
    The relay can be either forced on (activation time is 255), forced off
    (activation time is 0) or activated for N*100ms (activation time 1..254).
    """

    def __init__(self, my_id):
        """
        Initailaizes the positional index of various elements of BARP activate relay
        message.
        """
        super(MSG_4, self).__init__(my_id)
        self.relay_number_idx = slice(0, 1)
        self.activation_time_idx = slice(1, 2)
        self.reserved_idx = slice(2, 8)
        self.bitmap_idx = slice(8, 128 + 8)

    def decode(self, message):
        """
        Decodes activate relay message content
        Args:
            message (str): message without BARP header
        Returns
            dict: relay_number, activation_time,reserved, bitmap
        """
        relay_number = unpack(FORMATS.BYTE_FMT, message[self.relay_number_idx])
        activation_time = unpack(FORMATS.BYTE_FMT, message[self.activation_time_idx])
        reserved = unpack(FORMATS.BYTE_FMT * 6, message[self.reserved_idx])
        bitmap = get_zone_from_bitmap(message[self.bitmap_idx])
        message_to_me = check_receiver_id_from_bitmap(
            message[self.bitmap_idx], self.my_id
        )
        return {
            "relay_number": relay_number,
            "activation_time": activation_time,
            "reserved": reserved,
            "bitmap": bitmap,
            "message_to_me": message_to_me,
        }

    def encode(self, relay_number, activation_time, receiver_id):
        """Encodes message as per BARP specification.
        Args:
            relay_number (int): range(1,255)
            activation_time (int) : range(1,255)
            receiver_id (int): Realistically in range of 1 to  100.
        Returns:
            136 byte array message.

        """
        return (
            pack(FORMATS.BYTE_FMT, relay_number)
            + pack(FORMATS.BYTE_FMT, activation_time)
            + pack(FORMATS.BYTE_FMT * 6, *[0] * 6)
            + id_to_bitmap(receiver_id)
        )


class MSG_5(MSG_BASE):
    """
    Class for BARP contol message 5 - set audio volume.
    Info:
    The master sends this message to set audio volume on the client(s). The
    volume is given in 5% steps. If the permanent byte is set to a non-zero
    value, the volume settings should be stored in a non-volatile memory of the
    client.
    """

    def __init__(self, my_id):
        """
        Initailaizes the positional index of various elements of BARP activate relay
        message.
        """
        super(MSG_5, self).__init__(my_id)
        self.volume_idx = slice(0, 1)
        self.change_type_idx = slice(1, 2)
        self.bitmap_idx = slice(8, 128 + 8)

    def decode(self, message):
        """
        Decodes activate relay message content
        Args:
            message (str): message without BARP header
        Returns
            dict: volume, chnage_type, bitmap
        """
        volume = unpack(FORMATS.BYTE_FMT, message[self.volume_idx])
        change_type = unpack(FORMATS.BYTE_FMT, message[self.change_type_idx])
        bitmap = get_zone_from_bitmap(message[self.bitmap_idx])
        message_to_me = check_receiver_id_from_bitmap(
            message[self.bitmap_idx], self.my_id
        )
        return {
            "volume": volume,
            "change_type": change_type,
            "bitmap": bitmap,
            "message_to_me": message_to_me,
        }

    def encode(self, volume, change_type, receiver_id):
        """Encodes message as per BARP specification.
        Args:
            volume (int): range(1,255)
            change_type (int) : range(1,255)
            receiver_id (int): Realistically in range of 1 to  100.
        Returns:
            136 byte array message.
        """
        return (
            pack(FORMATS.BYTE_FMT, volume)
            + pack(FORMATS.BYTE_FMT, change_type)
            + pack(FORMATS.BYTE_FMT * 6, *[0] * 6)
            + id_to_bitmap(receiver_id)
        )


class MSG_6(MSG_BASE):
    """
    Class for BARP contol message 6 - set background channel.
    Info:
    This message tells the client(s) to start playing a specific background
    music channel. If the channel number is 0 the client should stop playing
    background music. If the permanent byte is set to a non-zero value, the
    background channel number should be stored in a non-volatile memory of the
    client.
    """

    def __init__(self, my_id):
        super(MSG_6, self).__init__(my_id)
        self.channel_number_idx = slice(0, 1)
        self.change_type_idx = slice(1, 2)
        self.bitmap_idx = slice(8, 128 + 8)

    def decode(self, message):
        """
        Decodes set background channel content.
        Args:
            message (str): message without BARP header
        Returns
            dict: channel_number, chnage_type, bitmap
        """
        channel_number = unpack(FORMATS.BYTE_FMT, message[self.channel_number_idx])
        change_type = unpack(FORMATS.BYTE_FMT, message[self.change_type_idx])
        bitmap = get_zone_from_bitmap(message[self.bitmap_idx])
        message_to_me = check_receiver_id_from_bitmap(
            message[self.bitmap_idx], self.my_id
        )
        return {
            "channel_number": channel_number,
            "change_type": change_type,
            "bitmap": bitmap,
            "message_to_me": message_to_me,
        }

    def encode(self, channel_number, change_type, receiver_id):
        """
        Encodes message as per BARP specification.
        Args:
            channel_number (int): range(1,255)
            change_type (int) : range(1,255)
            receiver_id (int): Realistically in range of 1 to  100.
        Returns:
            136 byte array message.
        """
        return (
            pack(FORMATS.BYTE_FMT, channel_number)
            + pack(FORMATS.BYTE_FMT, change_type)
            + pack(FORMATS.BYTE_FMT * 6, *[0] * 6)
            + id_to_bitmap(receiver_id)
        )


class MSG_7(MSG_BASE):
    """
    Class for BARP contol message 7 - Reject incoming call.
    Info:
    This message is sent by the master to reject a "call request" from a
    client. When the master replies with byte reason=2, the client should call
    the next master in its list (if any). This may be used as kind of call
    forwarding (see diagram Call forwarding by call reject).
    """

    def __init__(self, my_id):
        """
        Initailaizes the positional index of various elements of BARP activate relay
        message.
        """
        super(MSG_7, self).__init__(my_id)
        self.reason_idx = slice(0, 1)

    def decode(self, message):
        """
        Decodes activate relay message content
        Input:
            message (str): message without BARP header
        Returns
            dict: volume, chnage_type, bitmap
        """
        reason = unpack(FORMATS.BYTE_FMT, message[self.reason_idx])
        return {"reason": reason}

    def encode(self, reason):
        """
        Encodes message as per BARP specification.
        Args:
            reason (int): 0,1,2
        Returns:
            1 byte array
        """
        return pack(FORMATS.BYTE_FMT, reason)


class MSG_8(MSG_BASE):
    """
    Class for BARP contol message 8 - End group request.

    Info:
    This message is a request to leave a page group or to end a call.
    """

    def __init__(self, my_id):
        super(MSG_8, self).__init__(my_id)
        self.port_idx = slice(2)
        self.bitmap_idx = slice(8, 128 + 8)

    def decode(self, message):
        """
        Decodes activate relay message content
        Input:
            message (str): message without BARP header
        Returns
            dict: volume, chnage_type, bitmap
        """
        port = unpack(FORMATS.LITTLE_ENDIAN_16b_FMT, message[self.port_idx])
        bitmap = get_zone_from_bitmap(message[self.bitmap_idx])
        message_to_me = check_receiver_id_from_bitmap(
            message[self.bitmap_idx], self.my_id
        )
        return {"port": port, "bitmap": bitmap, "message_to_me": message_to_me}

    def encode(self, port, receiver_id):
        """
        Encodes message as per BARP specification.
        Args:
            port (int): range(1,65535)
            receiver_id (int): Realistically in range of 1 to  100.
        Returns:
            136 byte array message.
        """
        return (
            pack(FORMATS.LITTLE_ENDIAN_16b_FMT, port)
            + pack(FORMATS.BYTE_FMT * 6, *[0] * 6)
            + id_to_bitmap(receiver_id)
        )


class MSG_128(MSG_BASE):
    """
    Class for BARP contol message 128 - status message.
    Note: The message indexes don't match the barp.odt documentation.
    """

    def __init__(self, my_id):
        """
        Initailaizes the positional index of various elements of BARP status message.
        """
        super(MSG_128, self).__init__(my_id)
        self.state_idx = slice(0, 1)
        self.priority_idx = slice(1, 2)
        self.sender_idx = slice(2, 4)
        self.volume_idx = slice(4, 5)
        # these two indexes are not clear.
        self.config_bgm_channel_idx = slice(5, 6)
        self.client_prop_idx = slice(6, 7)

    def decode(self, message):
        """
        Decodes status message content
        Input:
            message (str): message without BARP header
        Returns
            dict: state, priority,sender, volumem config_bgm_channel, client_prop
        """
        state = unpack(FORMATS.BYTE_FMT, message[self.state_idx])
        priority = unpack(FORMATS.BYTE_FMT, message[self.priority_idx])
        sender = unpack(FORMATS.LITTLE_ENDIAN_16b_FMT, message[self.sender_idx])
        volume = unpack(FORMATS.BYTE_FMT, message[self.volume_idx])
        config_bgm_channel = unpack(
            FORMATS.BYTE_FMT, message[self.config_bgm_channel_idx]
        )
        if len(message) > 6:
            client_prop = unpack(FORMATS.BYTE_FMT, message[self.client_prop_idx])
        else:
            client_prop = None
        return {
            "state": state,
            "priority": priority,
            "sender": sender,
            "volume": volume,
            "config_bgm_channel": config_bgm_channel,
            "client_prop": client_prop,
        }

    def encode(
        self, state, priority, sender_id, volume, config_bgm_channel, client_prop=None
    ):
        """
        Encodes message as per BARP specification.
        Args:
            state (int): range(0,255)
            priority (int): Realistically in range of 1 to  100.
            sender_id (int): Realistically in range of 1 to  100.
            volume (int): Actual volume, range(0,255).
            config_bgm_channel (int): Configured BGM channel, range(0,255).
            client_prop (int): Realistically in range of 1 to  100.
        Returns:
            7 byte array message.
        """
        message = (
            pack(FORMATS.BYTE_FMT, state)
            + pack(FORMATS.BYTE_FMT, priority)
            + pack(FORMATS.LITTLE_ENDIAN_16b_FMT, sender_id)
            + pack(FORMATS.BYTE_FMT, volume)
            + pack(FORMATS.BYTE_FMT, config_bgm_channel)
        )
        if client_prop is None:
            return message
        else:
            return message + pack(FORMATS.BYTE_FMT, client_prop)


class MSG_129(MSG_BASE):
    """
    Class for BARP contol message 129 - call request.
    Indo:
    The call request is send by the client to request a call to the master. The
    request is repeated every 5 seconds until the master sends either a "reject
    incoming call" or a "page group request" message.
    """

    def __init__(self, my_id):
        super(MSG_129, self).__init__(my_id)
        self.priority_idx = slice(0, 1)
        self.description_len_idx = slice(1, 2)

    def decode(self, message):
        """
        Decodes status message content
        Input:
            message (str): message without BARP header
        Returns
            dict:  priority,description, textual description.
        """
        priority = unpack(FORMATS.BYTE_FMT, message[self.priority_idx])
        description_len = unpack(FORMATS.BYTE_FMT, message[self.description_len_idx])
        description_idx = slice(2, description_len + 2)
        description = unpack(
            FORMATS.STRING_FMT * description_len, message[description_idx]
        )
        return {
            "priority": priority,
            "description_len": description_len,
            "description": description,
        }

    def encode(self, priority, description):
        """
        Encodes message according to BARP specification.

        Args:
            priority (int): range(0,255).
            description (str): Description.

        Returns:
            2+ byte packed array.
        """
        description_parsed = [ch.encode() for ch in description]
        return (
            pack(FORMATS.BYTE_FMT, priority)
            + pack(FORMATS.BYTE_FMT, len(description_parsed))
            + pack(FORMATS.STRING_FMT * len(description), *description_parsed)
        )


class MSG_130(MSG_BASE):
    """
    Class for BARP contol message 130 - custom command.
    Indo:
    This message tells the master(s) to execute a specific command. The
    commands and the corresponding actions to be performed have to be defined
    in advance according to the specific needs, and is out of scope of this
    specification. In this sense this message may be considered as a customer
    specific command extension of the BARP protocol.
    """

    def __init__(self, my_id):
        super(MSG_130, self).__init__(my_id)
        self.command_num_idx = slice(0, 1)
        self.bitmap_idx = slice(1, 128 + 1)

    def decode(self, message):
        """
        Decodes status message content
        Input:
            message (str): message without BARP header
        Returns
            dict:  priority,description, textual description.
        """
        command_num = unpack(FORMATS.BYTE_FMT, message[self.command_num_idx])
        bitmap = get_zone_from_bitmap(message[self.bitmap_idx])
        message_to_me = check_receiver_id_from_bitmap(
            message[self.bitmap_idx], self.my_id
        )

        return {
            "command_num": command_num,
            "bitmap": bitmap,
            "message_to_me": message_to_me,
        }

    def encode(self, command_num, receiver_id):
        """
        Encodes message according to BARP specification.

        Args:
            command_num (int): range(0,255)
            receiver_id (int): Realistically in range of 1 to  100.

        Returns:
            136 byte packed array.
        """
        return (pack(FORMATS.BYTE_FMT, command_num) + id_to_bitmap(receiver_id))
