from BARP.constants import FORMATS
from BARP.utils import unpack_fields
from struct import pack
from BARP.simple_paging_utils import SimplePagingUtils


class BARP_header:
    def __init__(self, app_type=""):
        """
        Initializes the positional index of various elements of BARP activate relay
        message.
        """
        self.magic = pack(FORMATS.STRING_FMT * 4, b"B", b"A", b"R", b"\x00")
        self.status_message = pack(FORMATS.BYTE_FMT, 128)
        self.version = pack(FORMATS.BYTE_FMT, 1)
        self.reserved_4 = pack(FORMATS.BYTE_FMT * 4, 0, 0, 0, 0)
        self.reserved_2 = pack(FORMATS.BYTE_FMT * 2, 0, 0)

        self.magic_idx = slice(4)
        self.version_idx = slice(4, 5)
        self.message_type_idx = slice(5, 6)
        self.flags_idx = slice(6, 8)  # part of station name in simple paging.
        self.sender_id_idx = slice(22, 24)
        self.receiver_id_idx = slice(24, 26)
        self.sequence_number_idx = slice(26, 28)
        self.app_type = app_type
        if app_type == "simple paging":
            self.station_name_idx = SimplePagingUtils.station_name_idx
            self.device_type_idx = SimplePagingUtils.device_type_idx
            self.mac_idx = SimplePagingUtils.mac_idx

    def decode(self, header):
        """
        Decodes info from a BARP header.
        Input:
            header (str): string of length 34.
        Returns:
            dict: magic, version,message_type,flags,sender_id,receiver_id,
                sequence_number
        """
        magic = header[self.magic_idx]
        version = header[self.version_idx]
        message_type = header[self.message_type_idx]
        flags = header[self.flags_idx]
        sender_id = header[self.sender_id_idx]
        receiver_id = header[self.receiver_id_idx]
        sequence_number = header[self.sequence_number_idx]
        header_content = {
            "magic": unpack_fields(FORMATS.STRING_FMT * 4, magic),
            "version": unpack_fields(FORMATS.BYTE_FMT, version),
            "message_type": unpack_fields(FORMATS.BYTE_FMT, message_type),
            "sender_id": unpack_fields(FORMATS.LITTLE_ENDIAN_16b_FMT, sender_id),
            "receiver_id": unpack_fields(FORMATS.LITTLE_ENDIAN_16b_FMT, receiver_id),
            "sequence_number": unpack_fields(
                FORMATS.LITTLE_ENDIAN_16b_FMT, sequence_number
            ),
        }
        if self.app_type == "simple paging":
            station_name = header[self.station_name_idx]
            mac = header[self.mac_idx]
            device_type = header[self.device_type_idx]
            header_content.update(
                {
                    "station_name": SimplePagingUtils.unpack_station_name(station_name),
                    "device_type": SimplePagingUtils.unpack_device_type(device_type),
                    "mac": SimplePagingUtils.unpack_mac(mac),
                }
            )
        else:
            header_content.update(
                special_fields={
                    "flags": unpack_fields(FORMATS.LITTLE_ENDIAN_16b_FMT, flags),
                }
            )

        return header_content

    def encode(
        self,
        message_type,
        flags,
        sender_id,
        receiver_id,
        sequence_number,
        device_type=1,
        station_name="",
    ):
        """
        Encodes info into a BARP Header.
        argument device_type and station_name are optional and only used in case of simple paging.
        """
        magic = self.magic
        version = self.version
        message_type = pack(FORMATS.BYTE_FMT, message_type)
        flags = pack(FORMATS.LITTLE_ENDIAN_16b_FMT, flags)
        if self.app_type == "simple paging":
            mac = SimplePagingUtils.pack_mac()
            device_type = SimplePagingUtils.pack_device_type(device_type)
            station_name = (
                SimplePagingUtils.pack_station_name(station_name) + device_type
            )

        else:
            mac = self.reserved_2 + self.reserved_4
            station_name = flags + (self.reserved_4) * 3 + self.reserved_2
        sender_id = pack(FORMATS.LITTLE_ENDIAN_16b_FMT, sender_id)
        receiver_id = pack(FORMATS.LITTLE_ENDIAN_16b_FMT, receiver_id)
        sequence_number = pack(FORMATS.LITTLE_ENDIAN_16b_FMT, sequence_number)
        return (
            magic
            + version
            + message_type
            + station_name
            + sender_id
            + receiver_id
            + sequence_number
            + mac
        )
