#!/bin/sh

source /usr/bin/jshn.sh
export _json_no_warning=1

error_exit() {
  echo "Error: $1" 1>&2
  exit 1
}

print_var() {
  if [ ! -z $1 ]; then echo $1; else echo "Undefined";  fi;
}

print_usage() {
  echo "usage: $1"
  echo "Set the specified production info."
  echo ""
  echo "  -i|--product_id <product_id>             to set the Product ID"
  echo "  -n|--product_name <product_name>         to set the Product Name"
  echo "  -d|--production_date <production_date>   to set the Product Date"
  echo "  -b|--ipam_type <ipam_type>               to set the Legacy IPAM type"
  echo "  -c|--ipam_name <ipam_name>               to set the IPAM name"
  echo "  -w|--hw_type <hw_type>                   to set the Legacy HW Type"
  echo "  -v|--device_revision <device_revision>   to set the Device HW revision"
  echo "  -t|--test_status <test_status>           to set the test status"
  echo "  -a|--image_name <image_name>             to set the image name"
  echo "  -g|--info_sign <info_sign>               to set the production info signature"
  echo "  -e|--error_status <status>               to set the device error status"
  echo "  -h|--help                                to print this help page"
  exit 0
}

json_add_if_not_exists() {
  if ! json_select $1; then json_add_object $1; fi
}

getopt --test > /dev/null
if [[ $? -ne 4 ]]; then
    echo "I'm sorry, `getopt --test` failed in this environment."
    exit 1
fi

OPTIONS=i:n:d:w:v:t:a:b:c:f:g:e:h
LONGOPTIONS=product_id:,product_name:,production_date:,ipam_name:,ipam_type:,hw_type:device_revision:,test_status:,image_name:,image_features:,image_sign:,error_status:,help

# -temporarily store output to be able to check for errors
# -e.g. use "--options" parameter by name to activate quoting/enhanced mode
# -pass arguments only via   -- "$@"   to separate them correctly
PARSED=$(getopt --options=$OPTIONS --longoptions=$LONGOPTIONS --name "$0" -- "$@")
if [[ $? -ne 0 ]]; then
    # e.g. $? == 1
    #  then getopt has complained about wrong arguments to stdout
    exit 2
fi
# read getopt's output this way to handle the quoting right:
eval set -- "$PARSED"

# now enjoy the options in order and nicely split until we see --
while true; do
    case "$1" in
        -i|--product_id)
            PRODUCT_ID="$2"
            shift 2
            ;;
        -n|--product_name)
            PRODUCT_NAME="$2"
            shift 2
            ;;
        -d|--production_date)
            PRODUCT_DATE="$2"
            shift 2
            ;;
        -b|--ipam_type)
            IPAM_TYPE="$2"
            shift 2
            ;;
        -c|--ipam_name)
            IPAM_NAME="$2"
            shift 2
            ;;
        -w|--hw_type)
            HW_TYPE="$2"
            shift 2
            ;;
        -v|--device_hw)
            DEVICE_REVISION="$2"
            shift 2
            ;;
        -t|--test_status)
            TEST_STATUS="$2"
            shift 2
            ;;
        -a|--image_name)
            IMAGE_NAME="$2"
            shift 2
            ;;
        -g|--info_sign)
            INFO_SIGN="$2"
            shift 2
            ;;
        -e|--error-status)
            ERROR_STATUS="$2"
            shift 2
            ;;
        -h|--help)
            print_usage $0
            exit 1
            ;;
        --)
            shift
            break
            ;;
        *)
            error_exit "Programming error"
            ;;
    esac
done

if ! qiba-spi-get-info >/dev/null 2>/dev/null; then
  HW_INFO="{ }"
else
  HW_INFO=`qiba-spi-get-info`
fi
json_load "$HW_INFO"

if [ ! -z "$PRODUCT_ID" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "Product_ID" "$PRODUCT_ID"
fi

if [ ! -z "$PRODUCT_NAME" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "Product_Name" "$PRODUCT_NAME"
fi

if [ ! -z "$PRODUCT_DATE" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "Production_Date" "$PRODUCT_DATE"
fi

if [ ! -z "$IPAM_TYPE" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "Legacy_IPAM_Type" "$IPAM_TYPE"
fi

if [ ! -z "$IPAM_NAME" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "IPAM_Name" "$IPAM_NAME"
fi

if [ ! -z "$HW_TYPE" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "Legacy_HW_Type" "$HW_TYPE"
fi

if [ ! -z "$DEVICE_REVISION" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "Device_Revision" "$DEVICE_REVISION"
fi

if [ ! -z "$INFO_SIGN" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "Signature" "$INFO_SIGN"
fi

if [ ! -z "$ERROR_STATUS" ]; then
  json_add_if_not_exists HW_DEVICE
  json_add_string "Error_Status" "$ERROR_STATUS"
fi

if [ ! -z "$TEST_STATUS" ]; then
  json_add_if_not_exists HW_TEST
  json_add_string "Status" "$TEST_STATUS"
fi

if [ ! -z "$IMAGE_NAME" ]; then
  json_add_if_not_exists IMAGE
  json_add_string "Name" "$IMAGE_NAME"
fi

json_dump > /tmp/json
if ! qiba-spi-set-info /tmp/json 1>&2 >/dev/null; then
  error_exit "cannot store production info into SPI flash"
fi

exit 0
