from typing import Any

import json
import logging
import os
import secrets
import time
from flask import Blueprint, Response, request
from uci import UciException
from typing import Callable

from barix.web.webui_password import setWebUiPassword
from barix.web.exceptions import InvalidRequestBodyContentError, InvalidOldPasswordError, SettingDateAndTimeError
from barix.web.uci import getValueOfUci, formatUciTupleIntoString, setUciConfigs
from barix.web.auth import login_required, generateNewSessionKey
# from restart_services import createListOfServicesToRestart
from barix.web.system_funcs import setDateAndTime, setVolume, mute
from barix.web.AES67_rest import BacoAudioLoopREST
from barix.web.utils import readFromJSONFile

from restart_services import createListOfServicesToRestart

logger = logging.getLogger('flask-backend')

# bp = Blueprint('settings_API', __name__)

# API_KEY_FILE_PATH = "/mnt/data/config/api_key.json"

class SettingAPI:
    def __init__(self):
        self.blueprint = Blueprint('settings_API', __name__)

        @self.blueprint.route('/api/rest/v1/settings', methods=['PUT'])
        @login_required
        def setDeviceSettings():
            return_msg = {}
            try:
                data = json.loads(request.data)
                logger.debug("Device settings to set: {}".format(data))
            except Exception as e:
                logger.error(e, exc_info=True)
                return Response("Invalid request body", status=400)

            if 'pwdSettings' in data.keys():
                try:
                    setWebUiPassword(data['pwdSettings'])
                except InvalidRequestBodyContentError:
                    return Response("Invalid request body", status=400)
                except InvalidOldPasswordError:
                    return Response("Old password is invalid", status=400)
                except Exception as e:
                    logger.error(e, exc_info=True)
                    return Response(status=500)

            if 'ucis' in data.keys():
                if not 'commit' in data.keys():
                    logger.error("commit not in uciSettings")
                    return Response("Invalid request body", status=400)
                jsonUcis = data['ucis']
                try:
                    # TODO pass this "callback"
                    ucisConfigured = setUciConfigs(
                        jsonUcis, commit=data['commit'],
                        restartServices=True,
                        selectServicesToRestartCallback=createListOfServicesToRestart
                    )
                    uciListSet = []
                    for uciTuple in ucisConfigured:
                        uciListSet.append(formatUciTupleIntoString(uciTuple))
                    return_msg = {"ucisSet": uciListSet}
                except UciException as e:
                    return Response("Invalid "+e, status=400)
                except Exception as e:
                    logger.error(e, exc_info=True)
                    return Response(status=500)

            if 'timeSettings' in data.keys():
                try:
                    setDateAndTime(data['timeSettings'])
                except SettingDateAndTimeError:
                    return Response("Invalid date and time to set", status=400)
                except Exception:
                    return Response(status=500)

            return Response(json.dumps(return_msg), status=200)


        @self.blueprint.route('/api/rest/v1/settings/real-time-control', methods=['PUT'])
        @login_required
        def handleSetRealTimeControl():

            try:
                uciList = json.loads(request.data)
                uciKey = list(uciList.keys())[0]
                uciValue: Any = int(uciList[uciKey])
                # logger.warn(f"### real-time {uciKey}: {uciValue}")
            except Exception as ex:
                logger.error(ex)
                return Response(json.dumps({ "message": "Bad formatted request body"}), status=400)

            # Validators
            if uciKey == "application.audio.volume":
                if 0 > uciValue > 100:
                    return Response(json.dumps({"message": f"Parameter \"{uciKey}\" out of range: {uciValue}"}), status=400)
            elif uciKey == "application.audio.muted":
                pass
            else:
                return Response(json.dumps({"message": f"Parameter \"{uciKey}\" not supported."}), status=400)

            # Setters
            try:
                appMode = getValueOfUci("application","main_config","mode")
                if uciKey == "application.audio.volume":
                    if appMode == "aes67":
                        audioLoop = BacoAudioLoopREST(8088)
                        setVolume(uciKey, uciValue, "application.audio.muted",audioLoop.setVolume)
                    else:
                        setVolume(uciKey, uciValue, "application.audio.muted")
                elif uciKey == "application.audio.muted":
                    logger.warn("SET MUTED {}".format(uciValue))
                    if appMode == "aes67":
                        audioLoop = BacoAudioLoopREST(8088)
                        mute(uciKey, uciValue, "application.audio.volume", audioLoop.setVolume)
                    else:
                        mute(uciKey, uciValue, "application.audio.volume")
                else:
                    return Response(json.dumps({"message": f"Parameter set not implemented. Parameter: {uciKey}"}), status=400)
            except Exception as ex2:
                logger.error(ex2)
                return Response(status=500)

            return Response(status=200)


        # @bp.route('/api/rest/v1/settings/api', methods=['GET'])
        # @login_required
        # def getAPIKey():
        #     api_key_dict = {}
        #     try:
        #         if os.path.isfile(API_KEY_FILE_PATH):
        #             api_key_dict = readFromJSONFile(API_KEY_FILE_PATH)
        #     except Exception as e:
        #         logger.error("An error occurred while reading from file {}:{}".format(API_KEY_FILE_PATH,e))
        #         raise e
        #     else:
        #         return api_key_dict
        #
        # def generateAPIKey():
        #     try:
        #         api_key = secrets.token_urlsafe(32)
        #         timestamp = time.time()
        #         return {"api-key":api_key, "timestamp": timestamp}
        #     except Exception as e:
        #         logger.error(e, exc_info=True)
        #         raise e
        #
        # def storeAPIKey(api_key_dict):
        #     try:
        #         with open(API_KEY_FILE_PATH, 'w') as outfile:
        #             json.dump(api_key_dict, outfile)
        #     except Exception as e:
        #         logger.error("An error occurred while writing into file {}:{}".format(API_KEY_FILE_PATH,e))
        #         raise e
        #
        # @bp.route('/api/rest/v1/settings/api', methods=['POST'])
        # @login_required
        # def updateAPIKey():
        #     try:
        #         api_key_dict = generateAPIKey()
        #         storeAPIKey(api_key_dict)
        #         return Response(json.dumps(api_key_dict), status=200)
        #     except Exception as e:
        #         logger.error(e, exc_info=True)
        #         return Response(status=500)
