import audioop
import time
from queue import Queue, Full
from threading import Thread

import subprocess

import alsaaudio
from barix.system.barix_enums import BARPProtocol

from lib.constants import AudioControllerState, Constants
from PagingMaster import PagingMaster

import logging
logger = logging.getLogger(__name__)

class AudioController:
    """
    """

    def __init__(self, config: dict, callback=None):
        self._all_config = config
        if callback:
            logger.debug("set _notify_idle_state_fn")
            self._notify_idle_state_fn = callback
        
        # Audio config
        self.sample_rate = int(self._all_config["sample_rate"])
        self.audio_format = self._all_config["audio_format"]
        
        self.playback_device = self._all_config["audio_device"]["playback"]

        logger.debug(f"Init AudioController with: {self.sample_rate} {self.audio_format}")

        # Objects
        self.clients = []
        self._paging_master = PagingMaster(config=config)
        
        # Flags
        self._controller_state = AudioControllerState.IDLE
        self.local_running = False
        self.group_running = False
        self._kill = False
        self._kill_worker_tasks = False


        self.block_size = 480
        self.period_size = 160
        if self.sample_rate == 24000:
            self.block_size = 1440
            self.period_size = 480

        # self.delay = 0.03 # 30 ms

        # Queues
        self._queue_tasks = Queue(100) 
        self._queue_local = Queue(50) # Constants.Q_SIZE
        self._queue_group = Queue(50)

        # Threads
        self.thread_tasks = None
        self.thread_local = None
        self.thread_group = None

        self.feed_time = 0
        self.list_samplerate = []

        self.thread_send_page_group = None  # Call _paging_master.send_page_group_request

        self.thread_tasks = Thread(
            target=self._worker_tasks,
            daemon=True,
            name="worker_tasks")
        self.thread_tasks.start()


    @property
    def status(self):
        return self._controller_state

    @status.setter
    def status(self, new_status):
        if new_status != self.status:
            logger.debug(f"AudioControllerStatus {self._controller_state} new {new_status}")
            self._controller_state = new_status
            if new_status == AudioControllerState.IDLE and self._notify_idle_state_fn:
                self._notify_idle_state_fn(new_status)


    def _send_audio_socket(self, data):
        if self.audio_format == "ulaw":
            data_processed = audioop.lin2ulaw(data, 2)
        elif self.audio_format == "alaw":
            data_processed = audioop.lin2alaw(data, 2)
        else:
            raise "Invalid audio format"
        self._paging_master.send_data_audio_socket(data_processed)


    def get_new_alsa_player(self, codec):
        if codec == "MULAW":
            player_format = alsaaudio.PCM_FORMAT_MU_LAW
        elif codec == "ALAW":
            player_format = alsaaudio.PCM_FORMAT_A_LAW
        else:
            player_format = alsaaudio.PCM_FORMAT_S16_LE

        
        # playback_device = "plug:analog_out_8k"
        # if self.sample_rate == 24000:
        #     playback_device = "plug:analog_out_24k"


        alsa_player = alsaaudio.PCM(
            alsaaudio.PCM_PLAYBACK, alsaaudio.PCM_NORMAL, self.playback_device  # plug:master1
        )
        alsa_player.setchannels(1)
        # alsa_player.setperiodsize(32)
        alsa_player.setperiodsize(self.period_size)
        alsa_player.setrate(self.sample_rate)
        alsa_player.setformat(player_format)

        data = alsa_player.info()

        logger.debug(f"playback_device: {self.playback_device}")
        logger.debug(f"period size: {data['period_size']}")
        logger.debug(f"get_new_alsa_player {self.sample_rate} {player_format}")
        
        import json
        logger.debug(json.dumps(alsa_player.info()))

        return alsa_player

    # "force" stop
    def stop_audio(self):
        self._kill = True
        
        if self.thread_local:
            self.thread_local.join()
        self.thread_local = None
        logger.debug("end thread_local")

        if self.thread_group:
            self.thread_group.join()
        self.thread_group = None
        logger.debug("end thread_group")

        # clear queue
        self._queue_tasks = Queue(100)
        self._queue_local = Queue(50)
        self._queue_group = Queue(50)

        self.status = AudioControllerState.IDLE

        logger.debug("audio stopped")
        

    def add_task_play_file(self, data):
        data['task']="play_file"
        self._queue_tasks.put(data)
        logger.debug("add_task_play_file")

    def add_task_play_recorded(self, data):
        data['task']="play_recorded"
        self._queue_tasks.put(data)
        logger.debug("add_task_play_recorded")

    def add_task_page_audio(self, data):
        data['task']="page_audio"
        self._queue_tasks.put(data)
        logger.debug("add_task_page_audio")

    def add_task_start(self, data):
        data['task']="start"
        self._queue_tasks.put(data)
        logger.info("add_task_start")

    # when finish playing audio, stop threads
    def add_task_stop(self):
        data = {}
        data['task']="stop"
        self._queue_tasks.put(data)
        logger.info("add_task_stop")


    def _add_to_queues(self, audio):
        while not self._kill:
            try:
                self._queue_local.put(audio, block=False)
                # logger.debug(f"feed queue {len(audio)}")

                break  # Exit loop once added
            except Full:
                logger.warning("Queue is full, waiting...")
                time.sleep(0.01)  # Sleep for 10ms before retrying

        if not self._kill and self.clients:
            self._queue_group.put(audio)


    def _run_queue_audio_file(self, filepath):
        block_size = self.block_size * 10
        command = f"ffmpeg -y -i {filepath} -acodec pcm_mulaw -ar {self.sample_rate} -ac 1 -f wav -"

        logger.debug(f"run cmd: {command}")

        # Start ffmpeg process
        ffmpeg_proc = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.DEVNULL, shell=True)

        # Stream audio as it comes from ffmpeg
        try:
            # remove file header
            audio = ffmpeg_proc.stdout.read(600)

            while not self._kill:
                audio = ffmpeg_proc.stdout.read(block_size)
                if not audio:
                    break

                audio_data = audioop.ulaw2lin(audio, 2)
                self._add_to_queues(audio_data)
                
        finally:
            ffmpeg_proc.stdout.close()
            ffmpeg_proc.wait()


    def _run_queue_audio_data(self, audio_data):
        block_size = self.block_size * 12
        for i in range(0, len(audio_data), block_size):
            audio = audio_data[i:i + block_size]
            
            self._add_to_queues(audio)            
            if self._kill:
                break


    def _run_task_start(self, clients):
        self._kill = False

        self.clients = clients
        logger.debug(f"clients {self.clients}")

        self.thread_local = Thread(
            target=self._worker_local,
            daemon=True,
            name="worker_local")
        self.thread_local.start()

        if self.clients:
            self.thread_group = Thread(
                target=self._worker_group,
                daemon=True,
                name="worker_group")
            self.thread_group.start()


    # when finish playing audio, stop threads
    def _run_task_stop(self):
        # wait queue empty
        # stop threads
        while not self._kill:
            
            #not self._queue_local.empty() or not self._queue_group.empty()
            # logger.debug(f"queue empty {self._queue_local.empty()}")

            if self._queue_local.empty():
                logger.debug("Queue empty")
                break

            # time.sleep(0.01)
            # logger.debug("wait queue empty")
            time.sleep(0.05)
        
        self.stop_audio()


    def _worker_tasks(self):
        logger.info("_worker_tasks running")
        while not self._kill_worker_tasks:
            try:
                if not self._queue_tasks.empty():
                    data = self._queue_tasks.get(block=False)
                    
                    logger.debug(f"process task {data['task']}")
                    if  data['task']=="start":
                        self._run_task_start(data['clients'])
                    elif data['task']=="stop":
                        self._run_task_stop()
                    elif data['task']=="play_file":
                        self._run_queue_audio_file(data['filepath'])
                    elif data['task']=="play_recorded":
                        self._run_queue_audio_data(data['audio'])
                    elif data['task']=="page_audio":
                        self._run_queue_audio_data(data['audio'])

                    # logger.debug(f"finish process task {data['task']}")

                    if self._queue_tasks is not None and not self._queue_tasks.empty():
                        self._queue_tasks.task_done()
            except Exception as e:
                logger.warning(f"_worker_tasks Following occurred: {e}", exc_info=True)
            time.sleep(0.001)
        logger.info("_worker_tasks terminated")


    def _worker_local(self):
        logger.info("_worker_local running")
        local_alsa = self.get_new_alsa_player(codec="")
        data = b''
        
        while not self._kill:
            try:
                if not self._queue_local.empty():
                    # logger.debug(f"q_queue_local.qsize() {self._queue_local.qsize()}")
                    
                    data += self._queue_local.get(block=False)
                    
                    logger.debug(f"consume _queue_local {len(data)}")
                    while len(data) >= self.period_size:
                        audio = data[:self.period_size]
                        local_alsa.write(audio)
                        data = data[self.period_size:]  # Remove processed audio


                    self._queue_local.task_done()
                else:
                    if self.sample_rate == 24000:
                        audio = bytes([0xFF] * 480)
                    else:
                        audio = bytes([0xFF] * 160)
                    # logger.debug("add silence")
                    # local_alsa.write(audio)

            except Exception as e:
                logger.warning(f"_worker_local Following occurred: {e}")
            time.sleep(0.001)
        logger.info("_worker_local terminated")


    def _worker_group(self):
        logger.info("_worker_group running")
        
        self.thread_send_page_group = Thread(
            target=self._paging_master.send_page_group_request,
            daemon=True,
            kwargs={"clients": self.clients},
            name="PageGroup",
        )
        self.thread_send_page_group.start()

        first_fast_packets = 0
        

        block_size = self.block_size * 2
        # 960 8kHz
        # 2880 24kHz
        
        data = b''
        INTERVAL = 0.03  # 30 milliseconds
        while not self._kill:
            try:
                if not self._queue_group.empty():
                    # logger.debug("consume _queue_group")

                    data = self._queue_group.get(block=False)

                    for i in range(0, len(data), block_size):
                        audio = data[i:i + block_size]
                        self._send_audio_socket(audio)
                                                
                        if first_fast_packets < Constants.PACKETS_NUMBER:
                            time.sleep(0.004)
                            first_fast_packets += 1
                        else:
                            time.sleep(0.025)
                    

                    self._queue_group.task_done()
            except Exception as e:
                logger.warning(f"Following occurred: {e}")
            time.sleep(0.001)
        logger.info("_worker_group terminated")

        self._paging_master.send_end_group_request()

        while self._paging_master.state != BARPProtocol.NOT_USED:
            time.sleep(0.01)

        # self.status = AudioControllerState.IDLE
        logger.info(f"finished playing audio_data")
