#!/usr/bin/env python3
"""
UDP Command API for Barionet M44
Handles UDP commands for IO operations with IoMapping integration
"""

import socket
import json
import os
import sys
import time
import threading
import signal
import logging
import logging.handlers
import subprocess
import re
from pathlib import Path
from typing import Dict, List, Optional, Tuple, Any
from datetime import datetime

try:
    from iomapping import IoMapping
except ImportError:
    print("Error: iomapping module not found. Please ensure IoMapping is installed.")
    sys.exit(1)

# Global shutdown flag
EXIT_APP = False

class ConfigManager:
    """Manages application configuration"""
    
    def __init__(self):
        self.config = self.load_config()
        
    def load_config(self) -> Dict:
        """Load configuration from config.json or default_config.json"""
        config_files = ['config.json', 'default_config.json']
        
        for config_file in config_files:
            if os.path.exists(config_file):
                try:
                    with open(config_file, 'r') as f:
                        data = json.load(f)
                        if 'AppParam' in data:
                            return data['AppParam']
                except Exception as e:
                    print(f"Error loading {config_file}: {e}")
                    
        # Default configuration if no file found
        return {
            "UDP_api_enable": True,
            "UDP_port": 12301,
            "allowed_ips": "",
            "password": "",
            "UDP_state_info_enable": False,
            "UDP_target_address": "",
            "UDP_state_info_interval": 0,
            "enable_syslog": False,
            "syslog_address": ""
        }

class SyslogHandler:
    """Handles syslog logging"""
    
    def __init__(self, config: Dict):
        self.enabled = config.get('enable_syslog', False)
        self.address = config.get('syslog_address', '')
        self.logger = logging.getLogger('UDPCommandAPI')
        self.logger.setLevel(logging.INFO)
        
        # Console handler - only if running from terminal
        if sys.stdout.isatty():
            console_handler = logging.StreamHandler()
            console_handler.setLevel(logging.INFO)
            formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
            console_handler.setFormatter(formatter)
            self.logger.addHandler(console_handler)
        
        # Syslog handler
        if self.enabled and self.address:
            try:
                host, port = self.address.split(':')
                port = int(port)
                syslog_handler = logging.handlers.SysLogHandler(
                    address=(host, port),
                    facility=logging.handlers.SysLogHandler.LOG_LOCAL0
                )
                formatter = logging.Formatter('UDPCommandAPI: %(message)s')
                syslog_handler.setFormatter(formatter)
                self.logger.addHandler(syslog_handler)
            except Exception as e:
                print(f"Failed to setup syslog: {e}")
                
    def log(self, level: str, message: str):
        """Log a message"""
        if level == 'info':
            self.logger.info(message)
        elif level == 'error':
            self.logger.error(message)
        elif level == 'warning':
            self.logger.warning(message)
        else:
            self.logger.debug(message)

class IOAddressValidator:
    """Validates IO addresses based on addressing table"""
    
    def __init__(self):
        # Define address ranges and their properties
        # Format: (start, end, readable, writable, bit_size)
        self.address_map = [
            (1, 4, True, True, 1),        # Built-in relays
            (5, 8, False, False, 1),       # Reserved
            (9, 9, True, True, 1),         # RTS
            (10, 10, True, True, 1),       # Virtual IO
            (11, 42, True, True, 1),       # UX8 relays (or virtual if no UX8)
            (43, 100, True, True, 1),      # Virtual IO
            (101, 108, True, True, 1),     # Digital outputs
            (109, 200, True, True, 1),     # Virtual IO
            (201, 204, True, False, 1),    # Digital inputs
            (205, 208, False, False, 1),   # Reserved
            (209, 209, True, False, 1),    # CTS
            (210, 210, True, True, 1),     # Virtual IO
            (211, 242, True, False, 1),    # UX8 inputs (or virtual if no UX8)
            (243, 300, True, True, 1),     # Virtual IO
            (301, 304, True, True, 1),     # Pullups
            (305, 308, False, False, 1),   # Reserved
            (309, 400, True, True, 1),     # Virtual IO
            (401, 404, True, True, 32),    # Input counters
            (405, 408, False, False, 32),  # Reserved
            (409, 410, True, True, 32),    # Virtual registers
            (411, 442, True, True, 32),    # UX8 counters
            (443, 500, True, True, 32),    # Virtual registers
            (501, 504, True, False, 16),   # Analog inputs
            (505, 508, False, False, 16),  # Reserved
            (509, 510, True, True, 16),    # Virtual registers
            (511, 542, True, False, 16),   # UX8 analog
            (543, 600, True, True, 16),    # Virtual registers
            (601, 650, True, False, 16),   # Temperature sensors
            (651, 750, True, False, 32),   # Temperature addresses
            (751, 1200, True, True, 16),   # Virtual IO
            (1201, 1206, True, False, 16), # Device info
            (1207, 1207, True, True, 1),   # USB enable
            (1208, 1211, True, True, 4),   # LEDs
            (1212, 1244, True, True, 1),   # Analog enable
            (60001, 60010, True, False, 16) # Count/detection
        ]
        
    def is_readable(self, address: int) -> bool:
        """Check if address is readable"""
        for start, end, readable, _, _ in self.address_map:
            if start <= address <= end:
                return readable
        return False
        
    def is_writable(self, address: int) -> bool:
        """Check if address is writable"""
        for start, end, _, writable, _ in self.address_map:
            if start <= address <= end:
                return writable
        return False
        
    def get_bit_size(self, address: int) -> int:
        """Get bit size for address"""
        for start, end, _, _, bit_size in self.address_map:
            if start <= address <= end:
                return bit_size
        return 0
        
    def is_valid_value(self, address: int, value: int) -> bool:
        """Check if value is valid for address"""
        bit_size = self.get_bit_size(address)
        if bit_size == 1:
            return 0 <= value <= 1
        elif bit_size == 4:
            return 0 <= value <= 15
        elif bit_size == 16:
            return 0 <= value <= 65535
        elif bit_size == 32:
            return 0 <= value <= 4294967295
        return False
        
    def is_relay_or_output_or_virtual(self, address: int) -> bool:
        """Check if address is a relay, digital output, or 1-bit virtual IO"""
        if self.get_bit_size(address) != 1:
            return False
        if not self.is_writable(address):
            return False
        # Relays, outputs, and virtual IOs
        ranges = [(1, 4), (9, 10), (11, 42), (43, 100), (101, 108), 
                  (109, 200), (210, 210), (243, 300), (309, 400)]
        for start, end in ranges:
            if start <= address <= end:
                return True
        return False

class UX8Detector:
    """Detects UX8 extensions"""
    
    def __init__(self, io_service: IoMapping, logger: SyslogHandler):
        self.io_service = io_service
        self.logger = logger
        self.ux8_count = 0
        self.ux8_addresses = {
            'relays': [],
            'inputs': []
        }
        
    def detect(self) -> int:
        """Detect connected UX8 units"""
        count = 0
        for i in range(4):
            address = 60007 + i
            try:
                value = self.io_service.read_value(address)
                if value == 1:
                    count += 1
                    # Add relay addresses for this UX8
                    relay_start = 11 + (i * 8)
                    relay_end = relay_start + 7
                    self.ux8_addresses['relays'].extend(range(relay_start, relay_end + 1))
                    # Add input addresses for this UX8
                    input_start = 211 + (i * 8)
                    input_end = input_start + 7
                    self.ux8_addresses['inputs'].extend(range(input_start, input_end + 1))
            except Exception as e:
                self.logger.log('error', f"Error detecting UX8 at address {address}: {e}")
                
        self.ux8_count = count
        self.logger.log('info', f"Detected {count} UX8 extension(s)")
        return count
        
    def get_notification_addresses(self) -> Dict[str, List[int]]:
        """Get addresses that should have notifications enabled"""
        addresses = {
            'relays': list(range(1, 5)),  # Built-in relays always included
            'inputs': list(range(201, 205))  # Built-in inputs always included
        }
        
        # Add UX8 addresses if detected
        addresses['relays'].extend(self.ux8_addresses['relays'])
        addresses['inputs'].extend(self.ux8_addresses['inputs'])
        
        return addresses
        
    def update_validator_for_virtual_io(self, validator: IOAddressValidator):
        """Update validator to treat unused UX8 addresses as virtual IOs"""
        # Only relay addresses (11-42) can become writable virtual IOs
        # Input addresses (211-242) always remain read-only
        
        if self.ux8_count == 0:
            # Update address map for 11-42 to be writable (virtual IOs)
            for i, (start, end, readable, writable, bit_size) in enumerate(validator.address_map):
                if start == 11 and end == 42:
                    validator.address_map[i] = (11, 42, True, True, 1)
                # 211-242 remain read-only
        else:
            # Set unused UX8 relay addresses as virtual IOs (writable)
            for i in range(self.ux8_count, 4):
                relay_start = 11 + (i * 8)
                relay_end = relay_start + 7
                
                # Only update relay ranges to be writable
                for j, (start, end, readable, writable, bit_size) in enumerate(validator.address_map):
                    if start == 11 and end == 42:
                        # Create new entry for unused relay range as writable
                        if i == self.ux8_count:
                            # First unused UX8 slot
                            validator.address_map.append((relay_start, relay_end, True, True, 1))
                # Input addresses 211-242 always remain read-only

class TimerManager:
    """Manages timed operations for addresses"""
    
    def __init__(self, io_service: IoMapping, logger: SyslogHandler):
        self.io_service = io_service
        self.logger = logger
        self.timers = {}
        self.lock = threading.Lock()
        
    def start_timer(self, address: int, duration: float):
        """Start a timer for an address"""
        with self.lock:
            # Cancel existing timer if any
            if address in self.timers:
                self.timers[address].cancel()
                
            # Create new timer
            timer = threading.Timer(duration, self._timer_callback, args=(address,))
            self.timers[address] = timer
            timer.start()
            
    def cancel_timer(self, address: int):
        """Cancel a timer for an address"""
        with self.lock:
            if address in self.timers:
                self.timers[address].cancel()
                del self.timers[address]
                
    def cancel_all_timers(self):
        """Cancel all active timers"""
        with self.lock:
            for timer in self.timers.values():
                timer.cancel()
            self.timers.clear()
            
    def _timer_callback(self, address: int):
        """Timer callback to reset address to 0"""
        try:
            self.io_service.write_value(address, 0)
            self.logger.log('info', f"Timer expired for address {address}, reset to 0")
        except Exception as e:
            self.logger.log('error', f"Error resetting address {address}: {e}")
        
        with self.lock:
            if address in self.timers:
                del self.timers[address]

class CommandHandler:
    """Handles UDP commands"""
    
    def __init__(self, io_service: IoMapping, validator: IOAddressValidator, 
                 timer_manager: TimerManager, ux8_detector: UX8Detector, 
                 logger: SyslogHandler):
        self.io_service = io_service
        self.validator = validator
        self.timer_manager = timer_manager
        self.ux8_detector = ux8_detector
        self.logger = logger
        
    def handle_command(self, command: str) -> str:
        """Process a command and return response"""
        try:
            # Handle command concatenation
            if '&' in command:
                return self.handle_concatenated_commands(command)
                
            # Parse single command
            command = command.strip()
            
            # Check for specific commands
            if command == 'version':
                return self.handle_version()
            elif command == 'c=65535':
                return self.handle_dhcp_name()
            elif command == 'iolist':
                return self.handle_iolist()
            elif command.startswith('getio,'):
                return self.handle_getio(command)
            elif command.startswith('setio,'):
                return self.handle_setio(command)
            else:
                return "cmderr\r"
                
        except Exception as e:
            self.logger.log('error', f"Error handling command: {e}")
            return "cmderr\r"
            
    def handle_concatenated_commands(self, commands: str) -> str:
        """Handle multiple commands separated by &"""
        parts = commands.split('&')
        responses = []
        
        for part in parts:
            if not part:
                continue
            response = self.handle_command(part)
            # Remove \r for concatenation
            response = response.rstrip('\r')
            responses.append(response)
            
        return '&'.join(responses) + '\r'
        
    def handle_version(self) -> str:
        """Handle version command"""
        try:
            # Get hardware info
            result = subprocess.run(['qiba-spi-get-info'], capture_output=True, text=True, timeout=5)
            if result.returncode != 0:
                return "command failed\r"
                
            hw_info = json.loads(result.stdout)
            product_name = hw_info.get('HW_DEVICE', {}).get('Product_Name', '')
            image_name = hw_info.get('IMAGE', {}).get('Name', '')
            
            # Get version
            result = subprocess.run(['cat', '/barix/info/VERSION'], capture_output=True, text=True, timeout=5)
            if result.returncode != 0:
                return "command failed\r"
                
            version = result.stdout.strip()
            
            return f"version,{product_name} {image_name} {version}\r"
            
        except Exception as e:
            self.logger.log('error', f"Error getting version: {e}")
            return "command failed\r"
            
    def handle_dhcp_name(self) -> str:
        """Handle c=65535 command"""
        try:
            result = subprocess.run(['uci', 'show', 'network.eth0.dhcpname'], 
                                  capture_output=True, text=True, timeout=5)
            
            # Check if entry not found
            if result.returncode != 0:
                if "Entry not found" in result.stderr:
                    return "<BARIONET><NAME>DHCP HOSTNAME NOT CONFIGURED</NAME></BARIONET>\r"
                else:
                    return "command failed\r"
                
            # Parse output: network.eth0.dhcpname='BarionetM44'
            output = result.stdout.strip()
            match = re.search(r"='(.+)'", output)
            if match:
                dhcp_name = match.group(1)
                return f"<BARIONET><NAME>{dhcp_name}</NAME></BARIONET>\r"
            else:
                return "command failed\r"
                
        except Exception as e:
            self.logger.log('error', f"Error getting DHCP name: {e}")
            return "command failed\r"
            
    def handle_iolist(self) -> str:
        """Handle iolist command"""
        try:
            # Read required addresses
            val1 = self.io_service.read_value(60006)
            val2 = self.io_service.read_value(60004)
            val3 = self.io_service.read_value(60005)
            val4 = self.io_service.read_value(60003)
            val5 = 0  # Always 0
            val6 = self.io_service.read_value(60002)
            
            # Count 1-wire devices
            w1_path = Path('/sys/bus/w1/devices')
            if w1_path.exists():
                # Count directories that look like 1-wire devices (exclude w1_bus_master)
                devices = [d for d in w1_path.iterdir() if d.is_dir() and not d.name.startswith('w1_bus_master')]
                val7 = len(devices)
            else:
                val7 = 0
                
            return f"io,{val1},{val2},{val3},{val4},{val5},{val6},{val7}\r"
            
        except Exception as e:
            self.logger.log('error', f"Error handling iolist: {e}")
            return "cmderr\r"
            
    def handle_getio(self, command: str) -> str:
        """Handle getio command"""
        try:
            parts = command.split(',')
            if len(parts) != 2:
                return "cmderr\r"
                
            address = int(parts[1])
            
            if not self.validator.is_readable(address):
                return "cmderr\r"
                
            value = self.io_service.read_value(address)
            return f"state,{address},{value}\r"
            
        except (ValueError, Exception) as e:
            self.logger.log('error', f"Error handling getio: {e}")
            return "cmderr\r"
            
    def handle_setio(self, command: str) -> str:
        """Handle setio command"""
        try:
            parts = command.split(',')
            if len(parts) != 3:
                return "cmderr\r"
                
            address = int(parts[1])
            value = int(parts[2])
            
            if not self.validator.is_writable(address):
                return "cmderr\r"
                
            # Check for special functions (relay/output/virtual IO only)
            if self.validator.is_relay_or_output_or_virtual(address):
                # Toggle function
                if value == 999:
                    current_value = self.io_service.read_value(address)
                    new_value = 1 if current_value == 0 else 0
                    self.io_service.write_value(address, new_value)
                    self.timer_manager.cancel_timer(address)
                    return f"state,{address},{new_value}\r"
                    
                # Timed function
                elif (2 <= value <= 998) or (1000 <= value <= 9999):
                    duration = value / 10.0
                    self.io_service.write_value(address, 1)
                    self.timer_manager.start_timer(address, duration)
                    return f"state,{address},1\r"
                    
                # Normal write
                elif value in [0, 1]:
                    self.io_service.write_value(address, value)
                    # Cancel any active timer if writing 0
                    if value == 0:
                        self.timer_manager.cancel_timer(address)
                    return f"state,{address},{value}\r"
                else:
                    return "cmderr\r"
            else:
                # For non-1-bit addresses, validate value range
                if not self.validator.is_valid_value(address, value):
                    return "cmderr\r"
                    
                self.io_service.write_value(address, value)
                return f"state,{address},{value}\r"
                
        except (ValueError, Exception) as e:
            self.logger.log('error', f"Error handling setio: {e}")
            return "cmderr\r"

class NotificationService:
    """Handles state change and periodic notifications"""
    
    def __init__(self, config: Dict, io_service: IoMapping, ux8_detector: UX8Detector, 
                 logger: SyslogHandler):
        self.config = config
        self.io_service = io_service
        self.ux8_detector = ux8_detector
        self.logger = logger
        self.enabled = config.get('UDP_state_info_enable', False)
        self.target_address = config.get('UDP_target_address', '')
        self.interval = config.get('UDP_state_info_interval', 0)
        self.notification_thread = None
        self.periodic_thread = None
        self.sock = None
        self.last_values = {}
        
        if self.enabled and self.target_address:
            self.setup_socket()
            
    def setup_socket(self):
        """Setup UDP socket for notifications"""
        try:
            self.sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            host, port = self.target_address.split(':')
            self.target = (host, int(port))
        except Exception as e:
            self.logger.log('error', f"Failed to setup notification socket: {e}")
            self.enabled = False
            
    def start(self):
        """Start notification services"""
        if not self.enabled:
            self.logger.log('info', f"Notification service disabled")
            return
            
        # Setup state change notifications
        addresses = self.ux8_detector.get_notification_addresses()
        all_addresses = addresses['relays'] + addresses['inputs']
        
        try:
            self.io_service.enable_notifications_range(all_addresses, self.notification_callback)
            self.logger.log('info', f"Enabled notifications for {len(all_addresses)} addresses")    
        except Exception as e:
            self.logger.log('error', f"Failed to enable notifications: {e}")
            # Continue even if notifications fail
            
        # Start periodic dump if configured
        if self.interval > 0:
            self.periodic_thread = threading.Thread(target=self.periodic_dump_worker)
            self.periodic_thread.daemon = True
            self.periodic_thread.start()
            self.logger.log('info', f"Started periodic state dump with {self.interval}s interval")
            
    def stop(self):
        """Stop notification services"""
        if not self.enabled:
            return
            
        # Disable notifications
        addresses = self.ux8_detector.get_notification_addresses()
        all_addresses = addresses['relays'] + addresses['inputs']
        
        try:
            self.io_service.disable_notifications_range(all_addresses)
        except Exception as e:
            self.logger.log('error', f"Failed to disable notifications: {e}")
            
        if self.sock:
            self.sock.close()
            
    def notification_callback(self, address: int, value: int):
        """Callback for state change notifications"""
        try:
            # Suppress redundant notifications
            if self.last_values.get(address) == value:
                return 

            self.last_values[address] = value

            message = f"statechange,{address},{value}\r"
            self.sock.sendto(message.encode(), self.target)
            self.logger.log('info', f"Sent notification: {message.strip()}")
        except Exception as e:
            self.logger.log('error', f"Failed to send notification: {e}")
            
    def periodic_dump_worker(self):
        """Worker thread for periodic state dumps"""
        global EXIT_APP
        
        while not EXIT_APP:
            try:
                time.sleep(self.interval)
                if EXIT_APP:
                    break
                    
                # Build state dump
                messages = []
                addresses = self.ux8_detector.get_notification_addresses()
                
                # Add relay states
                for addr in sorted(addresses['relays']):
                    try:
                        value = self.io_service.read_value(addr)
                        messages.append(f"state,{addr},{value}")
                    except:
                        pass
                        
                # Add input states
                for addr in sorted(addresses['inputs']):
                    try:
                        value = self.io_service.read_value(addr)
                        messages.append(f"state,{addr},{value}")
                    except:
                        pass
                        
                if messages:
                    dump = '\r'.join(messages) + '\r'
                    self.sock.sendto(dump.encode(), self.target)
                    self.logger.log('info', f"Sent periodic dump with {len(messages)} states")
                    
            except Exception as e:
                self.logger.log('error', f"Error in periodic dump: {e}")

class UDPServer:
    """Main UDP server"""
    
    def __init__(self, config: Dict, io_service: IoMapping, logger: SyslogHandler):
        self.config = config
        self.io_service = io_service
        self.logger = logger
        self.port = config.get('UDP_port', 12301)
        self.allowed_ips = self.parse_allowed_ips(config.get('allowed_ips', ''))
        self.password = config.get('password', '')
        self.sock = None
        self.validator = IOAddressValidator()
        self.timer_manager = TimerManager(io_service, logger)
        self.ux8_detector = UX8Detector(io_service, logger)
        self.command_handler = None
        self.notification_service = None
        
    def parse_allowed_ips(self, ip_string: str) -> List[str]:
        """Parse allowed IPs from configuration"""
        if not ip_string:
            return []
            
        # Validate format with regex
        pattern = r'^$|^((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)(\s*,\s*((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?))*$'
        
        if not re.match(pattern, ip_string):
            self.logger.log('error', f"Invalid allowed_ips format: {ip_string}")
            return []
            
        return [ip.strip() for ip in ip_string.split(',') if ip.strip()]
        
    def is_ip_allowed(self, client_ip: str) -> bool:
        """Check if client IP is allowed"""
        if not self.allowed_ips:
            return True
        return client_ip in self.allowed_ips
        
    def authenticate_message(self, message: str) -> Tuple[bool, str]:
        """Authenticate message with password if configured"""
        if not self.password:
            return True, message
            
        # Check for password prefix
        if message.startswith(f"a={self.password}&"):
            return True, message[len(f"a={self.password}&"):]
        else:
            return False, ""
            
    def initialize(self):
        """Initialize server components"""
        try:
            # Detect UX8 units
            self.logger.log('info', "Detecting UX8 units...")
            self.ux8_detector.detect()
            
            # Update validator for virtual IOs
            self.logger.log('info', "Updating validator for virtual IOs...")
            self.ux8_detector.update_validator_for_virtual_io(self.validator)
            
            # Initialize all writable 1-bit addresses to 0
            self.logger.log('info', "Initializing outputs to 0...")
            self.initialize_outputs()
            
            # Create command handler
            self.logger.log('info', "Creating command handler...")
            self.command_handler = CommandHandler(
                self.io_service, self.validator, self.timer_manager, 
                self.ux8_detector, self.logger
            )
            
            # Setup notification service
            self.logger.log('info', "Setting up notification service...")
            self.notification_service = NotificationService(
                self.config, self.io_service, self.ux8_detector, self.logger
            )
            self.notification_service.start()

            self.logger.log('info', "Server initialization complete")
        except Exception as e:
            self.logger.log('error', f"Error during server initialization: {e}")
            raise
        
    def initialize_outputs(self):
        """Initialize all writable 1-bit addresses to 0"""
        # Ranges to initialize
        ranges = [
            (1, 4),      # Built-in relays
            (9, 10),     # RTS, Virtual IO
            (11, 42),    # UX8 relays or virtual
            (43, 100),   # Virtual IO
            (101, 108),  # Digital outputs
            (109, 200),  # Virtual IO
            (210, 210),  # Virtual IO
            (243, 300),  # Virtual IO
            (309, 400),  # Virtual IO
        ]
        
        initialized_count = 0
        for start, end in ranges:
            for addr in range(start, end + 1):
                if self.validator.is_writable(addr) and self.validator.get_bit_size(addr) == 1:
                    # Only write 0 if the value is not 0 already
                    current = self.io_service.read_value(addr)
                    if current != 0:
                        try:
                            self.io_service.write_value(addr, 0)
                            initialized_count += 1
                        except Exception:
                            # Some addresses might not be available, continue
                            pass
                        
        self.logger.log('info', f"Initialized {initialized_count} writable 1-bit addresses to 0")
        
    def start(self):
        """Start UDP server"""
        try:
            self.sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            #self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            #self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            self.sock.bind(('', self.port))
            self.logger.log('info', f"UDP server started on port {self.port}")
            self.run()
        except Exception as e:
            self.logger.log('error', f"Failed to start UDP server: {e}")
            raise
            
    def run(self):
        """Main server loop"""
        global EXIT_APP
        
        while not EXIT_APP:
            try:
                # Set timeout for socket to allow checking EXIT_APP
                self.sock.settimeout(0.1)
                
                try:
                    data, addr = self.sock.recvfrom(256)
                except socket.timeout:
                    # Run IoMapping notifications (non-blocking)
                    if self.notification_service and self.notification_service.enabled:
                        try:
                            self.io_service.run(False)
                        except:
                            pass
                    continue
                    
                client_ip = addr[0]
                
                # Decode message
                try:
                    message = data.decode('utf-8').strip()
                    message = message.rstrip('\r\n\00').strip() #Accepts commands terminated with \r\n\00
                except:
                    self.logger.log('error', f"Invalid bytes received from {client_ip}")
                    continue
                    
                # Log incoming message
                self.logger.log('info', f"Received from {client_ip}: {message}")
                
                # Check IP filter
                if not self.is_ip_allowed(client_ip):
                    self.logger.log('warning', f"Blocked request from unauthorized IP: {client_ip}")
                    continue
                    
                # Authenticate message
                authenticated, command = self.authenticate_message(message)
                if not authenticated:
                    response = "operation not allowed\r"
                    self.sock.sendto(response.encode(), addr)
                    self.logger.log('warning', f"Authentication failed from {client_ip}")
                    continue
                    
                # Validate lowercase (except for c=65535 which is validated entirely)
                if command != 'c=65535' and command.lower() != command:
                    response = "cmderr\r"
                    self.sock.sendto(response.encode(), addr)
                    continue
                    
                # Process command
                response = self.command_handler.handle_command(command)
                
                # Send response
                self.sock.sendto(response.encode(), addr)
                
                # Run IoMapping notifications (non-blocking)
                if self.notification_service and self.notification_service.enabled:
                    try:
                        self.io_service.run(False)
                    except:
                        pass
                        
            except Exception as e:
                self.logger.log('error', f"Error in server loop: {e}")
                
    def stop(self):
        """Stop UDP server"""
        self.logger.log('info', "Stopping UDP server...")
        
        # Cancel all timers
        self.timer_manager.cancel_all_timers()
        
        # Stop notifications
        if self.notification_service:
            self.notification_service.stop()
            
        # Close socket
        if self.sock:
            self.sock.close()
            
        self.logger.log('info', "UDP server stopped")

class Application:
    """Main application controller"""
    
    def __init__(self):
        self.config_manager = ConfigManager()
        self.config = self.config_manager.config
        self.logger = SyslogHandler(self.config)
        self.io_service = None
        self.server = None
        
    def initialize_io_service(self) -> bool:
        """Initialize IoMapping service with retry"""
        max_retries = 10
        retry_delay = 1
        
        for attempt in range(max_retries):
            try:
                self.io_service = IoMapping()
                # Test if service is working
                self.io_service.read_value(1)
                self.logger.log('info', "IoMapping service initialized successfully")
                return True
            except Exception as e:
                if "org.barix.IoMapping was not provided" in str(e):
                    self.logger.log('warning', f"IoMapping service not ready, attempt {attempt + 1}/{max_retries}")
                    time.sleep(retry_delay)
                else:
                    self.logger.log('error', f"Error initializing IoMapping: {e}")
                    time.sleep(retry_delay)
                    
        self.logger.log('error', "Failed to initialize IoMapping service after 10 attempts")
        return False
        
    def run(self):
        """Run the application"""
        global EXIT_APP
        
        # Check if UDP API is enabled
        if not self.config.get('UDP_api_enable', True):
            self.logger.log('info', "UDP API is disabled in configuration")
            return
            
        # Setup signal handlers
        signal.signal(signal.SIGINT, self.signal_handler)
        signal.signal(signal.SIGTERM, self.signal_handler)
        
        try:
            # Initial startup delay
            self.logger.log('info', "Starting up, waiting 3 seconds for system initialization...")
            time.sleep(3)
            
            # Initialize IoMapping service
            if not self.initialize_io_service():
                self.logger.log('error', "Cannot proceed without IoMapping service")
                return
                
            # Create and initialize server
            self.server = UDPServer(self.config, self.io_service, self.logger)
            self.server.initialize()
            
            # Start server
            self.server.start()
            
        except KeyboardInterrupt:
            self.logger.log('info', "Received keyboard interrupt")
        except Exception as e:
            self.logger.log('error', f"Application error: {e}")
        finally:
            self.shutdown()
            
    def signal_handler(self, signum, frame):
        """Handle shutdown signals"""
        global EXIT_APP
        EXIT_APP = True
        self.logger.log('info', f"Received signal {signum}, initiating shutdown...")
        
    def shutdown(self):
        """Graceful shutdown"""
        global EXIT_APP
        EXIT_APP = True
        
        if self.server:
            self.server.stop()
            
        self.logger.log('info', "Application shutdown complete")

def main():
    """Main entry point"""
    app = Application()
    app.run()

if __name__ == "__main__":
    main()