require "io"; --Default
require('uci'); --Default
local uv = require("luv"); --Install luv
local str_utils = require('str_utils'); --Package
local xml2lua = require("xml2lua"); --Package
local tree = require("xmlhandler.tree"); --Package
local biudp = require("biudp");
local json = require("json");
local dhcp = require("dhcp");
local syslog = require("syslog")

SOFTWARE_VERSION = "1.0.0";

BIND_ADDRESS = "127.0.0.1";
NODE_ADDRESS = "127.0.0.1";

AUDIO_NODE_PORT = 9231;
GPIO_NODE_PORT = 9232;
BASE_NODE_PORT = 9233;
INTERFACE_NODE_PORT = 9230;

PATH_REGID = "/root/flexa/regId";
COMMAND_PORT = 32512;

MAC_ADDRESS = "";
MODEL_NAME = ""

local clientIpAddresses = {};
local search_timer, http_worker, gpi_worker, tcpServer
local gpioEnabled, gpi, gpo = false, 0, 0
local baseNodeEnabled = false;
local audioPlayEnabled, audioCaptureEnabled = false, false
local audioFormat, audioInputs, audioOutputs = "", 0, 0
local audioTwoWay, audioOneWay = false, false
local argServerAddr
local regId

debug = false

-- Safe logging function that works with or without syslog
local function log_info(message)
    if syslog.is_enabled() then
        syslog.info(message)
    end
    if debug then
        io.write("[INFO] " .. message .. "\n")
        io.flush()
    end
end

local function log_error(message)
    if syslog.is_enabled() then
        syslog.error(message)
    end
    io.write("[ERROR] " .. message .. "\n")
    io.flush()
end

local function log_warning(message)
    if syslog.is_enabled() then
        syslog.warning(message)
    end
    if debug then
        io.write("[WARNING] " .. message .. "\n")
        io.flush()
    end
end

local function log_debug(message)
    if syslog.is_enabled() and debug then
        syslog.debug(message)
    end
    if debug then
        io.write("[DEBUG] " .. message .. "\n")
        io.flush()
    end
end

function loadConfig()
    local config = {
        enable_synapps = false,
        synapps_server_1 = "",
        registration_interval = 30,
        remote_syslog = {
            syslog_enabled = false,
            syslog_ip = "",
            syslog_port = 514
        }
    }
    
    local success, err = pcall(function()
        local configFile = io.open("config.json", "r")
        if not configFile then
            log_warning("config.json not found, creating default configuration")
            
            local defaultConfig = json.encode({
                AppParam = {
                    enable_synapps = true,
                    synapps_server_1 = "",
                    registration_interval = 30,
                    remote_syslog = {
                        syslog_enabled = false,
                        syslog_ip = "",
                        syslog_port = 514
                    }
                }
            })
            
            local newConfigFile = io.open("config.json", "w")
            if newConfigFile then
                newConfigFile:write(defaultConfig)
                newConfigFile:close()
                log_info("Created default config.json file")
            else
                error("Cannot create config.json file")
            end
            
            config.enable_synapps = true
            return config
        end
        
        local configContent = configFile:read("*all")
        configFile:close()
        
        if not configContent or configContent == "" then
            error("Config file is empty")
        end
        
        local parsedConfig = json.decode(configContent)
        if not parsedConfig or not parsedConfig.AppParam then
            error("Invalid config format - AppParam section missing")
        end
        
        local appParam = parsedConfig.AppParam
        
        -- Parse SynApps settings
        if type(appParam.enable_synapps) == "boolean" then
            config.enable_synapps = appParam.enable_synapps
        end
        
        if type(appParam.synapps_server_1) == "string" then
            config.synapps_server_1 = appParam.synapps_server_1
        end

        if type(appParam.registration_interval) == "number" then
            config.registration_interval = (appParam.registration_interval*1000)

        end
        
        -- Parse syslog settings
        if appParam.remote_syslog and type(appParam.remote_syslog) == "table" then
            local syslog_config = appParam.remote_syslog
            
            if type(syslog_config.syslog_enabled) == "boolean" then
                config.remote_syslog.syslog_enabled = syslog_config.syslog_enabled
            end
            
            if type(syslog_config.syslog_ip) == "string" then
                config.remote_syslog.syslog_ip = syslog_config.syslog_ip
            end
            
            if type(syslog_config.syslog_port) == "number" then
                config.remote_syslog.syslog_port = syslog_config.syslog_port
            end
        end
        
        return config
    end)
    
    if not success then
        log_error("Failed to load configuration: " .. tostring(err))
        log_info("Using default configuration")
        config.enable_synapps = true
    end
    
    -- Initialize syslog early
    syslog.init(config.remote_syslog)
    
    log_info("Configuration loaded - SynApps: " .. tostring(config.enable_synapps) .. 
             ", Server: " .. (config.synapps_server_1 ~= "" and config.synapps_server_1 or "auto") ..
             ", Syslog: " .. tostring(config.remote_syslog.syslog_enabled))
    
    return config
end

function getMac()
    local success, result = pcall(function()
        -- Method 1: Try qiba-spi-get-mac-address
        local f = io.popen("qiba-spi-get-mac-address")
        if f then
            local output = f:read("*all")
            f:close()
            
            if output then
                output = output:gsub("%s+$", "")
                local macAddr = output:match("ethaddr=(.+)")
                
                if macAddr and macAddr ~= "" then
                    MAC_ADDRESS = string.lower(macAddr)
                    log_info("MAC address obtained: " .. MAC_ADDRESS)
                    return true
                end
            end
        end
        
        -- Method 2: Fallback to UCI
        log_debug("Falling back to UCI method for MAC address")
        local x = uci.cursor()
        local uci_result = x:get("network", "wan_dev", "macaddr")
        
        if uci_result and uci_result ~= "" then
            MAC_ADDRESS = string.lower(uci_result)
            log_info("MAC address obtained via UCI: " .. MAC_ADDRESS)
            return true
        end
        
        return false
    end)
    
    if not success or not result then
        log_error("Could not obtain MAC address via any method")
        MAC_ADDRESS = "00:00:00:00:00:00"
        log_warning("Using fallback MAC address: " .. MAC_ADDRESS)
    end
end

function queueServersData(servers)
    local success, err = pcall(function()
        local enc = json.encode({
            servers = servers,
            gpioEnabled = gpioEnabled,
            gpi = gpi,
            gpo = gpo,
            baseNodeEnabled = baseNodeEnabled,
            audioInputs = audioInputs,
            audioOutputs = audioOutputs,
            audioFormat = audioFormat,
            audioOneWay = audioOneWay,
            audioTwoWay = audioTwoWay
        })
        
        local description = MODEL_NAME
        if regId then
            description = MODEL_NAME .. " " .. regId
        end
        
        http_worker:queue(enc, MAC_ADDRESS, description, nil, nil)
        log_debug("Queued server registration for " .. #servers .. " server(s)")
    end)
    
    if not success then
        log_error("Failed to queue server data: " .. tostring(err))
    end
end

function getStaticSynAppsServer()
    -- Get local interfaces
    local addresses = getIPAddress();

    -- Check which interface is the output for the given server
    local file = io.popen('ip route get '..argServerAddr);
    local route = file:read('*all');
    file:close();

    local arr = str_utils.split(route," ");
    
    local addrIdx = 5;
    if arr[2] == 'via' then
        addrIdx = 7;
    end

    for i=1,#addresses,1 do
        if addresses[i].addr == arr[addrIdx] then
            local servers = {};
            servers[1] = {
                server = argServerAddr,
                device = addresses[i].addr
            };
            queueServersData(servers);
            break;
        end
    end
end

function getSynAppsServers()
    if baseNodeEnabled~=true then
        return;
    end

    local addresses = getIPAddress();

    for i=1,#addresses,1 do
        dhcp.obtainOptions(addresses[i].mac, addresses[i].addr, {72}, function(result)
            if result==nil then
                return;
            end
            if result~=nil and result[72]~=nil then

                local servers = {};

                local opt72 = result[72];
                local ptr = 1;
                while #opt72>ptr do
                    local ipAddr = tostring(opt72:byte(ptr)).."."..tostring(opt72:byte(ptr+1)).."."..tostring(opt72:byte(ptr+2)).."."..tostring(opt72:byte(ptr+3));
                    ptr = ptr+4;
                    servers[#servers+1] = {
                        server = ipAddr,
                        device = addresses[i].addr
                    };
                end

                if #servers>0 then
                    queueServersData(servers);
                end
            end
        end, 1000, addresses[i].brd);
    end
end

function getIPAddress()
    local file = io.popen('ip -f inet addr');

    local output = file:read('*all');
    file:close();

    local arr = str_utils.split(output,"\r\n");

    local interfaceArr = {};

    local interface;

    for i=1,#arr,1 do
        if i%3==1 then
            interface = str_utils.split(arr[i],":")[2]:sub(2);
        end
        if i%3==2 then
            local lineArr = str_utils.split(arr[i]," ");
            if lineArr[3]=="brd" then --Only get those with broadcast address
                local inetAddr = str_utils.split(lineArr[2],"/")[1];
                local brdAddr = lineArr[4];
                interfaceArr[interface]={interface=str_utils.split(interface,"@")[1],addr=inetAddr,brd=brdAddr};
            end
        end
    end

    local returnArr = {};

    file = io.popen('ip -f link addr');

    output = file:read('*all');
    file:close();

    arr = str_utils.split(output,"\r\n");

    for i=1,#arr,1 do
        if i%2==1 then
            interface = str_utils.split(arr[i],":")[2]:sub(2);
        end
        if i%2==0 then
            local mac = str_utils.split(arr[i]," ")[2];
            if interfaceArr[interface]~=nil then
                interfaceArr[interface].mac = mac;
                returnArr[#returnArr+1] = interfaceArr[interface];
            end
        end
    end

    if debug then
        for i=1,#returnArr,1 do
            io.write(returnArr[i].interface.." "..returnArr[i].addr.."/"..returnArr[i].brd.." "..returnArr[i].mac,'\n');
            io.flush();
        end
    end

    return returnArr;
end

function sendGpiStateUpdate(gpinputid, state)
    if #clientIpAddresses == 0 then
        log_warning("No registered servers to send GPI update")
        return
    end
    
    for i = 1, #clientIpAddresses do
        local success, err = pcall(function()
            log_info("Sending GPI " .. gpinputid+1 .. " state " .. state .. " to " .. clientIpAddresses[i])
            gpi_worker:queue(clientIpAddresses[i], gpinputid, state, MAC_ADDRESS)
        end)
        
        if not success then
            log_error("Failed to send GPI update to " .. clientIpAddresses[i] .. ": " .. tostring(err))
        end
    end
end

function doGpiWork(address, gpinputid, state, mac)
    local success, result = pcall(function()
        local http = require("socket.http")
        local ltn12 = require("ltn12")
        local socket = require("socket")
        
        local function worker_log_debug(message)
            io.write("[WORKER-DEBUG] " .. message .. "\n")
            io.flush()
        end
        
        local function worker_log_error(message)
            io.write("[WORKER-ERROR] " .. message .. "\n")
            io.flush()
        end
        
        http.TIMEOUT = 5
        
        local url = "http://" .. address .. "/SA-Announce/PhoneServices/GPIOCallback.aspx?gpinputid=" .. 
                   tostring(gpinputid) .. "&state=" .. tostring(state) .. "&mac=" .. tostring(mac)
        
        worker_log_debug("GPIO Callback Request: " .. url)
        
        local response_body = {}
        local res, code = http.request{
            url = url,
            method = "GET",
            headers = { ["Content-Length"] = 0 },
            source = ltn12.source.string(''),
            sink = ltn12.sink.table(response_body),
        }
        
        -- Log the response
        local response_text = table.concat(response_body)
        worker_log_debug("GPIO Callback Response: HTTP " .. tostring(code) .. 
                        " Body: " .. (response_text ~= "" and response_text or "(empty)"))
        
        if code and code >= 200 and code < 300 then
            return true
        else
            worker_log_error("GPIO Callback failed: HTTP " .. tostring(code))
            return false
        end
    end)
    
    if not success then
        io.write("[WORKER-ERROR] GPIO Callback exception: " .. tostring(result) .. "\n")
        io.flush()
        return false
    end
    
    return result
end

function getModel()
    local success, model = pcall(function()
        -- Method 1: Try /proc/cpuinfo
        for line in io.lines("/proc/cpuinfo") do
            local extracted_model, replacements = string.gsub(line, "machine            : Barix ", "")
            if replacements == 1 then
                log_debug("Model obtained from system cmd: " .. extracted_model)
                return extracted_model
            end
        end
        
        -- Method 2: Try qiba-spi-get-info
        log_debug("Trying qiba utility for model detection")
        local f = io.popen("qiba-spi-get-info")
        if f then
            local output = f:read("*all")
            f:close()
            
            if output and output ~= "" then
                output = output:gsub("%s+$", "")
                local parsedInfo = json.decode(output)
                
                if parsedInfo and parsedInfo.HW_DEVICE and 
                   parsedInfo.HW_DEVICE.Product_Name and
                   parsedInfo.HW_DEVICE.Product_Name ~= "" then
                    
                    local product_name = parsedInfo.HW_DEVICE.Product_Name:gsub("_", " ")
                    log_debug("Model obtained from qiba utility: " .. product_name)
                    return product_name
                end
            end
        end
        
        return "Model N/A"
    end)
    
    if not success then
        log_error("Error detecting model: " .. tostring(model))
        return "Model N/A"
    end
    
    return model or "Model N/A"
end

function doWork(data, mac, description, overrideAddress, self_address)
    local success, result = pcall(function()
        local http = require("socket.http")
        local url = require("socket.url")
        local ltn12 = require("ltn12")
        local str_utils = require('str_utils')
        local json = require('json')
        local socket = require("socket")
        
        local function worker_log_info(message)
            io.write("[WORKER-DEBUG] " .. message .. "\n")
            io.flush()
        end
        
        local function worker_log_error(message)
            io.write("[WORKER-ERROR] " .. message .. "\n")
            io.flush()
        end
        
        local function getFirmwareVersion()
            local version = ""
            local f = io.open('/barix/info/VERSION', 'r')
            if f then
                version = f:read('*all')
                f:close()
                -- Clean up version string
                if version then
                    version = version:gsub("%s+", "")
                end
            end
            return version ~= "" and version or "unknown"
        end

        local function getModel()
            local success, model = pcall(function()
                -- Method 1: Try /proc/cpuinfo
                for line in io.lines("/proc/cpuinfo") do
                    local extracted_model, replacements = string.gsub(line, "machine            : Barix ", "")
                    if replacements == 1 then
                        return extracted_model
                    end
                end
                
                -- Method 2: Try qiba-spi-get-info
                local f = io.popen("qiba-spi-get-info")
                if f then
                    local output = f:read("*all")
                    f:close()
                    
                    if output and output ~= "" then
                        output = output:gsub("%s+$", "")
                        local parsedInfo = json.decode(output)
                        
                        if parsedInfo and parsedInfo.HW_DEVICE and 
                           parsedInfo.HW_DEVICE.Product_Name and
                           parsedInfo.HW_DEVICE.Product_Name ~= "" then
                            
                            local product_name = parsedInfo.HW_DEVICE.Product_Name:gsub("_", " ")
                            return product_name
                        end
                    end
                end
                
                return "Model N/A"
            end)
            
            return success and model or "Model N/A"
        end

        http.TIMEOUT = 10

        local obj = json.decode(data)
        if not obj or not obj.servers then
            error("Invalid work data - missing servers")
        end

        local MANUFACTURER = "Barix"
        local MODEL = getModel()
        local FIRMWARE = regId and ("Flexa SynApps v" .. SOFTWARE_VERSION) or ("FW v" .. getFirmwareVersion())
        local API_VERSION = "1.1"
        local CLOCK_ATTACHED = true
        local SUPPORT_IMAGE = false
        local SUPPORT_TEXT = false
        local SUPPORT_STROBE = false
        local SUPPORT_XML_COMMANDS = true
        local COMMAND_PORT = 32512
        local PROTOCOL = "tcp"

        -- Check if model contains "Barionet" and override audio capabilities
        local reportedAudioOneWay = obj.audioOneWay
        local reportedAudioTwoWay = obj.audioTwoWay
        local reportedAudioInputs = obj.audioInputs
        local reportedAudioOutputs = obj.audioOutputs
        
        if MODEL and string.find(string.lower(MODEL), "barionet") then
            reportedAudioOneWay = false
            reportedAudioTwoWay = false
            reportedAudioInputs = 0
            reportedAudioOutputs = 0
        end

        local function sendPost(address, deviceAddr, mac, description)
            local reqBody = "Manufacturer=" .. url.escape(MANUFACTURER) ..
                           "&Model=" .. url.escape(MODEL) ..
                           "&Description=" .. url.escape(description) ..
                           "&IPAddress=" .. url.escape(deviceAddr) ..
                           "&MAC=" .. url.escape(mac) ..
                           "&Firmware=" .. url.escape(FIRMWARE) ..
                           "&APIVersion=1.1" ..
                           "&ClockAttached=" .. url.escape(tostring(CLOCK_ATTACHED)) ..
                           "&SupportGPI=" .. url.escape(tostring(obj.gpi)) ..
                           "&SupportGPO=" .. url.escape(tostring(obj.gpo)) ..
                           "&SupportImage=" .. url.escape(tostring(SUPPORT_IMAGE)) ..
                           "&SupportText=" .. url.escape(tostring(SUPPORT_TEXT)) ..
                           "&SupportStrobe=" .. url.escape(tostring(SUPPORT_STROBE)) ..
                           "&SupportXMLCommands=" .. url.escape(tostring(SUPPORT_XML_COMMANDS)) ..
                           "&CommandPort=" .. url.escape(tostring(COMMAND_PORT)) ..
                           "&Protocol=" .. url.escape(PROTOCOL) ..
                           "&Status=OK" ..
                           "&ErrorCode=0" ..
                           "&SupportOneWay=" .. url.escape(tostring(reportedAudioOneWay)) ..
                           "&SupportTwoWay=" .. url.escape(tostring(reportedAudioTwoWay)) ..
                           "&AudioInputs=" .. url.escape(tostring(reportedAudioInputs)) ..
                           "&AudioOutputs=" .. url.escape(tostring(reportedAudioOutputs)) ..
                           "&SupportCodecs=" .. url.escape(obj.audioFormat or "")

            local response_body = {}
            local useUrl = "http://" .. address .. "/sa-announce/api/saannounceapi.asmx/RegisterIpDevice"
            worker_log_info("Registration request: "..useUrl)

            local res, code, response_headers = http.request{
                url = useUrl,
                method = "POST", 
                headers = {
                    ["Content-Type"] = "application/x-www-form-urlencoded",
                    ["Content-Length"] = #reqBody,
                    ["User-Agent"] = MODEL
                },
                source = ltn12.source.string(reqBody),
                sink = ltn12.sink.table(response_body)
            }
            worker_log_info("Code Response Received: "..code)

            return code
        end
        
        -- Send HTTP requests to all servers
        local registeredServers = {}
        local failedServers = {}
        
        worker_log_info("Starting server registration with " .. #obj.servers .. " server(s)")
        
        for i = 1, #obj.servers do
            local server = obj.servers[i]
            local reg_success, code = pcall(sendPost, server.server, server.device, mac, description)
            
            if reg_success and code and code >= 200 and code < 300 then
                table.insert(registeredServers, server.server)
                worker_log_info("Successfully registered with server " .. server.server .. " (HTTP " .. code .. ")")
            else
                local error_msg = "Failed to register with server " .. server.server
                if reg_success and code then
                    error_msg = error_msg .. " (HTTP " .. code .. ")"
                elseif not reg_success then
                    error_msg = error_msg .. " (Connection error: " .. tostring(code) .. ")"
                else
                    error_msg = error_msg .. " (Unknown error)"
                end
                
                table.insert(failedServers, server.server)
                worker_log_error(error_msg)
            end
        end
        
        if #registeredServers > 0 then
            if #failedServers > 0 then
                worker_log_info("Registration completed - Success: " .. #registeredServers .. ", Failed: " .. #failedServers)
            end
            return table.concat(registeredServers, ',')
        else
            worker_log_error("Registration failed - No servers reachable (" .. #failedServers .. " attempted)")
            return nil
        end
    end)
    
    if not success then
        return nil
    end
    return result
end

function executeCommand(cmd, oncomplete, self_address)
    if debug then
        log_debug(cmd,'\n');
    end

    if cmd=="Test" then
        --Just respond
        biudp.sendUdpCommand({
            command_name="test"
        }, BASE_NODE_PORT, oncomplete);
    end
    if cmd=="Status" then
        biudp.sendUdpCommand({
            command_name="status"
        }, BASE_NODE_PORT, oncomplete);
    end
    if cmd=="Reset" then
        biudp.sendUdpCommand({
            command_name="reset"
        }, BASE_NODE_PORT, oncomplete);
    end

    if str_utils.starts_with(cmd,"GPO:") then
        local arr = str_utils.split(cmd,":");
        if arr[2]~=nil and arr[3]~=nil and arr[4]~=nil then
            local gpioId = tonumber(arr[2]);
            local state = tonumber(arr[3]);
            local duration = tonumber(arr[4]);

            biudp.sendUdpCommand({
                command_name="setoutput",
                pin=gpioId,
                value=state,
                duration=duration
            }, GPIO_NODE_PORT, oncomplete);
        end
    end

    --RTPRx://x.x.x.x:pppp:vvv:P
    if str_utils.starts_with(cmd, "RTPRx://") then
        local str = cmd:sub((#"RTPRx://")+1);
        local arr = str_utils.split(str, ":");
        if #arr>2 then
            local addr = arr[1];
            local portStr = arr[2];
            local port = tonumber(portStr);
            local volumeStr = arr[3];
            if volumeStr=="Stop" then
                biudp.sendUdpCommand({
                    command_name="stop",
                    address=addr,
                    port=port
                }, AUDIO_NODE_PORT, function(resp)
                    if resp==nil then
                        oncomplete(nil);
                    elseif resp.error~=nil and resp.code~=nil then
                        oncomplete(nil);
                    else
                        oncomplete(resp);
                    end
                end);
            else
                local volume = tonumber(volumeStr);
                local priorityStr = arr[4];
                local priority = tonumber(priorityStr);
                if not str_utils.str_contains(audioFormat, "G711") then
                    oncomplete(nil);
                    return;
                end
                biudp.sendUdpCommand({
                    command_name="play",
                    address=addr,
                    port=port,
                    audioFormat="ULAW_8KHZ",
                    volume=volume/10,
                    self_address=self_address
                }, AUDIO_NODE_PORT, function(resp)
                    if resp==nil then
                        oncomplete(nil);
                    elseif resp.error~=nil and resp.code~=nil then
                        oncomplete(nil);
                    else
                        oncomplete(resp);
                    end
                end);
            end

        end
    end

    --RTPMRx://x.x.x.x:pppp:vvv:P
    if str_utils.starts_with(cmd, "RTPMRx://") then
        local str = cmd:sub((#"RTPMRx://")+1);
        local arr = str_utils.split(str, ":");
        if #arr>2 then
            local addr = arr[1];
            local portStr = arr[2];
            local port = tonumber(portStr);
            local volumeStr = arr[3];
            if volumeStr=="Stop" then
                biudp.sendUdpCommand({
                    command_name="stop",
                    address=addr,
                    port=port
                }, AUDIO_NODE_PORT, function(resp)
                    if resp==nil then
                        oncomplete(nil);
                    elseif resp.error~=nil and resp.code~=nil then
                        oncomplete(nil);
                    else
                        oncomplete(resp);
                    end
                end);
            else
                local volume = tonumber(volumeStr);
                local priorityStr = arr[4];
                local priority = tonumber(priorityStr);
                if not str_utils.str_contains(audioFormat, "G711") then
                    oncomplete(nil);
                    return;
                end
                biudp.sendUdpCommand({
                    command_name="play",
                    address=addr,
                    port=port,
                    audioFormat="ULAW_8KHZ",
                    volume=volume/10,
                    self_address=self_address
                }, AUDIO_NODE_PORT, function(resp)
                    if resp==nil then
                        oncomplete(nil);
                    elseif resp.error~=nil and resp.code~=nil then
                        oncomplete(nil);
                    else
                        oncomplete(resp);
                    end
                end);
            end

        end
    end
end

function initWorkers()
    http_worker = uv.new_work(doWork, function(returnStr)
        if returnStr then
            clientIpAddresses = str_utils.split(returnStr, ',')
            log_info("Registered with " .. returnStr)
        end
    end)
    
    gpi_worker = uv.new_work(doGpiWork, function() end)
end

function initTcpServer()
    local success, err = pcall(function()
        tcpServer = uv.new_tcp()
        tcpServer:bind("0.0.0.0", COMMAND_PORT)
        
        tcpServer:listen(128, function(conn_err)
            if conn_err then
                log_error("TCP connection error: " .. tostring(conn_err))
                return
            end
            
            local client = uv.new_tcp()
            tcpServer:accept(client)
            
            client:read_start(function(read_err, chunk)
                if read_err then
                    log_error("TCP read error: " .. tostring(read_err))
                    pcall(function()
                        client:shutdown()
                        client:close()
                    end)
                    return
                end
                
                if chunk then
                    local cmd_success, cmd_err = pcall(function()
                        -- Repair GPO chunk
                        if str_utils.str_contains(chunk, "GPO:") then
                            chunk = chunk:gsub("% \">", " \"/>")
                        end

                        log_debug("Received XML command: " .. chunk:sub(1, 200) .. (chunk:len() > 200 and "..." or ""))

                        local xmlHandler = tree:new()
                        local xmlParser = xml2lua.parser(xmlHandler)
                        xmlParser:parse(chunk)

                        if xmlHandler and xmlHandler.root and 
                           xmlHandler.root.IPDeviceExecute and 
                           xmlHandler.root.IPDeviceExecute.ExecuteItem then

                            if xmlHandler.root.IPDeviceExecute.ExecuteItem[1] then
                                -- Array of commands
                                local responses = {}
                                local totalCommands = #xmlHandler.root.IPDeviceExecute.ExecuteItem
                                log_debug("Processing " .. totalCommands .. " XML commands")
                                
                                for i = 1, totalCommands do
                                    if xmlHandler.root.IPDeviceExecute.ExecuteItem[i]._attr and
                                       xmlHandler.root.IPDeviceExecute.ExecuteItem[i]._attr.URL then

                                        executeCommand(xmlHandler.root.IPDeviceExecute.ExecuteItem[i]._attr.URL, 
                                                     function(obj, customBody)
                                            if customBody then
                                                responses[i] = customBody
                                            elseif not obj then
                                                responses[i] = "<ExecuteItem Result=\"MALFORMED REQUEST\" />\n"
                                                log_debug("Command " .. i .. " failed - malformed request")
                                            else
                                                responses[i] = "<ExecuteItem Result=\"SUCCESS\" />\n"
                                                log_debug("Command " .. i .. " completed successfully")
                                            end

                                            -- Check if all commands are complete
                                            local allComplete = true
                                            for e = 1, totalCommands do
                                                if not responses[e] then
                                                    allComplete = false
                                                    break
                                                end
                                            end

                                            if allComplete then
                                                local xmlResponse = "<IPDeviceExecute>\n"
                                                for e = 1, totalCommands do
                                                    xmlResponse = xmlResponse .. responses[e]
                                                end
                                                xmlResponse = xmlResponse .. "</IPDeviceExecute>\n"
                                                
                                                local write_success, write_err = pcall(function()
                                                    client:write(xmlResponse)
                                                end)
                                                
                                                if not write_success then
                                                    log_error("Failed to send XML response: " .. tostring(write_err))
                                                else
                                                    log_debug("XML response sent for " .. totalCommands .. " commands")
                                                end
                                            end
                                        end, client:getsockname().ip)
                                    end
                                end
                            else
                                -- Single command
                                if xmlHandler.root.IPDeviceExecute.ExecuteItem._attr and
                                   xmlHandler.root.IPDeviceExecute.ExecuteItem._attr.URL then

                                    log_debug("Processing single XML command")
                                    executeCommand(xmlHandler.root.IPDeviceExecute.ExecuteItem._attr.URL, 
                                                 function(obj)
                                        local response
                                        if not obj then
                                            response = "<IPDeviceExecute><ExecuteItem Result=\"MALFORMED REQUEST\"/></IPDeviceExecute>\n"
                                            log_debug("Single command failed - malformed request")
                                        else
                                            response = "<IPDeviceExecute><ExecuteItem Result=\"SUCCESS\"/></IPDeviceExecute>\n"
                                            log_debug("Single command completed successfully")
                                        end
                                        
                                        local write_success, write_err = pcall(function()
                                            client:write(response)
                                        end)
                                        
                                        if not write_success then
                                            log_error("Failed to send XML response: " .. tostring(write_err))
                                        else
                                            log_debug("XML response sent for single command")
                                        end
                                    end, client:getsockname().ip)
                                end
                            end
                        else
                            log_warning("Invalid XML structure received")
                            -- Send error response for invalid XML
                            local error_response = "<IPDeviceExecute><ExecuteItem Result=\"INVALID XML\"/></IPDeviceExecute>\n"
                            pcall(function()
                                client:write(error_response)
                            end)
                        end
                    end)
                    
                    if not cmd_success then
                        log_error("XML command processing failed: " .. tostring(cmd_err))
                        -- Send error response for processing failure
                        local error_response = "<IPDeviceExecute><ExecuteItem Result=\"PROCESSING ERROR\"/></IPDeviceExecute>\n"
                        pcall(function()
                            client:write(error_response)
                        end)
                    end
                end
            end)
        end)
        
        log_info("TCP command server listening on port " .. COMMAND_PORT)
    end)
    
    if not success then
        log_error("Failed to initialize TCP server: " .. tostring(err))
        error("Cannot start TCP command server")
    end
end

function processUdpCommand(obj)
    if not obj or not obj.command_name then
        log_error("Invalid UDP command received")
        return nil
    end
    
    local cmd = obj.command_name
    log_debug("Processing UDP command: " .. cmd)
    
    if cmd == "gpistatechange" and obj.pin and obj.value ~= nil then
        sendGpiStateUpdate(obj.pin, obj.value)
        return { command_name = "gpistatechange" }
    end
    
    log_warning("Unknown UDP command: " .. cmd)
    return nil
end

function checkNodes(callback)
    local done = 0
    local function checkAllDone()
        done = done + 1
        if done == 3 and callback then
            callback()
        end
    end
    
    -- Check GPIO Node
    biudp.sendUdpCommand({
        command_name = "getgpiopins"
    }, GPIO_NODE_PORT, function(obj)
        if obj then
            gpioEnabled = true
            gpi = obj.inputs or 0
            gpo = obj.outputs or 0
            log_debug("GPIO node connected - Inputs: " .. gpi .. ", Outputs: " .. gpo)
        else
            gpioEnabled = false
            gpi, gpo = 0, 0
            log_warning("GPIO node unavailable")
        end
        checkAllDone()
    end)
    
    -- Check Base Node
    biudp.sendUdpCommand({
        command_name = "alive"
    }, BASE_NODE_PORT, function(obj)
        if obj then
            baseNodeEnabled = true
            log_debug("Base node connected")
        else
            baseNodeEnabled = false
            log_warning("Base node unavailable")
        end
        checkAllDone()
    end)
    
    -- Check Audio Node
    biudp.sendUdpCommand({
        command_name = "getaudioinfo"
    }, AUDIO_NODE_PORT, function(obj)
        if obj then
            audioPlayEnabled = obj.play or false
            audioCaptureEnabled = obj.capture or false
            audioOutputs = obj.outputs or 0
            audioInputs = obj.inputs or 0
            audioFormat = str_utils.str_contains(obj.formats or "", "ULAW_8KHZ") and "G711" or ""
            audioTwoWay = obj.twoWay or false
            audioOneWay = obj.oneWay or false
            log_debug("Audio node connected - Formats: " .. (obj.formats or "none"))
        else
            audioPlayEnabled, audioCaptureEnabled = false, false
            audioOutputs, audioInputs = 0, 0
            audioFormat = ""
            audioTwoWay, audioOneWay = false, false
            log_warning("Audio node unavailable")
        end
        checkAllDone()
    end)
end

function runNodes()
    local success, err = pcall(function()
        
        uv.new_thread(function()
            require("gpioNode")
        end)
        
        uv.new_thread(function()
            require("baseNode")
        end)
        
        uv.new_thread(function()
            require("audioNode")
        end)
        
        log_info("All node threads started")
    end)
    
    if not success then
        log_error("Failed to start node threads: " .. tostring(err))
        error("Critical error: Cannot start required services")
    end
end

function main()
    log_info("Starting SynApps application v" .. SOFTWARE_VERSION)
    
    -- Load configuration
    local config = loadConfig()
    
    if not config.enable_synapps then
        log_info("SynApps disabled in configuration - exiting")
        os.exit(0)
    end
    
    -- Set server address
    if config.synapps_server_1 ~= "" then
        argServerAddr = config.synapps_server_1
        log_info("Using static SynApps server: " .. argServerAddr)
    else
        log_info("Using automatic SynApps server discovery")
    end
    
    -- Initialize system
    local success, err = pcall(function()
        runNodes()
        getMac()
        MODEL_NAME = getModel()
        
        -- Read registration ID
        local f = io.open(PATH_REGID, 'r')
        if f then
            regId = f:read("*all")
            f:close()
            if regId then
                regId = regId:gsub("%s+", "") -- trim whitespace
                log_info("Registration ID loaded: " .. regId)
            end
        end
        
        log_info("System initialized - Model: " .. MODEL_NAME .. ", MAC: " .. MAC_ADDRESS)
    end)
    
    if not success then
        log_error("System initialization failed: " .. tostring(err))
        error("Critical initialization error")
    end
    
    -- Initialize services
    pcall(function()
        initWorkers()
        initTcpServer()
        biudp.initUdpServer(processUdpCommand, uv, INTERFACE_NODE_PORT, BIND_ADDRESS, NODE_ADDRESS)
        
        -- Start periodic node checking and server discovery
        search_timer = uv.new_timer()
        search_timer:start(0, config.registration_interval, function()
            local fnGetServers = argServerAddr and getStaticSynAppsServer or getSynAppsServers
            checkNodes(fnGetServers)
        end)
        
        log_info("All services initialized successfully")
    end)
end

-- Error handler for the main event loop
local function errorHandler(err)
    log_error("Unhandled error in main loop: " .. tostring(err))
    if syslog.is_enabled() then
        syslog.critical("Application error: " .. tostring(err))
    end
end

-- Start application with error handling
local success, err = pcall(main)
if not success then
    log_error("Application startup failed: " .. tostring(err))
    if syslog.is_enabled() then
        syslog.critical("Application startup failed: " .. tostring(err))
    end
    os.exit(1)
end

log_info("Application started successfully")

-- Main event loop with error handling
while true do
    local loop_success, loop_err = pcall(uv.run)
    if not loop_success then
        errorHandler(loop_err)
        -- Brief pause before continuing
        os.execute("sleep 0.1")
    end
end