local uv = require("luv")
local biudp = require("biudp")
local syslog = require("syslog")

GPIO_NODE_PORT = 9232
INTERFACE_NODE_PORT = 9230

PATH_REL = "/dev/gpio/rel"
PATH_OUT = "/dev/gpio/out"
PATH_IN = "/dev/gpio/in"  -- Fixed typo

GPI_CHECK_TIME = 20 -- 20ms

RELAYS = 0
INPUTS = 0
OUTPUTS = 0

debug = false

local relayArr = {}
local inputArr = {}
local outputArr = {}
local gpiLastState

-- Logging functions
local function log_info(message)
    if syslog.is_enabled() then
        syslog.info("GPIO: " .. message)
    end
    if debug then
        io.write("[GPIO-INFO] " .. message .. "\n")
        io.flush()
    end
end

local function log_error(message)
    if syslog.is_enabled() then
        syslog.error("GPIO: " .. message)
    end
    io.write("[GPIO-ERROR] " .. message .. "\n")
    io.flush()
end

local function log_warning(message)
    if syslog.is_enabled() then
        syslog.warning("GPIO: " .. message)
    end
    if debug then
        io.write("[GPIO-WARNING] " .. message .. "\n")
        io.flush()
    end
end

local function log_debug(message)
    if syslog.is_enabled() then
        syslog.debug("GPIO: " .. message)
    end
    if debug then
        io.write("[GPIO-DEBUG] " .. message .. "\n")
        io.flush()
    end
end

function getGPIO()
    local success, err = pcall(function()
        local relays, inputs, outputs = 0, 0, 0
        
        -- Initialize arrays
        relayArr = {}
        inputArr = {}
        outputArr = {}
        
        -- Discover relays
        repeat
            local file = io.open(PATH_REL .. tostring(relays + 1) .. "/value", 'r')
            if file then
                relayArr[#relayArr + 1] = file
                relays = relays + 1
            end
        until(file == nil)
        
        -- Discover inputs
        repeat
            local file = io.open(PATH_IN .. tostring(inputs + 1) .. "/value", 'r')
            if file then
                inputArr[#inputArr + 1] = file
                inputs = inputs + 1
            end
        until(file == nil)
        
        -- Discover outputs
        repeat
            local file = io.open(PATH_OUT .. tostring(outputs + 1) .. "/value", 'r')
            if file then
                outputArr[#outputArr + 1] = file
                outputs = outputs + 1
            end
        until(file == nil)
        
        RELAYS = relays
        OUTPUTS = outputs
        INPUTS = inputs
        
        log_info("GPIO initialized - Relays: " .. RELAYS .. ", Inputs: " .. INPUTS .. ", Outputs: " .. OUTPUTS)
        
        if INPUTS == 0 and RELAYS == 0 and OUTPUTS == 0 then
            log_warning("No GPIO devices found")
        end
        
        return true
    end)
    
    if not success then
        log_error("GPIO initialization failed: " .. tostring(err))
        RELAYS, INPUTS, OUTPUTS = 0, 0, 0
    end
end

function sendGpiStateUpdate(gpinputid, state)
    local success, err = pcall(function()
        log_info("GPI state change - Pin " .. gpinputid .. " = " .. tostring(state))
        
        biudp.sendUdpCommand({
            command_name = "gpistatechange",
            pin = gpinputid,
            value = state
        }, INTERFACE_NODE_PORT)
    end)
    
    if not success then
        log_error("Failed to send GPI state update: " .. tostring(err))
    end
end

function checkGpi()
    if INPUTS == 0 then
        return -- No inputs to check
    end
    
    local success, err = pcall(function()
        if not gpiLastState then
            -- Initialize state tracking
            gpiLastState = {}
            log_info("Initializing GPI monitoring for " .. INPUTS .. " inputs")
            
            for i = 1, #inputArr do
                inputArr[i]:seek('set', 0)
                inputArr[i]:flush()
                local initialVal = inputArr[i]:read('*n')
                gpiLastState[i] = initialVal
                log_debug("GPI pin " .. (i-1) .. " initial state: " .. tostring(initialVal))
            end
            
            log_info("GPI monitoring active")
        else
            -- Check for state changes
            for i = 1, #inputArr do
                inputArr[i]:seek('set', 0)
                inputArr[i]:flush()
                
                local currentVal = inputArr[i]:read('*n')
                inputArr[i]:seek('set', 0)
                local confirmVal = inputArr[i]:read('*n')
                currentVal = confirmVal
                
                if currentVal ~= gpiLastState[i] then
                    log_info("GPI state change detected - Pin " .. (i-1) .. ": " .. 
                            tostring(gpiLastState[i]) .. " -> " .. tostring(currentVal))
                    
                    sendGpiStateUpdate(i-1, currentVal)
                end
                
                gpiLastState[i] = currentVal
            end
        end
    end)
    
    if not success then
        log_error("GPI check failed: " .. tostring(err))
    end
end

function processUdpCommand(obj)
    if not obj or not obj.command_name then
        log_error("Invalid UDP command received")
        return nil
    end
    
    local cmd = obj.command_name
    log_debug("Processing command: " .. cmd)
    
    if cmd == "getgpiopins" then
        return {
            command_name = "getgpiopins",
            inputs = INPUTS,
            outputs = OUTPUTS + RELAYS
        }
    end
    
    if cmd == "setoutput" then
        local gpioId = obj.pin
        local state = obj.value
        local duration = obj.duration
        
        if not gpioId or state == nil then
            log_error("Invalid setoutput command - missing pin or value")
            return {
                command_name = "setoutput",
                error = "Invalid parameters",
                code = 1
            }
        end
        
        local success, err = pcall(function()
            -- Handle timed operations
            if duration and duration > 0 then
                local initialState = nil
                
                if gpioId < RELAYS then
                    -- Relays
                    relayArr[gpioId + 1]:seek('set', 0)
                    initialState = relayArr[gpioId + 1]:read('*n')
                    
                    local pinTimer = uv.new_timer()
                    pinTimer:start(duration, 0, function()
                        pinTimer:stop()
                        pinTimer:close()
                        
                        local restore_success, restore_err = pcall(function()
                            os.execute("echo " .. tostring(initialState) .. 
                                     " > /dev/gpio/rel" .. tostring(gpioId + 1) .. "/value")
                        end)
                        
                        if not restore_success then
                            log_error("Failed to restore relay " .. gpioId .. ": " .. tostring(restore_err))
                        else
                            log_debug("Restored relay " .. gpioId .. " to " .. tostring(initialState))
                        end
                    end)
                else
                    -- Outputs
                    local outputIndex = gpioId - RELAYS + 1
                    if outputIndex <= #outputArr then
                        outputArr[outputIndex]:seek('set', 0)
                        initialState = outputArr[outputIndex]:read('*n')
                        
                        local pinTimer = uv.new_timer()
                        pinTimer:start(duration, 0, function()
                            pinTimer:stop()
                            pinTimer:close()
                            
                            local restore_success, restore_err = pcall(function()
                                os.execute("echo " .. tostring(initialState) .. 
                                         " > /dev/gpio/out" .. tostring(outputIndex) .. "/value")
                            end)
                            
                            if not restore_success then
                                log_error("Failed to restore output " .. gpioId .. ": " .. tostring(restore_err))
                            else
                                log_debug("Restored output " .. gpioId .. " to " .. tostring(initialState))
                            end
                        end)
                    end
                end
            end
            
            -- Set current state
            if gpioId < RELAYS then
                -- Relays
                log_info("Setting relay " .. gpioId .. " to " .. tostring(state))
                os.execute("echo " .. tostring(state) .. " > /dev/gpio/rel" .. tostring(gpioId + 1) .. "/value")
            else
                -- Outputs
                local outputIndex = gpioId - RELAYS + 1
                if outputIndex <= OUTPUTS then
                    log_info("Setting output " .. gpioId .. " to " .. tostring(state))
                    os.execute("echo " .. tostring(state) .. " > /dev/gpio/out" .. tostring(outputIndex) .. "/value")
                else
                    error("Invalid output pin: " .. gpioId)
                end
            end
        end)
        
        if not success then
            log_error("Failed to set output " .. gpioId .. ": " .. tostring(err))
            return {
                command_name = "setoutput",
                error = "Operation failed",
                code = 2
            }
        end
        
        return { command_name = "setoutput" }
    end
    
    if cmd == "getinputvalues" then
        return {
            command_name = "getinputvalues",
            values = gpiLastState or {}
        }
    end
    
    log_warning("Unknown command: " .. cmd)
    return nil
end

function main()
    log_info("Starting GPIO Node")
    
    local success, err = pcall(function()
        getGPIO()
        
        if INPUTS > 0 then
            local gpi_timer = uv.new_timer()
            gpi_timer:start(GPI_CHECK_TIME, GPI_CHECK_TIME, checkGpi)
            log_info("GPI monitoring started - checking every " .. GPI_CHECK_TIME .. "ms")
        else
            log_info("No GPIO inputs found - monitoring disabled")
        end
        
        biudp.initUdpServer(processUdpCommand, uv, GPIO_NODE_PORT, "127.0.0.1", "127.0.0.1")
        log_info("GPIO Node initialized on port " .. GPIO_NODE_PORT)
    end)
    
    if not success then
        log_error("GPIO Node startup failed: " .. tostring(err))
        error("Critical GPIO Node error")
    end
end

-- Error handler for the main event loop
local function errorHandler(err)
    log_error("Unhandled error in GPIO Node: " .. tostring(err))
    if syslog.is_enabled() then
        syslog.critical("GPIO Node error: " .. tostring(err))
    end
end

-- Start GPIO Node with error handling
local success, err = pcall(main)
if not success then
    log_error("GPIO Node startup failed: " .. tostring(err))
    os.exit(1)
end

-- Main event loop with error handling
while true do
    local loop_success, loop_err = pcall(uv.run)
    if not loop_success then
        errorHandler(loop_err)
        -- Brief pause before continuing
        os.execute("sleep 0.1")
    end
end