-- syslog.lua - Remote Syslog Module
local socket = require("socket")

local syslog = {}

-- Syslog severity levels (RFC 3164)
syslog.EMERGENCY = 0    -- System is unusable
syslog.ALERT = 1        -- Action must be taken immediately  
syslog.CRITICAL = 2     -- Critical conditions
syslog.ERROR = 3        -- Error conditions
syslog.WARNING = 4      -- Warning conditions
syslog.NOTICE = 5       -- Normal but significant condition
syslog.INFO = 6         -- Informational messages
syslog.DEBUG = 7        -- Debug-level messages

-- Syslog facilities (RFC 3164) 
local FACILITY_LOCAL0 = 16 * 8  -- Local use facility 0

-- Configuration
local config = {
    enabled = false,
    server_ip = nil,
    server_port = 514,
    facility = FACILITY_LOCAL0,
    hostname = "barionet",
    app_name = "synapps"
}

-- Initialize syslog with configuration
function syslog.init(syslog_config)
    if syslog_config and type(syslog_config) == "table" then
        config.enabled = syslog_config.syslog_enabled or false
        config.server_ip = syslog_config.syslog_ip
        config.server_port = syslog_config.syslog_port or 514
        
        -- Get hostname if possible
        local hostname_file = io.popen("hostname")
        if hostname_file then
            local hostname = hostname_file:read("*line")
            hostname_file:close()
            if hostname and hostname ~= "" then
                config.hostname = hostname
            end
        end
        
        if config.enabled and config.server_ip and config.server_ip ~= "" then
            syslog.info("Remote syslog initialized - server: " .. config.server_ip .. ":" .. config.server_port)
            return true
        end
    end
    return false
end

-- Format syslog message according to RFC 3164
local function format_message(severity, message)
    local priority = config.facility + severity
    local timestamp = os.date("%b %d %H:%M:%S")
    return "<" .. priority .. ">" .. timestamp .. " " .. config.hostname .. " " .. config.app_name .. ": " .. message
end

-- Send syslog message
local function send_message(severity, message)
    if not config.enabled or not config.server_ip or config.server_ip == "" then
        return false
    end
    
    local success, err = pcall(function()
        local udp = socket.udp()
        if udp then
            udp:settimeout(1) -- 1 second timeout
            local formatted_msg = format_message(severity, message)
            udp:sendto(formatted_msg, config.server_ip, config.server_port)
            udp:close()
        end
    end)
    
    return success
end

-- Public logging functions
function syslog.emergency(message)
    send_message(syslog.EMERGENCY, message)
end

function syslog.alert(message)
    send_message(syslog.ALERT, message)
end

function syslog.critical(message)
    send_message(syslog.CRITICAL, message)
end

function syslog.error(message)
    send_message(syslog.ERROR, message)
end

function syslog.warning(message)
    send_message(syslog.WARNING, message)
end

function syslog.notice(message)
    send_message(syslog.NOTICE, message)
end

function syslog.info(message)
    send_message(syslog.INFO, message)
end

function syslog.debug(message)
    send_message(syslog.DEBUG, message)
end

-- Check if syslog is enabled
function syslog.is_enabled()
    return config.enabled and config.server_ip and config.server_ip ~= ""
end

return syslog