require "io";
local uv = require("luv"); --Install luv
local biudp = require("biudp");
local str_utils = require("str_utils");

AUDIO_NODE_PORT = 9231;
INTERFACE_NODE_PORT = 9230;

--[[

IMPORTANT NOTICE: SUPPORTS ONE USB SOUND CARD ONLY!!!

Commands:
	Listening to (9231<-):
		getaudioinfo:
			request params: empty
			response params: play, capture, outputs, inputs, formats, twoWay, oneWay
		play:
			request params: self_address, address, port, volume, audioFormat
			response params: empty
		stop:
			request params: address, port
			response params: qos
		
		record:
			request params: address, port, id, alert, captureFormat
			response params: empty
		stop_record:
			request params: id, alert
			response params: empty

		stopall:
			request params: empty
			response params: empty
		
]]--

debug = false

local playThreads = {};
local captureThread = {};

function processUdpCommand(obj)
	local cmd = obj.command_name;

	if cmd=="getaudioinfo" then

		local f = io.popen("amixer");
		local str = f:read("*all");
		f:close();

		if debug then
			print("From amixer: "..str);
		end

		if #str==0 then
			io.write("Failed to access sound card!",'\n');
			io.flush();
	  		return {
	  			command_name="getaudioinfo",
	  			play=false,
	  			capture=false,
	  			outputs=0,
	  			inputs=0,
	  			formats="",
	  			twoWay=false,
	  			oneWay=false
	  		};
		end

	  	return {
	  		command_name="getaudioinfo",
	  		play=true,
	  		capture=true,
	  		outputs=1,
	  		inputs=1,
	  		formats="ULAW_8KHZ,PCM_44KHZ",
	  		twoWay=true,
	  		oneWay=true
	  	};
	end

	if cmd=="play" then
		--Ack the request
		if obj.audioFormat==nil then
			obj.audioFormat="ULAW_8KHZ";
		end


		if obj.address~=nil and obj.port~=nil and obj.volume~=nil and obj.self_address~=nil then
			if obj.audioFormat=="PCM_44KHZ" then
				if startStream("PCM_44KHZ",obj.self_address, obj.address, obj.port, obj.audioFormat, obj.volume) then
					return {
			  			command_name="play"
			  		};
				else
			  		return {
			  			command_name="play",
			  			error="Internal error: No playback device!",
			  			code=7
			  		};
				end
			else
				if startStream("ULAW_8KHZ",obj.self_address, obj.address, obj.port, obj.audioFormat, obj.volume) then
					return {
			  			command_name="play"
			  		};
				else 
			  		return {
			  			command_name="play",
			  			error="Internal error: No playback device!",
			  			code=7
			  		};
				end
			end
		end

  		return {
  			command_name="play",
  			error="Internal error: Invalid values",
  			code=8
  		};
	end

	if cmd=="stop" then
		if obj.address~=nil and obj.port~=nil then
			local thr = playThreads[obj.address..":"..tostring(obj.port)];
			if thr~=nil then
				thr.process:kill(0);
				thr.udpSocket:set_membership(obj.address,thr.addressSelf,"leave");
				thr.udpSocket:recv_stop();
				thr.udpSocket:close();

		  		return {
		  			command_name="stop",
		  			qos=thr.qosData
		  		};
			end
		end
  		return {
  			command_name="stop",
  			error="Stream not active",
  			code=1
  		};
	end

	if cmd=="stopall" then

		for k,thr in pairs(playThreads) do
			if thr~=nil then
				thr.process:kill(0);
				thr.udpSocket:set_membership(obj.address,thr.addressSelf,"leave");
				thr.udpSocket:recv_stop();
				thr.udpSocket:close();
			end
		end

		return {
		  	command_name="stopall"
		};
		
	end
end

function startRecording()
	
end

function startStream(format, self_address, address, port, audioFormat, volume)

	local f = io.popen("amixer set Speaker "..tostring(volume*10).."%");
	local str = f:read("*all");
	f:close();

	if str_utils.starts_with(str,"amixer: Mixer attach default error") then
		io.write("Failed to access sound card!",'\n');
		io.flush();
	end

	local stdin = uv.new_pipe(true);
	local stdout = uv.new_pipe(true);
	local stderr = uv.new_pipe(true);

	local bytes_recv = 0;

	local udp = uv.new_udp();
	udp:bind('0.0.0.0',tonumber(port));

	local addrArr = str_utils.split(address, ".");
	local firstByte = tonumber(addrArr[1]);
	if firstByte>=224 and firstByte<=239 then
		udp:set_membership(address,self_address,"join");
		udp:set_multicast_loop(true);
		udp:set_multicast_ttl(255);
	end

	local seqNum = -1;

	local qosData = {
		pktCounter=0,
		nonconsecPkts=0,
		consecPkts=0,
		duplicatePkts=0,
		ignoredPkts=0
	};

	local seqRepetitions = 0;

	local lastTimestamp = -1;

	udp:recv_start(function(d,data,sockaddr)
		if data~=nil then
			local sq = (seqRepetitions*math.pow(256,2)) + (data:byte(3)*math.pow(256,1)) + data:byte(4);
			local ts = (data:byte(5)*math.pow(256,3)) + (data:byte(6)*math.pow(256,2)) + (data:byte(7)*math.pow(256,1)) + data:byte(8);
			local src = (data:byte(9)*math.pow(256,3)) + (data:byte(10)*math.pow(256,2)) + (data:byte(11)*math.pow(256,1)) + data:byte(12);
			if ts<=lastTimestamp then
				io.write("Duplicate packet!",'\n');
				io.flush();
				qosData.duplicatePkts = qosData.duplicatePkts+1;
				--Don't play this...
				qosData.ignoredPkts = qosData.ignoredPkts+1;
				return;
			end

			if math.fmod(sq,65536)<math.fmod(seqNum,65536) and ts>lastTimestamp then --If current sequence is lower than previous, but timestamp is higher
				--Wrap around
				io.write("Wraping around...",'\n');
				io.flush();
				seqRepetitions = seqRepetitions+1;
				sq = sq+65536;
			end

			stdin:write(data:sub(13));
			--stdin:flush();
			bytes_recv = bytes_recv + data:len()-12;
			io.write("size:"..tostring(bytes_recv).." sq:"..tostring(sq).." ts:"..tostring(ts).." src:"..tostring(src),'\n');
			io.flush();
			qosData.pktCounter = qosData.pktCounter+1;
			if seqNum==-1 or sq==seqNum+1 then
				qosData.consecPkts = qosData.consecPkts+1;
			else
				qosData.nonconsecPkts = qosData.nonconsecPkts+1;
			end
			seqNum = sq;
		end
	end);

	local argsFormat = {"-t","raw","-f","MU_LAW","-r","8000"};

	if format=="PCM_44KHZ" then
		argsFormat={"-t","raw","-f","U16","-r","16000"};
	end

	playThreads[address..":"..tostring(port)] = {
		process=uv.spawn("aplay", {
			args=argsFormat,
			stdio={
				stdin,stdout,stderr
			}
		}, function()
			print("Process ended!");
		end),
		udpSocket=udp,
		addressSelf=self_address,
		qosData=qosData
	};

	uv.read_start(stdout, function(err,data)
		io.write("|STDOUT| "..tostring(data),'\n');
		io.flush();
	end);
	uv.read_start(stderr, function(err,data)
		io.write("|STDERR| "..tostring(data),'\n');
		io.flush();
	end);

	return true;

end

function main()
	biudp.initUdpServer(processUdpCommand, uv, AUDIO_NODE_PORT, "127.0.0.1", "127.0.0.1");
end

main();
while true do
	uv.run();
end