local str_utils = require("str_utils");
local uv = require("luv");

DHCP_VENDOR_CLASS_IDENTIFIER = "Barix DHCP 0.1.1"

local module = {};
--[[

DHCP inform:
	- 1-byte message type [0x01 - boot request]
	- 1-byte hardware type [0x01 - Ethernet]
	- 1-byte hardware address length [0x06 - MAC is 6-byte long]
	- 1-byte hops [0x00 - zero hops]
	- 4-byte transaction ID [randomly generated]
	- 2-byte seconds elapsed [0x00 0x00]
	- 2-byte flags [0x80 0x00 - Broadcast response]
	- 4-byte my ip
	- 4-byte ip [0x00 0x00 0x00 0x00]
	- 4-byte server ip [0x00 0x00 0x00 0x00]
	- 4-byte relay agent ip [0x00 0x00 0x00 0x00]
	- 6-byte mac address
	- 10-byte mac address padding [0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00]
	- 64-byte nothing 
	- 128-byte nothing
	- 4-byte magic cookie [0x63 0x82 0x53 0x63]
	OPTIONS LIST
	- 0x35 dhcp message type - 0x01 [Length 1] 0x01 [Discover]
	- 0x3d Client ID - 0x07 [Length 7] 0x01 [Ethernet] 6-byte mac address
	- 0x39 Maximum DHCP Message size - 0x02 [Length 2] 0x02 0x40 [576]
	- 0x37 Parameter request list - [LENGTH] [Options...]
	- 0x3C Vendor class identifier - [Name length] [Name]
	- 0xFF End
	
	PAD to 300bytes
]]

math.randomseed(os.time());

--[[
	mac - string 00:00:00:00:00:00
	ipAddress - string 192.168.1.1
	options - {72}
]]
function module.generateInformPacket(mac, ipAddress, options, broadcastResponse)

	local arr = str_utils.split(ipAddress, ".");
	local ipArr = {};
	for i=1,4,1 do
		ipArr[i] = tonumber(arr[i]);
	end

	local arr2 = str_utils.split(mac, ":");
	local macArr = {};
	for i=1,6,1 do
		macArr[i] = tonumber(arr2[i], 16);
	end

	local packet = string.char(
		0x01,--Boot request
		0x01,--Ethernet
		0x06,--Hardware address length
		0x00 --Hops
	);

	local txId = {};
	for i=1,4,1 do
		txId[i] = math.random(0, 255);
		packet = packet..string.char(txId[i]);
	end

	local brdRplyByte = 0x00;
	if broadcast then
		brdRplyByte = 0x80;
	end

	packet = packet..string.char(
		0x00, 0x00, --Time elapsed
		brdRplyByte, 0x00, --Don't broadcast response
		ipArr[1], ipArr[2], ipArr[3], ipArr[4], --Self address
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		macArr[1], macArr[2], macArr[3], macArr[4], macArr[5], macArr[6],
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	);

	--Server host name
	for i=1,64,1 do 
		packet = packet..string.char(0x00);
	end

	--Boot file name
	for i=1,128,1 do 
		packet = packet..string.char(0x00);
	end

	packet = packet..string.char(
		0x63, 0x82, 0x53, 0x63 --Magic cookie
	);

	packet = packet..string.char(
		0x35, 0x01, 0x08, --dhcp message type INFORM
		0x3d, 0x07, 0x01, macArr[1], macArr[2], macArr[3], macArr[4], macArr[5], macArr[6], --client ID
		0x39, 0x02, 0x02, 0x40,
		0x37, #options
	);

	for i=1,#options,1 do
		packet = packet..string.char(options[i]);
	end

	packet = packet..string.char(0x3C, #DHCP_VENDOR_CLASS_IDENTIFIER)..DHCP_VENDOR_CLASS_IDENTIFIER;

	--End
	packet = packet..string.char(0xff);

	--Pad to 300 bytes
	while #packet<300 do
		packet = packet..string.char(0x00);
	end

	return packet, txId;

end
function module.obtainOptions(mac, ipAddress, options, callback, tim, brdAddr) 
	local udp = uv.new_udp();
	local timeout = uv.new_timer();

	local sndPacket, txId = module.generateInformPacket(mac, ipAddress, options, brdAddr==nil);

	local function stop()
		udp:recv_stop();
		udp:close();
		timeout:stop();
		timeout:close();
	end

	timeout:start(tim, 0, function()
		print("|DHCP| Timed out");
		stop();
		callback(nil);
	end);

	udp:bind(ipAddress, 52868);
	udp:recv_start(function(err, data, address)
		if err or data==nil then
			stop();
			callback(nil);
			return;
		end

		--Parse packet
		if #data<8 or data:byte(5)~=txId[1] or data:byte(6)~=txId[2] or data:byte(7)~=txId[3] or data:byte(8)~=txId[4] then
			print("|DHCP| Tx ID doesn't match inform ID");
			return;
		end

		local opts = {};

		if #data>240 then
			--Read options
			local readPtr = 241;
			while #data>readPtr do
				local optNumber = data:byte(readPtr);
				local length = data:byte(readPtr+1);
				readPtr = readPtr+2;
				opts[optNumber] = data:sub(readPtr, readPtr+length-1);
				readPtr = readPtr+length;
			end

			stop();
			callback(opts);

		end

	end);

	udp:set_broadcast(true);

	local brd = "255.255.255.255";
	if brdAddr~=nil then brd = brdAddr; end

	print("|DHCP| Sending INFORM mac: "..mac.." ip: "..ipAddress.." to: "..brd);
	udp:send(sndPacket, brd, 67);
end

return module;