#!/usr/bin/env python3
"""
BM44 IO Mapping Web Server
A Flask-based web server that provides HTTP endpoints for reading and writing
IO addresses on the Barionet M44 using the IO Mapping system.
"""

import json
import logging
import logging.handlers
import os
import signal
import socket
import sys
import threading
import time
from typing import Dict, Optional, Tuple, Any
from flask import Flask, request, Response
from iomapping import IoMapping, RegisterType

class TimedToggleManager:
    """Manages timed toggle operations for IO addresses."""
    
    def __init__(self, io_service: IoMapping):
        self.io_service = io_service
        self.active_toggles: Dict[int, threading.Timer] = {}
        self.lock = threading.Lock()
    
    def start_timed_toggle(self, address: int, duration_seconds: float) -> None:
        """Start a timed toggle operation for the given address."""
        with self.lock:
            # Cancel any existing timer for this address
            if address in self.active_toggles:
                self.active_toggles[address].cancel()
                del self.active_toggles[address]
            
            # Turn the address ON first
            try:
                self.io_service.write_value(address, 1)
                logger.info(f"Timed toggle: Set address {address} to ON for {duration_seconds}s")
            except Exception as e:
                logger.error(f"Failed to turn ON address {address} for timed toggle: {e}")
                return
            
            # Create timer to turn it OFF after duration
            def turn_off():
                try:
                    self.io_service.write_value(address, 0)
                    logger.info(f"Timed toggle: Auto-turned OFF address {address}")
                except Exception as e:
                    logger.error(f"Failed to auto-turn OFF address {address}: {e}")
                finally:
                    with self.lock:
                        if address in self.active_toggles:
                            del self.active_toggles[address]
            
            timer = threading.Timer(duration_seconds, turn_off)
            self.active_toggles[address] = timer
            timer.start()
    
    def cancel_toggle(self, address: int) -> None:
        """Cancel any active timed toggle for the given address."""
        with self.lock:
            if address in self.active_toggles:
                self.active_toggles[address].cancel()
                del self.active_toggles[address]
                logger.info(f"Cancelled timed toggle for address {address}")
    
    def shutdown(self) -> None:
        """Cancel all active timed toggles."""
        with self.lock:
            for timer in self.active_toggles.values():
                timer.cancel()
            self.active_toggles.clear()
            logger.info("All timed toggles cancelled")

class BM44WebServer:
    """Main web server class for BM44 IO Mapping."""
    
    def __init__(self):
        self.app = Flask(__name__)
        self.io_service = None
        self.toggle_manager = None
        self.config = self.load_config()
        self.address_lock = threading.Lock()
        self.setup_logging()
        self.setup_routes()
        self.setup_signal_handlers()
        
        # IO address configuration based on documentation
        self.address_config = self._build_address_config()
    
    def load_config(self) -> Dict[str, Any]:
        """Load configuration from config.json or default_config.json."""
        config_files = ['config.json', 'default_config.json']
        
        for config_file in config_files:
            if os.path.exists(config_file):
                try:
                    with open(config_file, 'r') as f:
                        config = json.load(f)
                    print(f"Loaded configuration from {config_file}")
                    return config.get('AppParam', {})
                except Exception as e:
                    print(f"Error loading {config_file}: {e}")
        
        # Fallback to default configuration
        print("Using default configuration")
        return {
            'port': 8080,
            'allowed_ips': '',
            'remote_syslog': {
                'syslog_enabled': False,
                'syslog_ip': '192.168.1.100',
                'syslog_port': 514
            }
        }
    
    def setup_logging(self) -> None:
        """Setup logging configuration."""
        global logger
        logger = logging.getLogger('BM44WebServer')
        logger.setLevel(logging.INFO)
        
        # Clear any existing handlers
        logger.handlers.clear()
        
        # Get syslog configuration
        syslog_config = self.config.get('remote_syslog', {})
        syslog_enabled = syslog_config.get('syslog_enabled', False)
        
        if syslog_enabled:
            syslog_ip = syslog_config.get('syslog_ip', '192.168.1.100')
            syslog_port = syslog_config.get('syslog_port', 514)
            
            # Validate syslog configuration
            if syslog_ip and syslog_port:
                try:
                    # Create syslog handler
                    syslog_handler = logging.handlers.SysLogHandler(
                        address=(syslog_ip, syslog_port),
                        socktype=socket.SOCK_DGRAM
                    )
                    
                    # Set syslog formatter
                    syslog_formatter = logging.Formatter(
                        'BM44WebServer[%(process)d]: %(levelname)s - %(message)s'
                    )
                    syslog_handler.setFormatter(syslog_formatter)
                    logger.addHandler(syslog_handler)
                    
                    print(f"Syslog logging enabled - sending to {syslog_ip}:{syslog_port}")
                    logger.info("BM44 Web Server started with syslog logging")
                    
                except Exception as e:
                    print(f"Failed to setup syslog logging: {e}")
                    print("Falling back to console logging")
                    self._setup_console_logging()
            else:
                print("Invalid syslog configuration - missing IP or port")
                print("Falling back to console logging")
                self._setup_console_logging()
        else:
            # Syslog disabled - use console logging if running from terminal
            self._setup_console_logging()
    
    def _setup_console_logging(self) -> None:
        """Setup console logging if running from terminal."""
        global logger
        
        # Only enable console logging when running from terminal
        if sys.stdout.isatty():
            handler = logging.StreamHandler(sys.stdout)
            formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
            handler.setFormatter(formatter)
            logger.addHandler(handler)
            print("Console logging enabled")
        else:
            # No logging when not running from terminal and syslog disabled
            logger.addHandler(logging.NullHandler())
            logger.setLevel(logging.CRITICAL)
    
    def _build_address_config(self) -> Dict[int, Dict[str, Any]]:
        """Build address configuration mapping based on documentation."""
        config = {}
        
        # 1-bit addresses - Relays
        for addr in range(1, 5):  # 1-4
            config[addr] = {'size': 1, 'type': 'relay', 'read': True, 'write': True}
        
        # RTS
        config[9] = {'size': 1, 'type': 'rts', 'read': True, 'write': True}
        
        # Virtual IO (1-bit)
        config[10] = {'size': 1, 'type': 'virtual', 'read': True, 'write': True}
        
        # UX8 Extension Relays (1-bit each)
        for addr in range(11, 43):  # 11-42
            config[addr] = {'size': 1, 'type': 'relay', 'read': True, 'write': True}
        
        # Virtual IO bits
        for addr in range(43, 101):  # 43-100
            config[addr] = {'size': 1, 'type': 'virtual', 'read': True, 'write': True}
        
        # Digital Outputs (reserved for future)
        for addr in range(101, 109):  # 101-108
            config[addr] = {'size': 1, 'type': 'output_digital', 'read': True, 'write': True}
        
        # Virtual IO bits
        for addr in range(109, 201):  # 109-200
            config[addr] = {'size': 1, 'type': 'virtual', 'read': True, 'write': True}
        
        # Digital Inputs (read-only)
        for addr in range(201, 205):  # 201-204
            config[addr] = {'size': 1, 'type': 'input_digital', 'read': True, 'write': False}
        
        # CTS (read-only)
        config[209] = {'size': 1, 'type': 'cts', 'read': True, 'write': False}
        
        # Virtual IO
        config[210] = {'size': 1, 'type': 'virtual', 'read': True, 'write': True}
        
        # UX8 Extension Digital Inputs (read-only)
        for addr in range(211, 243):  # 211-242
            config[addr] = {'size': 1, 'type': 'input_digital', 'read': True, 'write': False}
        
        # Virtual IO bits
        for addr in range(243, 301):  # 243-300
            config[addr] = {'size': 1, 'type': 'virtual', 'read': True, 'write': True}
        
        # Pullups
        for addr in range(301, 305):  # 301-304
            config[addr] = {'size': 1, 'type': 'pullup', 'read': True, 'write': True}
        
        # Virtual IO bits
        for addr in range(309, 401):  # 309-400
            config[addr] = {'size': 1, 'type': 'virtual', 'read': True, 'write': True}
        
        # 32-bit addresses - Input Counters
        for addr in range(401, 405):  # 401-404
            config[addr] = {'size': 32, 'type': 'input_counter', 'read': True, 'write': True}
        
        # Virtual 32-bit registers
        for addr in range(409, 501):  # 409-500
            config[addr] = {'size': 32, 'type': 'virtual', 'read': True, 'write': True}
        
        # 16-bit addresses - Analog Inputs
        for addr in range(501, 505):  # 501-504
            config[addr] = {'size': 16, 'type': 'input_analog', 'read': True, 'write': False}
        
        # Virtual 16-bit registers and UX8 analogs
        for addr in range(509, 601):  # 509-600
            config[addr] = {'size': 16, 'type': 'virtual', 'read': True, 'write': True}
        
        # Temperature sensors (read-only, 16-bit)
        for addr in range(601, 751):  # 601-750
            config[addr] = {'size': 16, 'type': 'sensor_temp', 'read': True, 'write': False}
        
        # Virtual 16-bit registers
        for addr in range(751, 1201):  # 751-1200
            config[addr] = {'size': 16, 'type': 'virtual', 'read': True, 'write': True}
        
        # Device status (read-only, various sizes)
        config[1201] = {'size': 16, 'type': 'device_current', 'read': True, 'write': False}
        config[1202] = {'size': 16, 'type': 'device_voltage', 'read': True, 'write': False}
        config[1203] = {'size': 16, 'type': 'cpu_temp', 'read': True, 'write': False}
        config[1204] = {'size': 32, 'type': 'uptime', 'read': True, 'write': False}
        config[1205] = {'size': 16, 'type': 'hw_type', 'read': True, 'write': False}
        config[1206] = {'size': 16, 'type': 'fw_version', 'read': True, 'write': False}
        
        # Control registers
        config[1207] = {'size': 1, 'type': 'usb_enable', 'read': True, 'write': True}
        for addr in range(1208, 1244):  # LED and enable registers
            config[addr] = {'size': 4 if addr <= 1211 else 1, 'type': 'control', 'read': True, 'write': True}
        
        # Count registers (read-only)
        for addr in range(60001, 60008):  # 60001-60007
            config[addr] = {'size': 16 if addr <= 60006 else 1, 'type': 'count', 'read': True, 'write': False}
        
        return config
    
    def get_allowed_ips_list(self) -> list:
        """Parse the allowed_ips configuration string into a list."""
        allowed_ips_str = self.config.get('allowed_ips', '')
        if not allowed_ips_str or allowed_ips_str.strip() == '':
            return []
        
        # Split by comma and clean up whitespace
        ips = [ip.strip() for ip in allowed_ips_str.split(',') if ip.strip()]
        return ips
    
    def setup_routes(self) -> None:
        """Setup Flask routes."""
        @self.app.before_request
        def check_allowed_ip():
            """Check if the request comes from an allowed IP."""
            allowed_ips = self.get_allowed_ips_list()
            
            # If no allowed IPs configured, allow all
            if not allowed_ips:
                return
            
            client_ip = request.environ.get('REMOTE_ADDR')
            if client_ip not in allowed_ips:
                logger.warning(f"Request blocked from unauthorized IP: {client_ip} (allowed: {', '.join(allowed_ips)})")
                return Response(
                    "Access denied\n",
                    status=403,
                    headers={
                        'Connection': 'close',
                        'Cache-Control': 'no-cache',
                        'Pragma': 'no-cache',
                        'Server': 'BarionetM44/CGI_Legacy_Handler'
                    }
                )
        
        @self.app.after_request
        def modify_server_header(response):
            """Modify the Server header for all responses."""
            response.headers['Server'] = 'BarionetM44/CGI_Legacy_Handler'
            return response
        
        @self.app.route('/rc.cgi', methods=['GET'])
        @self.app.route('/bas.cgi', methods=['GET'])
        def handle_request():
            return self.handle_io_request()
    
    def setup_signal_handlers(self) -> None:
        """Setup signal handlers for graceful shutdown."""
        def signal_handler(signum, frame):
            logger.info(f"Received signal {signum}, shutting down...")
            if self.toggle_manager:
                self.toggle_manager.shutdown()
            sys.exit(0)
        
        signal.signal(signal.SIGINT, signal_handler)
        signal.signal(signal.SIGTERM, signal_handler)
    
    def get_address_info(self, address: int) -> Optional[Dict[str, Any]]:
        """Get address configuration information."""
        return self.address_config.get(address)
    
    def handle_io_request(self) -> Response:
        """Handle incoming IO requests."""
        logger.info(f"Incoming request: {request.url}")
        
        try:
            # Handle write operations
            if 'o' in request.args:
                return self.handle_write_request(request.args.get('o'))
            
            # Handle read operations
            elif 'state' in request.args:
                return self.handle_read_request(request.args.get('state'))
            
            else:
                logger.warning("Invalid request - missing parameters")
                return self.create_response("Invalid request")
        
        except Exception as e:
            logger.error(f"Unexpected error handling request: {e}")
            return self.create_response("Internal error")
    
    def handle_write_request(self, param: str) -> Response:
        """Handle write operations."""
        try:
            # Parse address and value
            parts = param.split(',')
            if len(parts) != 2:
                logger.warning(f"Invalid write format: {param}")
                return self.create_response("Invalid format")
            
            address = int(parts[0])
            value = int(parts[1])
            
            logger.info(f"Write request: address={address}, value={value}")
            
            # Validate address
            addr_info = self.get_address_info(address)
            if not addr_info:
                logger.warning(f"Invalid address: {address}")
                return self.create_response("Invalid Address")
            
            if not addr_info['write']:
                logger.warning(f"Address {address} is read-only")
                return self.create_response("Invalid value for the requested address")
            
            # Handle the write operation
            with self.address_lock:
                success = self.perform_write_operation(address, value, addr_info)
            
            if success:
                response = self.create_response("<ack>")
                logger.info(f"Write response: 200 OK <ack>")
                return response
            else:
                return self.create_response("Invalid value for the requested address")
        
        except ValueError:
            logger.warning(f"Invalid numeric values in: {param}")
            return self.create_response("Invalid format")
        except Exception as e:
            logger.error(f"Error in write request: {e}")
            return self.create_response("Invalid value for the requested address")
    
    def perform_write_operation(self, address: int, value: int, addr_info: Dict[str, Any]) -> bool:
        """Perform the actual write operation based on address type and value."""
        bit_size = addr_info['size']
        
        try:
            # Cancel any existing timed toggle for this address
            self.toggle_manager.cancel_toggle(address)
            
            if bit_size == 1:
                # 1-bit addresses: handle special values
                if value == 0 or value == 1:
                    # Direct write
                    self.io_service.write_value(address, value)
                    logger.info(f"Direct write: address {address} = {value}")
                    return True
                
                elif value == 999:
                    # Immediate toggle
                    current_value = self.io_service.read_value(address)
                    new_value = 1 if current_value == 0 else 0
                    self.io_service.write_value(address, new_value)
                    logger.info(f"Immediate toggle: address {address} = {new_value} (was {current_value})")
                    return True
                
                elif (2 <= value <= 998) or (1000 <= value <= 9999):
                    # Timed toggle
                    duration = value / 10.0
                    self.toggle_manager.start_timed_toggle(address, duration)
                    return True
                
                else:
                    logger.warning(f"Invalid value {value} for 1-bit address {address}")
                    return False
            
            else:
                # Multi-bit addresses: direct write within limits
                max_value = (2 ** bit_size) - 1
                if 0 <= value <= max_value:
                    self.io_service.write_value(address, value)
                    logger.info(f"Direct write: address {address} = {value} ({bit_size}-bit)")
                    return True
                else:
                    logger.warning(f"Value {value} out of range for {bit_size}-bit address {address} (max: {max_value})")
                    return False
        
        except Exception as e:
            logger.error(f"IoMapping write failed for address {address}: {e}")
            return False
    
    def handle_read_request(self, param: str) -> Response:
        """Handle read operations."""
        try:
            address = int(param)
            logger.info(f"Read request: address={address}")
            
            # Validate address
            addr_info = self.get_address_info(address)
            if not addr_info:
                logger.warning(f"Invalid address: {address}")
                return self.create_response("Invalid Address")
            
            # Read the value
            try:
                value = self.io_service.read_value(address)
                response_body = f"<{address}>{value}<{address}>"
                response = self.create_response(response_body)
                logger.info(f"Read response: {response_body}")
                return response
            
            except Exception as e:
                logger.error(f"IoMapping read failed for address {address}: {e}")
                return self.create_response("Read error")
        
        except ValueError:
            logger.warning(f"Invalid address format: {param}")
            return self.create_response("Invalid Address")
        except Exception as e:
            logger.error(f"Error in read request: {e}")
            return self.create_response("Invalid Address")
    
    def create_response(self, body: str) -> Response:
        """Create a properly formatted HTTP response."""
        response = Response(
            f"{body}\n",
            status=200,
            headers={
                'Connection': 'close',
                'Cache-Control': 'no-cache',
                'Pragma': 'no-cache'
            }
        )
        return response
    
    def initialize_io_service(self) -> bool:
        """Initialize the IoMapping service."""
        try:
            self.io_service = IoMapping()
            self.toggle_manager = TimedToggleManager(self.io_service)
            logger.info("IoMapping service initialized successfully")
            return True
        except Exception as e:
            logger.error(f"Failed to initialize IoMapping service: {e}")
            return False
    
    def run(self) -> None:
        """Run the web server."""
        if not self.initialize_io_service():
            print("Failed to initialize IoMapping service. Exiting.")
            sys.exit(1)
        
        port = self.config.get('port', 8080)
        allowed_ips = self.get_allowed_ips_list()
        
        if allowed_ips:
            logger.info(f"Starting BM44 Web Server on 0.0.0.0:{port} (accepting requests only from: {', '.join(allowed_ips)})")
            print(f"Starting BM44 Web Server on 0.0.0.0:{port} (accepting requests only from: {', '.join(allowed_ips)})")
        else:
            logger.info(f"Starting BM44 Web Server on 0.0.0.0:{port} (accepting requests from any IP)")
            print(f"Starting BM44 Web Server on 0.0.0.0:{port} (accepting requests from any IP)")
        
        try:
            self.app.run(
                host='0.0.0.0',
                port=port,
                debug=False,
                use_reloader=False,
                threaded=True
            )
        except Exception as e:
            logger.error(f"Failed to start web server: {e}")
            print(f"Failed to start web server: {e}")
            sys.exit(1)

def main():
    """Main entry point."""
    server = BM44WebServer()
    server.run()

if __name__ == '__main__':
    main()