#!/usr/bin/env python3
"""
BM44 Digital Input UDP Notifier
Production-ready application for Barionet M44 that monitors digital inputs 
and sends UDP messages when inputs change from 0 to 1.
"""

import json
import logging
import logging.handlers
import os
import signal
import socket
import sys
import threading
import time
from typing import Dict, List, Optional, Tuple
from pathlib import Path


class IoMapping:
    """Interface to BM44 IO Mapping system via Python API"""
    
    def __init__(self):
        try:
            # Import the actual IoMapping module if available
            from iomapping import IoMapping as IoMappingService
            self._service = IoMappingService()
            self._mock_mode = False
        except ImportError:
            # Fallback to mock mode for development/testing
            self._service = None
            self._mock_mode = True
            self._mock_values = {201: 0, 202: 0, 203: 0, 204: 0}
            logging.warning("IoMapping service not available, running in mock mode")
    
    def read_value(self, address: int) -> int:
        """Read value from IO address"""
        if self._mock_mode:
            return self._mock_values.get(address, 0)
        return self._service.read_value(address)
    
    def enable_notifications(self, address: int, callback):
        """Enable notifications for address changes"""
        if not self._mock_mode:
            self._service.enable_notifications(address, callback)
    
    def disable_notifications(self, address: int):
        """Disable notifications for address"""
        if not self._mock_mode:
            self._service.disable_notifications(address)
    
    def run(self):
        """Process notifications (blocking call)"""
        if not self._mock_mode:
            # For real IoMapping service, just yield control - notifications are handled via callbacks
            time.sleep(0.1)
        else:
            time.sleep(0.1)  # Prevent busy loop in mock mode
    
    def is_alive(self) -> bool:
        """Check if service is alive"""
        if self._mock_mode:
            return True
        return self._service.is_alive()


class ConfigManager:
    """Configuration file manager with fallback support"""
    
    def __init__(self, config_path: str = "config.json", 
                 default_config_path: str = "default_config.json"):
        self.config_path = Path(config_path)
        self.default_config_path = Path(default_config_path)
        self._config: Dict = {}
        self._lock = threading.RLock()
        
    def load_config(self) -> Dict:
        """Load configuration with fallback to default"""
        with self._lock:
            try:
                # Try primary config first
                if self.config_path.exists():
                    with open(self.config_path, 'r') as f:
                        self._config = json.load(f)
                    logging.info(f"Loaded configuration from {self.config_path}")
                elif self.default_config_path.exists():
                    with open(self.default_config_path, 'r') as f:
                        self._config = json.load(f)
                    logging.info(f"Loaded default configuration from {self.default_config_path}")
                else:
                    raise FileNotFoundError("Neither config.json nor default_config.json found")
                
                # Validate configuration structure
                self._validate_config()
                
                return self._config.copy()
                
            except (json.JSONDecodeError, FileNotFoundError, ValueError) as e:
                logging.error(f"Configuration error: {e}")
                # Return minimal valid configuration
                return self._get_empty_config()
    
    def _validate_config(self) -> None:
        """Validate configuration structure"""
        if "AppParam" not in self._config:
            raise ValueError("Configuration missing 'AppParam' section")
        
        app_param = self._config["AppParam"]
        
        # Validate input sections
        for i in range(1, 5):
            input_key = f"input_{i}"
            if input_key in app_param:
                input_config = app_param[input_key]
                if not isinstance(input_config, dict):
                    raise ValueError(f"Configuration 'AppParam.{input_key}' must be an object")
                # Ensure required keys exist with defaults
                input_config.setdefault("target_list", "")
                input_config.setdefault("message_close", "")
                input_config.setdefault("message_open", "")
            else:
                # Add missing input section with defaults
                app_param[input_key] = {
                    "target_list": "",
                    "message_close": "",
                    "message_open": ""
                }
    
    def _get_empty_config(self) -> Dict:
        """Return empty configuration in AppParam nested format"""
        return {
            "AppParam": {
                f"input_{i}": {
                    "target_list": "",
                    "message_close": "",
                    "message_open": ""
                } for i in range(1, 5)
            }
        }
    
    def get_app_param(self, key: str, default: str = "") -> str:
        """Get application parameter with default (for simple string values)"""
        with self._lock:
            return self._config.get("AppParam", {}).get(key, default)
    
    def get_syslog_config(self) -> Dict[str, any]:
        """Get syslog configuration"""
        with self._lock:
            app_param = self._config.get("AppParam", {})
            return {
                "enabled": app_param.get("enable_syslog", False),
                "address": app_param.get("syslog_address", "")
            }
    
    def get_input_config(self, input_num: int) -> Dict[str, str]:
        """Get input configuration"""
        with self._lock:
            if "AppParam" not in self._config:
                return {
                    "target_list": "",
                    "message_close": "",
                    "message_open": ""
                }
            
            app_param = self._config["AppParam"]
            input_key = f"input_{input_num}"
            
            if input_key in app_param and isinstance(app_param[input_key], dict):
                return app_param[input_key].copy()
            
            # Return empty config if input not found
            return {
                "target_list": "",
                "message_close": "",
                "message_open": ""
            }
    
    def is_input_configured(self, input_num: int) -> bool:
        """Check if input has any configuration (target_list not empty)"""
        config = self.get_input_config(input_num)
        return bool(config.get("target_list", "").strip())


class UDPNotifier:
    """UDP message sender with connection pooling and error handling"""
    
    def __init__(self, timeout: float = 5.0):
        self.timeout = timeout
        self._socket_pool: Dict[str, socket.socket] = {}
        self._lock = threading.RLock()
        
    def send_message(self, targets: List[Tuple[str, int]], message: str) -> None:
        """Send UDP message to multiple targets"""
        if not targets or not message:
            return
            
        # Ensure message ends with line feed
        if not message.endswith('\n'):
            message += '\n'
        
        message_bytes = message.encode('utf-8')
        
        with self._lock:
            for host, port in targets:
                try:
                    self._send_to_target(host, port, message_bytes)
                    logging.info(f"Sent message to {host}:{port}: {message.strip()}")
                except Exception as e:
                    logging.error(f"Failed to send message to {host}:{port}: {e}")
    
    def _send_to_target(self, host: str, port: int, message_bytes: bytes) -> None:
        """Send message to single target with socket reuse"""
        target_key = f"{host}:{port}"
        
        try:
            # Get or create socket for this target
            sock = self._socket_pool.get(target_key)
            if sock is None:
                sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
                sock.settimeout(self.timeout)
                
                # Enable broadcast if target appears to be a broadcast address
                if self._is_broadcast_address(host):
                    sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
                    logging.debug(f"Enabled broadcast for {host}")
                
                self._socket_pool[target_key] = sock
            
            # Send the message
            sock.sendto(message_bytes, (host, port))
            
        except (socket.error, OSError) as e:
            # Remove failed socket from pool
            if target_key in self._socket_pool:
                try:
                    self._socket_pool[target_key].close()
                except:
                    pass
                del self._socket_pool[target_key]
            raise e
    
    def _is_broadcast_address(self, host: str) -> bool:
        """Check if address appears to be a broadcast address"""
        try:
            # Common broadcast patterns
            if host == "255.255.255.255":
                return True
            
            # Check for subnet broadcast (ends with .255)
            if host.endswith('.255'):
                return True
            
            # Additional check: try to detect broadcast by IP analysis
            parts = host.split('.')
            if len(parts) == 4:
                # Check for common broadcast patterns like x.x.x.255
                if parts[3] == '255':
                    return True
        except:
            pass
        
        return False
    
    def close_all(self) -> None:
        """Close all pooled sockets"""
        with self._lock:
            for sock in self._socket_pool.values():
                try:
                    sock.close()
                except:
                    pass
            self._socket_pool.clear()


class InputMonitor:
    """Digital input state monitor with change detection"""
    
    def __init__(self, io_mapping: IoMapping, config_manager: ConfigManager):
        self.io_mapping = io_mapping
        self.config_manager = config_manager
        self._current_states: Dict[int, int] = {}
        self._lock = threading.RLock()
        self._callbacks: Dict[int, List] = {}
        self._monitored_addresses: List[int] = []
        
        # Note: Don't initialize monitoring here - will be done after config load
    
    def initialize_monitoring(self) -> None:
        """Initialize monitoring after configuration is loaded"""
        # Determine which addresses to monitor based on configuration
        self._update_monitored_addresses()
        
        # Initialize current states for monitored addresses
        self._update_states()
    
    def _update_monitored_addresses(self) -> None:
        """Update list of addresses to monitor based on configuration"""
        self._monitored_addresses = []
        self._callbacks = {}
        
        for input_num in range(1, 5):
            address = 200 + input_num  # 201, 202, 203, 204
            
            if self.config_manager.is_input_configured(input_num):
                self._monitored_addresses.append(address)
                self._callbacks[address] = []
                logging.info(f"Input {input_num} (address {address}) will be monitored")
            else:
                logging.info(f"Input {input_num} (address {address}) not configured, skipping monitoring")
    
    def get_monitored_addresses(self) -> List[int]:
        """Get list of currently monitored addresses"""
        return self._monitored_addresses.copy()
    
    def _update_states(self) -> None:
        """Update current state readings for monitored addresses"""
        with self._lock:
            for addr in self._monitored_addresses:
                try:
                    self._current_states[addr] = self.io_mapping.read_value(addr)
                except Exception as e:
                    logging.error(f"Failed to read address {addr}: {e}")
                    self._current_states[addr] = 0
    
    def add_change_callback(self, address: int, callback) -> None:
        """Add callback for address state changes"""
        if address in self._callbacks:
            self._callbacks[address].append(callback)
    
    def _on_value_change(self, address: int, new_value: int) -> None:
        """Handle value change from IoMapping notifications"""
        with self._lock:
            old_value = self._current_states.get(address, 0)
            self._current_states[address] = new_value
            
            # Trigger callbacks for any state change (0->1 or 1->0)
            if old_value != new_value:
                input_num = address - 200
                logging.info(f"Input {input_num} (address {address}) changed: {old_value} -> {new_value}")
                
                for callback in self._callbacks.get(address, []):
                    try:
                        callback(address, old_value, new_value)
                    except Exception as e:
                        logging.error(f"Callback error for address {address}: {e}")
    
    def start_monitoring(self) -> None:
        """Start monitoring with IoMapping notifications for configured inputs only"""
        for addr in self._monitored_addresses:
            try:
                # Create a proper closure to capture the address
                # IoMapping callback receives (address, value) as parameters
                def make_callback(expected_address):
                    return lambda address, value: self._on_value_change(expected_address, value)
                
                self.io_mapping.enable_notifications(addr, make_callback(addr))
                logging.info(f"Enabled notifications for address {addr}")
            except Exception as e:
                logging.error(f"Failed to enable notifications for address {addr}: {e}")
    
    def stop_monitoring(self) -> None:
        """Stop monitoring notifications"""
        for addr in self._monitored_addresses:
            try:
                self.io_mapping.disable_notifications(addr)
            except Exception as e:
                logging.error(f"Failed to disable notifications for address {addr}: {e}")
    
    def get_current_state(self, address: int) -> int:
        """Get current state of address"""
        with self._lock:
            return self._current_states.get(address, 0)
    
    def refresh_configuration(self) -> None:
        """Refresh monitored addresses based on updated configuration"""
        logging.info("Refreshing input monitoring configuration...")
        
        # Stop current monitoring
        self.stop_monitoring()
        
        # Update monitored addresses
        old_addresses = self._monitored_addresses.copy()
        self._update_monitored_addresses()
        
        # Log changes
        added = set(self._monitored_addresses) - set(old_addresses)
        removed = set(old_addresses) - set(self._monitored_addresses)
        
        if added:
            logging.info(f"Added monitoring for addresses: {sorted(added)}")
        if removed:
            logging.info(f"Removed monitoring for addresses: {sorted(removed)}")
        
        # Initialize states for new addresses
        self._update_states()
        
        # Restart monitoring with new configuration
        self.start_monitoring()


class BM44InputNotifier:
    """Main application class"""
    
    # Digital input addresses 1-4 (201-204)
    INPUT_ADDRESSES = [201, 202, 203, 204]
    
    def __init__(self):
        self.running = False
        self.config_manager = ConfigManager()
        self.io_mapping = IoMapping()
        self.udp_notifier = UDPNotifier()
        self.input_monitor = InputMonitor(self.io_mapping, self.config_manager)
        self._shutdown_event = threading.Event()
        
        # Setup logging
        self._setup_logging()
        
        # Setup signal handlers
        signal.signal(signal.SIGINT, self._signal_handler)
        signal.signal(signal.SIGTERM, self._signal_handler)
    
    def _setup_logging(self) -> None:
        """Configure logging based on execution environment"""
        # Detect if running from terminal
        is_terminal = sys.stdout.isatty()
        
        # Configure logging format
        log_format = '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
        syslog_format = 'BM44-UDP-Trigger[%(process)d]: %(levelname)s - %(message)s'
        
        # Setup handlers
        handlers = []
        
        if is_terminal:
            # Add stdout handler for terminal execution
            stdout_handler = logging.StreamHandler(sys.stdout)
            stdout_handler.setFormatter(logging.Formatter(log_format))
            handlers.append(stdout_handler)
        else:
            # For non-terminal execution, still log to stdout but with different format
            stdout_handler = logging.StreamHandler(sys.stdout)
            stdout_handler.setFormatter(logging.Formatter(
                '%(asctime)s [%(levelname)s] %(message)s'))
            handlers.append(stdout_handler)
        
        # Configure root logger
        logging.basicConfig(
            level=logging.INFO,
            format=log_format,
            handlers=handlers
        )
        
        # Suppress excessive debug logs from other modules
        logging.getLogger('urllib3').setLevel(logging.WARNING)
    
    def _setup_syslog(self) -> None:
        """Setup syslog handler based on configuration"""
        syslog_config = self.config_manager.get_syslog_config()
        
        if not syslog_config["enabled"]:
            logging.info("Syslog disabled in configuration")
            return
        
        syslog_address = syslog_config["address"]
        if not syslog_address:
            logging.warning("Syslog enabled but no syslog_address configured")
            return
        
        try:
            # Parse syslog address
            if ':' in syslog_address:
                host, port_str = syslog_address.rsplit(':', 1)
                port = int(port_str)
                syslog_addr = (host, port)
            else:
                # Default syslog port
                syslog_addr = (syslog_address, 514)
            
            # Create syslog handler
            syslog_handler = logging.handlers.SysLogHandler(
                address=syslog_addr,
                socktype=socket.SOCK_DGRAM
            )
            
            # Set syslog format
            syslog_formatter = logging.Formatter(
                'BM44-UDP-Trigger[%(process)d]: %(levelname)s - %(message)s'
            )
            syslog_handler.setFormatter(syslog_formatter)
            
            # Add syslog handler to root logger
            root_logger = logging.getLogger()
            root_logger.addHandler(syslog_handler)
            
            logging.info(f"Syslog enabled - sending logs to {syslog_addr[0]}:{syslog_addr[1]}")
            
        except Exception as e:
            logging.error(f"Failed to setup syslog: {e}")
            logging.warning("Continuing without syslog")
    
    def _signal_handler(self, signum: int, frame) -> None:
        """Handle shutdown signals"""
        logging.info(f"Received signal {signum}, initiating shutdown...")
        self.shutdown()
    
    def _parse_target_list(self, target_string: str) -> List[Tuple[str, int]]:
        """Parse comma-separated IP:port list"""
        targets = []
        if not target_string.strip():
            return targets
        
        for target in target_string.split(','):
            target = target.strip()
            if not target:
                continue
                
            try:
                if ':' in target:
                    host, port_str = target.rsplit(':', 1)
                    port = int(port_str)
                    targets.append((host, port))
                else:
                    logging.warning(f"Invalid target format (no port): {target}")
            except ValueError as e:
                logging.error(f"Invalid target format: {target} - {e}")
        
        return targets
    
    def _on_input_change(self, address: int, old_value: int, new_value: int) -> None:
        """Handle input state change"""
        # Determine input number (1-4)
        input_num = address - 200  # 201->1, 202->2, etc.
        
        # Get configuration for this input
        input_config = self.config_manager.get_input_config(input_num)
        
        target_list_str = input_config.get("target_list", "")
        
        # Determine which message to send based on state change
        message = ""
        if old_value == 0 and new_value == 1:
            # Input closed (0 -> 1)
            message = input_config.get("message_close", "")
            transition = "closed"
        elif old_value == 1 and new_value == 0:
            # Input opened (1 -> 0)
            message = input_config.get("message_open", "")
            transition = "opened"
        else:
            # No transition or unexpected change
            logging.debug(f"Input {input_num} state change {old_value}->{new_value} (no action)")
            return
        
        if not target_list_str or not message:
            logging.debug(f"Input {input_num} {transition} but no target_list or message configured")
            return
        
        # Parse targets and send message
        targets = self._parse_target_list(target_list_str)
        if targets:
            logging.info(f"Input {input_num} {transition}, sending '{message}' to {len(targets)} targets")
            self.udp_notifier.send_message(targets, message)
        else:
            logging.warning(f"No valid targets found for input {input_num}")
    
    def initialize(self) -> bool:
        """Initialize application components"""
        try:
            # Load configuration
            config = self.config_manager.load_config()
            logging.info("Configuration loaded successfully")
            
            # Setup syslog after configuration is loaded
            self._setup_syslog()
            
            # Check IoMapping availability
            if not self.io_mapping.is_alive():
                logging.error("IoMapping service is not available")
                return False
            
            # Initialize input monitoring based on loaded configuration
            self.input_monitor.initialize_monitoring()
            
            # Check if any inputs are configured for monitoring
            monitored_addresses = self.input_monitor.get_monitored_addresses()
            if not monitored_addresses:
                logging.warning("No inputs configured for monitoring")
                return False
            
            # Setup input monitoring callbacks for configured inputs
            for addr in monitored_addresses:
                self.input_monitor.add_change_callback(addr, self._on_input_change)
            
            # Start input monitoring
            self.input_monitor.start_monitoring()
            
            logging.info(f"Application initialized successfully - monitoring {len(monitored_addresses)} inputs")
            return True
            
        except Exception as e:
            logging.error(f"Initialization failed: {e}")
            return False
    
    def run(self) -> None:
        """Main application loop"""
        # Hardcoded startup delay to ensure system components are ready
        logging.info("Startup delay: waiting 3 seconds for system components to be ready...")
        time.sleep(3)
        
        if not self.initialize():
            logging.error("Application initialization failed")
            return
        
        self.running = True
        logging.info("BM44 Input Notifier started")
        
        try:
            # Log current input states for monitored inputs only
            monitored_addresses = self.input_monitor.get_monitored_addresses()
            for addr in monitored_addresses:
                input_num = addr - 200
                state = self.input_monitor.get_current_state(addr)
                logging.info(f"Input {input_num} (address {addr}) initial state: {state}")
            
            # Main event loop
            while self.running and not self._shutdown_event.is_set():
                try:
                    # Process IoMapping events
                    self.io_mapping.run()
                    
                    # Check for shutdown
                    if self._shutdown_event.wait(0.1):
                        break
                        
                except KeyboardInterrupt:
                    logging.info("Keyboard interrupt received")
                    break
                except Exception as e:
                    logging.error(f"Error in main loop: {e}")
                    time.sleep(1)  # Prevent tight error loop
        
        finally:
            self.cleanup()
    
    def shutdown(self) -> None:
        """Initiate graceful shutdown"""
        self.running = False
        self._shutdown_event.set()
    
    def cleanup(self) -> None:
        """Clean up resources"""
        logging.info("Cleaning up resources...")
        
        try:
            self.input_monitor.stop_monitoring()
        except Exception as e:
            logging.error(f"Error stopping input monitor: {e}")
        
        try:
            self.udp_notifier.close_all()
        except Exception as e:
            logging.error(f"Error closing UDP connections: {e}")
        
        logging.info("BM44 Input Notifier stopped")


def main():
    """Application entry point"""
    app = BM44InputNotifier()
    try:
        app.run()
    except Exception as e:
        logging.error(f"Unhandled exception: {e}")
        return 1
    return 0


if __name__ == "__main__":
    sys.exit(main())