"""
Barionet MQTT Application
Production-grade MQTT control for Barionet devices with relay control and digital input monitoring.
"""

import json
import logging
import logging.handlers
import os
import signal
import socket
import sys
import threading
import time
from typing import Dict, Optional, Any, Set

import paho.mqtt.client as mqtt


class ConfigManager:
    """Handles configuration loading and validation with fallback mechanism."""
    
    def __init__(self, config_path: str = "config.json", fallback_path: str = "default_config.json"):
        self.config_path = config_path
        self.fallback_path = fallback_path
        self.config = {}
        self.used_fallback = False
        self._load_config()
    
    def _load_config(self) -> None:
        """Load and validate configuration from JSON file with fallback."""
        config_file = None
        
        try:
            # Try primary config first
            if os.path.exists(self.config_path):
                config_file = self.config_path
                print(f"Loading configuration from {self.config_path}")
            elif os.path.exists(self.fallback_path):
                config_file = self.fallback_path
                self.used_fallback = True
                print(f"Primary config not found, using fallback: {self.fallback_path}")
            else:
                raise FileNotFoundError(f"Neither {self.config_path} nor {self.fallback_path} found")
            
            with open(config_file, 'r') as f:
                raw_config = json.load(f)
            
            # Expect flat AppParam structure only
            if 'AppParam' not in raw_config:
                raise ValueError("Configuration must contain 'AppParam' section with flat parameters")
            
            self.config = raw_config['AppParam']
            self._validate_and_normalize_config()
            
        except Exception as e:
            print(f"CRITICAL: Failed to load configuration: {e}")
            sys.exit(1)
    
    def _validate_and_normalize_config(self) -> None:
        """Validate configuration and normalize data types."""
        # Required fields
        required_fields = [
            'mqtt_broker', 'mqtt_port', 'mqtt_client_id', 'mqtt_device_id'
        ]
        
        for field in required_fields:
            if field not in self.config:
                raise ValueError(f"Missing required configuration field: {field}")
        
        # Validate and normalize data types
        # MQTT settings
        if 'mqtt_port' in self.config:
            port = int(self.config['mqtt_port'])
            if not (1 <= port <= 65535):
                raise ValueError(f"Invalid mqtt_port: {port}. Must be between 1 and 65535")
            self.config['mqtt_port'] = port
            
        if 'mqtt_keepalive' in self.config:
            keepalive = int(self.config['mqtt_keepalive'])
            if not (10 <= keepalive <= 3600):
                raise ValueError(f"Invalid mqtt_keepalive: {keepalive}. Must be between 10 and 3600")
            self.config['mqtt_keepalive'] = keepalive
            
        if 'mqtt_qos' in self.config:
            qos = int(self.config['mqtt_qos'])
            if qos not in [0, 1, 2]:
                raise ValueError(f"Invalid mqtt_qos: {qos}. Must be 0, 1, or 2")
            self.config['mqtt_qos'] = qos
            
        if 'mqtt_last_will_qos' in self.config:
            lwt_qos = int(self.config['mqtt_last_will_qos'])
            if lwt_qos not in [0, 1, 2]:
                raise ValueError(f"Invalid mqtt_last_will_qos: {lwt_qos}. Must be 0, 1, or 2")
            self.config['mqtt_last_will_qos'] = lwt_qos
        
        # Handle "null" string values
        if self.config.get('mqtt_username') == "null":
            self.config['mqtt_username'] = None
        if self.config.get('mqtt_password') == "null":
            self.config['mqtt_password'] = None
        
        # Validate broker and device_id are not empty
        if not self.config.get('mqtt_broker', '').strip():
            raise ValueError("mqtt_broker cannot be empty")
        if not self.config.get('mqtt_device_id', '').strip():
            raise ValueError("mqtt_device_id cannot be empty")
        if not self.config.get('mqtt_client_id', '').strip():
            raise ValueError("mqtt_client_id cannot be empty")
        
        # Hardware settings
        if 'hardware_poll_interval' in self.config:
            interval = float(self.config['hardware_poll_interval'])
            if not (0.001 <= interval <= 10.0):
                raise ValueError(f"Invalid hardware_poll_interval: {interval}. Must be between 0.001 and 10.0")
            self.config['hardware_poll_interval'] = interval
        
        # Logging settings
        if 'logging_syslog_port' in self.config:
            syslog_port = int(self.config['logging_syslog_port'])
            if not (1 <= syslog_port <= 65535):
                raise ValueError(f"Invalid logging_syslog_port: {syslog_port}. Must be between 1 and 65535")
            self.config['logging_syslog_port'] = syslog_port
        
        # Validate logging level
        if 'logging_level' in self.config:
            level = self.config['logging_level'].upper()
            if level not in ['DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL']:
                raise ValueError(f"Invalid logging_level: {level}")
            self.config['logging_level'] = level
        
        # Application settings
        if 'application_reconnect_delay' in self.config:
            delay = float(self.config['application_reconnect_delay'])
            if not (1.0 <= delay <= 300.0):
                raise ValueError(f"Invalid application_reconnect_delay: {delay}. Must be between 1.0 and 300.0")
            self.config['application_reconnect_delay'] = delay
            
        if 'application_max_reconnect_attempts' in self.config:
            attempts = int(self.config['application_max_reconnect_attempts'])
            if not (1 <= attempts <= 100):
                raise ValueError(f"Invalid application_max_reconnect_attempts: {attempts}. Must be between 1 and 100")
            self.config['application_max_reconnect_attempts'] = attempts
            
        if 'application_health_check_interval' in self.config:
            interval = float(self.config['application_health_check_interval'])
            if not (30.0 <= interval <= 3600.0):
                raise ValueError(f"Invalid application_health_check_interval: {interval}. Must be between 30.0 and 3600.0")
            self.config['application_health_check_interval'] = interval
        
        # Set defaults for optional fields
        self.config.setdefault('mqtt_keepalive', 60)
        self.config.setdefault('mqtt_qos', 1)
        self.config.setdefault('mqtt_retain', False)
        self.config.setdefault('mqtt_last_will_qos', 1)
        self.config.setdefault('mqtt_last_will_retain', True)
        self.config.setdefault('hardware_poll_interval', 0.1)
        self.config.setdefault('logging_level', 'INFO')
        self.config.setdefault('logging_syslog_enabled', True)
        self.config.setdefault('logging_syslog_port', 514)
        self.config.setdefault('application_reconnect_delay', 5.0)
        self.config.setdefault('application_max_reconnect_attempts', 10)
        self.config.setdefault('application_health_check_interval', 300.0)
    
    def get(self, key: str, default: Any = None) -> Any:
        """Get configuration value by flat key."""
        return self.config.get(key, default)
    
    def is_using_fallback(self) -> bool:
        """Check if fallback configuration was used."""
        return self.used_fallback


class LoggingManager:
    """Handles logging configuration including syslog."""
    
    def __init__(self, config: ConfigManager):
        self.config = config
        self.logger = logging.getLogger('barionet')
        self._setup_logging()
    
    def _setup_logging(self) -> None:
        """Setup logging with syslog and optional console output."""
        log_level = getattr(logging, self.config.get('logging_level', 'INFO').upper())
        self.logger.setLevel(log_level)
        
        # Clear any existing handlers
        self.logger.handlers.clear()
        
        # Setup syslog if enabled
        if self.config.get('logging_syslog_enabled', True):
            self._setup_syslog()
        
        # Add console logging only if running from terminal
        if self._is_terminal():
            console_handler = logging.StreamHandler()
            console_formatter = logging.Formatter(
                '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
            )
            console_handler.setFormatter(console_formatter)
            self.logger.addHandler(console_handler)
    
    def _setup_syslog(self) -> None:
        """Setup remote syslog handler with hardcoded format."""
        try:
            syslog_handler = logging.handlers.SysLogHandler(
                address=(
                    self.config.get('logging_syslog_server', 'localhost'),
                    self.config.get('logging_syslog_port', 514)
                ),
                facility=logging.handlers.SysLogHandler.LOG_LOCAL0
            )
            
            # Hardcoded standard format
            hostname = socket.gethostname()
            syslog_format = f'{hostname} barionet[%(process)d]: %(levelname)s - %(message)s'
            syslog_formatter = logging.Formatter(syslog_format)
            syslog_handler.setFormatter(syslog_formatter)
            
            self.logger.addHandler(syslog_handler)
            
        except Exception as e:
            print(f"WARNING: Failed to setup syslog: {e}")
    
    def _is_terminal(self) -> bool:
        """Check if application is running from a terminal."""
        return hasattr(sys.stdout, 'isatty') and sys.stdout.isatty()
    
    def get_logger(self) -> logging.Logger:
        """Get the configured logger."""
        return self.logger


class HardwareManager:
    """Manages hardware I/O operations and detection."""
    
    def __init__(self, config: ConfigManager, logger: logging.Logger):
        self.config = config
        self.logger = logger
        self.available_relays: Set[int] = set()
        self.available_inputs: Set[int] = set()
        self.input_states: Dict[int, bool] = {}
        self.relay_states: Dict[int, bool] = {}
        self._lock = threading.RLock()
        
        self._detect_hardware()
    
    def _detect_hardware(self) -> None:
        """Detect available hardware based on file existence."""
        self.logger.info("Starting hardware detection...")
        
        # Reset available hardware
        self.available_relays.clear()
        self.available_inputs.clear()
        
        # Check relays (1-20)
        for i in range(1, 21):
            relay_path = f"/dev/gpio/rel{i}/value"
            if os.path.exists(relay_path):
                self.available_relays.add(i)
        
        # Check inputs (1-20)
        for i in range(1, 21):
            input_path = f"/dev/gpio/in{i}/value"
            if os.path.exists(input_path):
                self.available_inputs.add(i)
        
        # Determine UX8 extension count based on highest available number
        max_relay = max(self.available_relays) if self.available_relays else 0
        max_input = max(self.available_inputs) if self.available_inputs else 0
        max_io = max(max_relay, max_input)
        
        if max_io > 12:
            ux8_count = 2
        elif max_io > 4:
            ux8_count = 1
        else:
            ux8_count = 0
        
        self.logger.info(f"Hardware detection complete: {len(self.available_relays)} relays, "
                        f"{len(self.available_inputs)} inputs, {ux8_count} UX8 extensions")
        
        # Initialize input states
        for input_num in self.available_inputs:
            try:
                state = self._read_input(input_num)
                self.input_states[input_num] = state
            except Exception as e:
                self.logger.warning(f"Failed to read initial state for input {input_num}: {e}")
    
    def re_detect_hardware(self) -> bool:
        """Re-detect hardware and return True if changes occurred."""
        old_relays = self.available_relays.copy()
        old_inputs = self.available_inputs.copy()
        
        self._detect_hardware()
        
        return old_relays != self.available_relays or old_inputs != self.available_inputs
    
    def _read_input(self, input_num: int) -> bool:
        """Read digital input state."""
        if input_num not in self.available_inputs:
            raise ValueError(f"Input {input_num} not available")
        
        try:
            with open(f"/dev/gpio/in{input_num}/value", 'r') as f:
                value = f.read().strip()
                return value == '1'
        except Exception as e:
            self.logger.error(f"Failed to read input {input_num}: {e}")
            raise
    
    def _write_relay(self, relay_num: int, state: bool) -> None:
        """Write relay state."""
        if relay_num not in self.available_relays:
            raise ValueError(f"Relay {relay_num} not available")
        
        try:
            with open(f"/dev/gpio/rel{relay_num}/value", 'w') as f:
                f.write('1' if state else '0')
                f.flush()
            
            self.relay_states[relay_num] = state
            self.logger.info(f"Relay {relay_num} set to {'ON' if state else 'OFF'}")
            
        except Exception as e:
            self.logger.error(f"Failed to write relay {relay_num}: {e}")
            raise
    
    def set_relay(self, relay_num: int, state: bool) -> bool:
        """Set relay state with error handling."""
        with self._lock:
            try:
                if relay_num not in self.available_relays:
                    self.logger.warning(f"Attempt to control unavailable relay {relay_num}")
                    return False
                
                self._write_relay(relay_num, state)
                return True
                
            except Exception as e:
                self.logger.error(f"Failed to set relay {relay_num} to {state}: {e}")
                return False
    
    def get_relay_state(self, relay_num: int) -> Optional[bool]:
        """Get current relay state."""
        return self.relay_states.get(relay_num)
    
    def poll_inputs(self) -> Dict[int, bool]:
        """Poll all digital inputs and return state changes."""
        changes = {}
        
        with self._lock:
            for input_num in self.available_inputs:
                try:
                    current_state = self._read_input(input_num)
                    previous_state = self.input_states.get(input_num)
                    
                    if previous_state is None or current_state != previous_state:
                        changes[input_num] = current_state
                        self.input_states[input_num] = current_state
                        self.logger.debug(f"Input {input_num} changed to {'1' if current_state else '0'}")
                        
                except Exception as e:
                    self.logger.warning(f"Failed to poll input {input_num}: {e}")
        
        return changes
    
    def get_available_relays(self) -> Set[int]:
        """Get available relay numbers."""
        return self.available_relays.copy()
    
    def get_available_inputs(self) -> Set[int]:
        """Get available input numbers."""
        return self.available_inputs.copy()


class MQTTHandler:
    """Handles MQTT communication."""
    
    def __init__(self, config: ConfigManager, logger: logging.Logger, hardware: HardwareManager):
        self.config = config
        self.logger = logger
        self.hardware = hardware
        self.client = None
        self.connected = False
        self._setup_client()
    
    def _setup_client(self) -> None:
        """Setup MQTT client with configuration."""
        self.client = mqtt.Client(client_id=self.config.get('mqtt_client_id'))
        
        # Set credentials if provided
        username = self.config.get('mqtt_username')
        password = self.config.get('mqtt_password')
        if username and password:
            self.client.username_pw_set(username, password)
        
        # Set Last Will and Testament
        lwt_topic = self.config.get('mqtt_last_will_topic')
        if lwt_topic:
            self.client.will_set(
                lwt_topic,
                self.config.get('mqtt_last_will_payload', 'offline'),
                qos=self.config.get('mqtt_last_will_qos', 1),
                retain=self.config.get('mqtt_last_will_retain', True)
            )
        
        # Set callbacks
        self.client.on_connect = self._on_connect
        self.client.on_disconnect = self._on_disconnect
        self.client.on_message = self._on_message
        self.client.on_log = self._on_log
    
    def _on_log(self, client, userdata, level, buf):
        """MQTT client logging callback."""
        if level == mqtt.MQTT_LOG_DEBUG:
            self.logger.debug(f"MQTT: {buf}")
        elif level == mqtt.MQTT_LOG_INFO:
            self.logger.info(f"MQTT: {buf}")
        elif level == mqtt.MQTT_LOG_WARNING:
            self.logger.warning(f"MQTT: {buf}")
        elif level == mqtt.MQTT_LOG_ERR:
            self.logger.error(f"MQTT: {buf}")
    
    def _on_connect(self, client, userdata, flags, rc):
        """MQTT connection callback."""
        if rc == 0:
            self.connected = True
            self.logger.info("MQTT connected successfully")
            
            # Subscribe to relay control topics
            device_id = self.config.get('mqtt_device_id')
            for relay_num in self.hardware.get_available_relays():
                topic = f"barix/{device_id}/relay/{relay_num}/set"
                result = client.subscribe(topic, qos=self.config.get('mqtt_qos', 1))
                if result[0] == mqtt.MQTT_ERR_SUCCESS:
                    self.logger.debug(f"Subscribed to {topic}")
                else:
                    self.logger.warning(f"Failed to subscribe to {topic}: {result[0]}")
            
            # Publish online status
            self._publish_status("online")
            
            # Publish initial relay states
            self._publish_initial_states()
            
        else:
            self.logger.error(f"MQTT connection failed with code {rc}")
            self.connected = False
    
    def _on_disconnect(self, client, userdata, rc):
        """MQTT disconnection callback."""
        self.connected = False
        if rc != 0:
            self.logger.warning(f"MQTT disconnected unexpectedly (code: {rc})")
        else:
            self.logger.info("MQTT disconnected")
    
    def _on_message(self, client, userdata, msg):
        """MQTT message callback."""
        try:
            topic = msg.topic
            payload = msg.payload.decode('utf-8').strip()
            
            self.logger.debug(f"Received MQTT message: {topic} = {payload}")
            
            # Parse relay control messages
            device_id = self.config.get('mqtt_device_id')
            topic_prefix = f"barix/{device_id}/relay/"
            
            if topic.startswith(topic_prefix) and topic.endswith("/set"):
                # Extract relay number
                relay_part = topic[len(topic_prefix):-4]  # Remove "/set"
                try:
                    relay_num = int(relay_part)
                    self._handle_relay_command(relay_num, payload)
                except ValueError:
                    self.logger.warning(f"Invalid relay number in topic: {topic}")
            
        except Exception as e:
            self.logger.error(f"Error processing MQTT message: {e}")
    
    def _handle_relay_command(self, relay_num: int, payload: str) -> None:
        """Handle relay control command."""
        try:
            # Parse command
            state = None
            if payload.upper() in ['ON', '1']:
                state = True
            elif payload.upper() in ['OFF', '0']:
                state = False
            else:
                self.logger.warning(f"Invalid relay command payload: {payload}")
                return
            
            # Execute command
            if self.hardware.set_relay(relay_num, state):
                # Publish status feedback
                device_id = self.config.get('mqtt_device_id')
                status_topic = f"barix/{device_id}/relay/{relay_num}/status"
                status_payload = "ON" if state else "OFF"
                
                self.publish(status_topic, status_payload, retain=True)
                self.logger.info(f"Relay {relay_num} command executed: {status_payload}")
            else:
                self.logger.error(f"Failed to execute relay {relay_num} command")
                
        except Exception as e:
            self.logger.error(f"Error handling relay command: {e}")
    
    def _publish_status(self, status: str) -> None:
        """Publish device status."""
        device_id = self.config.get('mqtt_device_id')
        topic = f"barix/{device_id}/status"
        self.publish(topic, status, retain=True)
    
    def _publish_initial_states(self) -> None:
        """Publish initial relay and input states."""
        device_id = self.config.get('mqtt_device_id')
        
        # Publish relay states
        for relay_num in self.hardware.get_available_relays():
            state = self.hardware.get_relay_state(relay_num)
            if state is not None:
                topic = f"barix/{device_id}/relay/{relay_num}/status"
                payload = "ON" if state else "OFF"
                self.publish(topic, payload, retain=True)
        
        # Publish input states
        for input_num in self.hardware.get_available_inputs():
            state = self.hardware.input_states.get(input_num)
            if state is not None:
                topic = f"barix/{device_id}/input/{input_num}"
                payload = "1" if state else "0"
                self.publish(topic, payload, retain=False)
    
    def connect(self) -> bool:
        """Connect to MQTT broker."""
        try:
            self.client.connect(
                self.config.get('mqtt_broker'),
                self.config.get('mqtt_port'),
                self.config.get('mqtt_keepalive')
            )
            self.client.loop_start()
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to connect to MQTT broker: {e}")
            return False
    
    def disconnect(self) -> None:
        """Disconnect from MQTT broker."""
        if self.client:
            self._publish_status("offline")
            self.client.loop_stop()
            self.client.disconnect()
    
    def publish(self, topic: str, payload: str, qos: int = None, retain: bool = None) -> bool:
        """Publish MQTT message."""
        if not self.connected:
            self.logger.warning(f"Cannot publish to {topic}: MQTT not connected")
            return False
        
        try:
            if qos is None:
                qos = self.config.get('mqtt_qos', 1)
            if retain is None:
                retain = self.config.get('mqtt_retain', False)
            
            result = self.client.publish(topic, payload, qos=qos, retain=retain)
            if result.rc == mqtt.MQTT_ERR_SUCCESS:
                self.logger.debug(f"Published to {topic}: {payload}")
                return True
            else:
                self.logger.warning(f"Failed to publish to {topic}: {result.rc}")
                return False
                
        except Exception as e:
            self.logger.error(f"Error publishing to {topic}: {e}")
            return False
    
    def publish_input_change(self, input_num: int, state: bool) -> None:
        """Publish digital input state change."""
        device_id = self.config.get('mqtt_device_id')
        topic = f"barix/{device_id}/input/{input_num}"
        payload = "1" if state else "0"
        self.publish(topic, payload, retain=False)
    
    def is_connected(self) -> bool:
        """Check if MQTT is connected."""
        return self.connected


class ApplicationController:
    """Main application controller."""
    
    def __init__(self):
        self.config = ConfigManager()
        self.logging_manager = LoggingManager(self.config)
        self.logger = self.logging_manager.get_logger()
        
        # Log which configuration was used
        if self.config.is_using_fallback():
            self.logger.warning("Using fallback configuration (default_config.json)")
        else:
            self.logger.info("Using primary configuration (config.json)")
        
        self.hardware = HardwareManager(self.config, self.logger)
        self.mqtt = MQTTHandler(self.config, self.logger, self.hardware)
        
        self.running = False
        self.input_thread = None
        self.health_thread = None
        
        # Setup signal handlers
        signal.signal(signal.SIGTERM, self._signal_handler)
        signal.signal(signal.SIGINT, self._signal_handler)
    
    def _signal_handler(self, signum, frame):
        """Handle shutdown signals."""
        self.logger.info(f"Received signal {signum}, shutting down...")
        self.stop()
    
    def start(self) -> None:
        """Start the application."""
        self.logger.info("Starting Barionet MQTT application...")
        
        self.running = True
        
        # Connect to MQTT with retry logic
        self._connect_mqtt_with_retry()
        
        if not self.mqtt.is_connected():
            self.logger.critical("Failed to connect to MQTT broker, exiting")
            sys.exit(1)
        
        # Start input monitoring thread
        self.input_thread = threading.Thread(target=self._input_monitor_loop, daemon=True)
        self.input_thread.start()
        
        # Start health check thread
        self.health_thread = threading.Thread(target=self._health_check_loop, daemon=True)
        self.health_thread.start()
        
        self.logger.info("Barionet MQTT application started successfully")
        
        # Main loop
        try:
            while self.running:
                time.sleep(1)
                
                # Check MQTT connection and reconnect if needed
                if not self.mqtt.is_connected():
                    self.logger.warning("MQTT connection lost, attempting to reconnect...")
                    self._connect_mqtt_with_retry()
                    
        except KeyboardInterrupt:
            self.logger.info("Keyboard interrupt received")
        except Exception as e:
            self.logger.critical(f"Unexpected error in main loop: {e}")
        finally:
            self.stop()
    
    def _connect_mqtt_with_retry(self) -> None:
        """Connect to MQTT with exponential backoff retry."""
        max_attempts = self.config.get('application_max_reconnect_attempts', 10)
        base_delay = self.config.get('application_reconnect_delay', 5.0)
        
        for attempt in range(max_attempts):
            if not self.running:
                break
                
            self.logger.info(f"MQTT connection attempt {attempt + 1}/{max_attempts}")
            
            if self.mqtt.connect():
                # Wait for connection to be established with timeout
                connection_timeout = 10  # seconds
                for i in range(connection_timeout * 10):  # Check every 100ms
                    if self.mqtt.is_connected():
                        self.logger.info("MQTT connection established successfully")
                        return
                    time.sleep(0.1)
                self.logger.warning("MQTT connection timeout - connection not confirmed")
            
            if attempt < max_attempts - 1:
                delay = min(base_delay * (2 ** attempt), 300.0)  # Cap at 5 minutes
                self.logger.warning(f"MQTT connection failed, retrying in {delay:.1f}s...")
                time.sleep(delay)
        
        self.logger.error("Failed to connect to MQTT broker after all attempts")
    
    def _input_monitor_loop(self) -> None:
        """Monitor digital inputs for state changes."""
        self.logger.info("Starting input monitoring loop")
        poll_interval = self.config.get('hardware_poll_interval', 0.1)
        
        while self.running:
            try:
                changes = self.hardware.poll_inputs()
                
                for input_num, state in changes.items():
                    self.mqtt.publish_input_change(input_num, state)
                    self.logger.info(f"Input {input_num} state changed to {'1' if state else '0'}")
                
                time.sleep(poll_interval)
                
            except Exception as e:
                self.logger.error(f"Error in input monitoring loop: {e}")
                time.sleep(1)  # Avoid tight error loop
    
    def _health_check_loop(self) -> None:
        """Perform periodic health checks."""
        interval = self.config.get('application_health_check_interval', 300)
        
        while self.running:
            try:
                time.sleep(interval)
                
                if not self.running:
                    break
                
                # Health check: verify hardware is still accessible
                available_relays = len(self.hardware.get_available_relays())
                available_inputs = len(self.hardware.get_available_inputs())
                
                self.logger.info(f"Health check: {available_relays} relays, "
                               f"{available_inputs} inputs, "
                               f"MQTT connected: {self.mqtt.is_connected()}")
                
            except Exception as e:
                self.logger.error(f"Error in health check: {e}")
    
    def stop(self) -> None:
        """Stop the application gracefully."""
        if not self.running:
            return
            
        self.logger.info("Stopping Barionet MQTT application...")
        self.running = False
        
        # Disconnect MQTT
        self.mqtt.disconnect()
        
        # Wait for threads to finish
        if self.input_thread and self.input_thread.is_alive():
            self.input_thread.join(timeout=5)
        
        if self.health_thread and self.health_thread.is_alive():
            self.health_thread.join(timeout=5)
        
        self.logger.info("Barionet MQTT application stopped")


def main():
    """Main entry point."""
    try:
        app = ApplicationController()
        app.start()
    except Exception as e:
        print(f"CRITICAL: Failed to start application: {e}")
        sys.exit(1)


if __name__ == "__main__":
    main()