#!/usr/bin/env python3
"""
IO Tunnel Application - Filesystem GPIO Version

A bidirectional UDP tunnel for IO state synchronization between two endpoints.
Monitors local digital inputs via filesystem and transmits their state to a remote peer while
listening for remote input states to trigger local outputs.

Version: 2.2.0: 
- Increased GPIO handling speed
"""

import json
import socket
import threading
import time
import logging
import signal
import sys
import os
import select
from typing import Dict, List, Optional, Tuple
from dataclasses import dataclass

@dataclass
class AppConfig:
    """Application configuration data structure."""
    io_tunnel_enable: bool
    tunneling_port: int
    remote_target_address: str
    input_triggers: Dict[int, int]  # input_num -> relay_address (0 = no action)
    ux8_input_triggers: Dict[int, int]  # ux8_input_num -> relay_address (0 = no action)
    send_interval: int
    comm_loss_trigger: int
    startup_interval: int
    enable_syslog: bool
    syslog_address: str

    @classmethod
    def from_dict(cls, config_dict: dict) -> 'AppConfig':
        """Create AppConfig from dictionary with validation."""
        app_param = config_dict.get('AppParam', {})
        
        def validate_relay_address(value: str) -> int:
            """Validate relay address range (0, 1-4, 5-12)."""
            if not value.isdigit():
                return 0
            val = int(value)
            return val if (val == 0 or (1 <= val <= 4) or (5 <= val <= 12)) else 0
        
        # Parse original input triggers (inputs 1-4)
        input_triggers = {}
        for i in range(1, 5):
            key = f'input{i}_trigger'
            value = app_param.get(key, '0')
            input_triggers[i] = validate_relay_address(value)
        
        # Parse UX8 input triggers (inputs 1-8)
        ux8_input_triggers = {}
        for i in range(1, 9):
            key = f'ux8_input{i}_trigger'
            value = app_param.get(key, '0')
            ux8_input_triggers[i] = validate_relay_address(value)
        
        # Parse comm_loss_trigger
        comm_loss_trigger = validate_relay_address(app_param.get('comm_loss_trigger', '0'))
        
        # Parse and validate intervals
        send_interval = max(0, int(app_param.get('send_interval', 0)))
        startup_interval = max(0, int(app_param.get('startup_interval', 0)))
        
        # Validate tunneling port
        tunneling_port = int(app_param.get('tunneling_port', 12301))
        if not (1 <= tunneling_port <= 65535):
            tunneling_port = 12301
        
        return cls(
            io_tunnel_enable=bool(app_param.get('IO_tunnel_enable', False)),
            tunneling_port=tunneling_port,
            remote_target_address=str(app_param.get('remote_target_address', '')).strip(),
            input_triggers=input_triggers,
            ux8_input_triggers=ux8_input_triggers,
            send_interval=send_interval,
            comm_loss_trigger=comm_loss_trigger,
            startup_interval=startup_interval,
            enable_syslog=bool(app_param.get('enable_syslog', False)),
            syslog_address=str(app_param.get('syslog_address', '')).strip()
        )


class ConfigLoader:
    """Handles configuration loading with fallback mechanism."""
    
    def __init__(self, config_file: str = 'config.json', fallback_file: str = 'default_config.json'):
        self.config_file = config_file
        self.fallback_file = fallback_file
        self.logger = logging.getLogger(__name__)
    
    def load_config(self) -> AppConfig:
        """Load configuration with fallback mechanism."""
        config_data = None
        
        # Try primary config file
        if os.path.exists(self.config_file):
            try:
                with open(self.config_file, 'r') as f:
                    config_data = json.load(f)
                self.logger.info(f"Loaded configuration from {self.config_file}")
            except (json.JSONDecodeError, IOError) as e:
                self.logger.error(f"Failed to load {self.config_file}: {e}")
        
        # Fallback to default config
        if config_data is None and os.path.exists(self.fallback_file):
            try:
                with open(self.fallback_file, 'r') as f:
                    config_data = json.load(f)
                self.logger.info(f"Loaded fallback configuration from {self.fallback_file}")
            except (json.JSONDecodeError, IOError) as e:
                self.logger.error(f"Failed to load {self.fallback_file}: {e}")
        
        if config_data is None:
            raise RuntimeError("No valid configuration file found")
        
        return AppConfig.from_dict(config_data)


class MessageProtocol:
    """Handles message parsing and formatting."""
    
    @staticmethod
    def format_message(send_interval: int, input_states: List[int]) -> str:
        """Format message according to protocol: SSSSS,000000000000"""
        interval_str = f"{send_interval:05d}"
        
        # Ensure we have 12 input states, pad with zeros if needed
        states = (input_states + [0] * 12)[:12]
        states_str = ''.join(str(min(max(state, 0), 1)) for state in states)  # Clamp to 0-1
        
        return f"{interval_str},{states_str}"
    
    @staticmethod
    def parse_message(message: str) -> Optional[Tuple[int, List[int]]]:
        """Parse incoming message. Returns (send_interval, input_states) or None if invalid."""
        try:
            if ',' not in message:
                return None
            
            parts = message.split(',', 1)  # Split only on first comma
            if len(parts) != 2:
                return None
            
            interval_str, states_str = parts
            
            # Validate interval format
            if len(interval_str) != 5 or not interval_str.isdigit():
                return None
            
            send_interval = int(interval_str)
            
            # Support both 8-digit (legacy) and 12-digit (current) formats
            if len(states_str) == 8 and states_str.isdigit():
                # Legacy format: extend to 12 digits
                original_inputs = [int(states_str[i]) for i in range(4)]
                ux8_inputs_1_4 = [int(states_str[i]) for i in range(4, 8)]
                ux8_inputs_5_8 = [0] * 4
                input_states = original_inputs + ux8_inputs_1_4 + ux8_inputs_5_8
                
            elif len(states_str) == 12 and states_str.isdigit():
                # Current format: direct mapping
                input_states = [int(states_str[i]) for i in range(12)]
                
            else:
                return None
            
            return send_interval, input_states
            
        except (ValueError, IndexError):
            return None


class HighSpeedGPIOHandler:
    """High-speed GPIO handler with cached file handles and parallel processing."""
    
    def __init__(self):
        self.logger = logging.getLogger(__name__)
        self.input_handles: Dict[int, object] = {}
        self.output_handles: Dict[int, object] = {}
        self.input_paths: Dict[int, str] = {}
        self.output_paths: Dict[int, str] = {}
        self._handles_lock = threading.Lock()
        self._discover_and_cache_gpio()
    
    def _discover_and_cache_gpio(self):
        """Discover GPIO devices and cache file handles for maximum speed."""
        # Discover and cache input handles
        for i in range(1, 13):
            input_path = f"/dev/gpio/in{i}/value"
            if os.path.exists(input_path):
                try:
                    # Keep file handles open for fastest access
                    handle = open(input_path, 'r')
                    self.input_handles[i] = handle
                    self.input_paths[i] = input_path
                    
                    if i <= 4:
                        self.logger.info(f"Cached built-in input {i}")
                    else:
                        self.logger.info(f"Cached UX8 input {i}")
                except Exception as e:
                    self.logger.error(f"Failed to cache input {i}: {e}")
        
        # Discover and cache output handles
        for i in range(1, 13):
            output_path = f"/dev/gpio/rel{i}/value"
            if os.path.exists(output_path):
                try:
                    # Keep file handles open for fastest access
                    handle = open(output_path, 'w')
                    self.output_handles[i] = handle
                    self.output_paths[i] = output_path
                    
                    if i <= 4:
                        self.logger.info(f"Cached built-in relay {i}")
                    else:
                        self.logger.info(f"Cached UX8 relay {i}")
                except Exception as e:
                    self.logger.error(f"Failed to cache output {i}: {e}")
        
        self.logger.info(f"GPIO cache ready: {len(self.input_handles)} inputs, {len(self.output_handles)} outputs")
    
    def read_input_fast(self, input_num: int) -> Optional[int]:
        """Ultra-fast input read using cached handle."""
        handle = self.input_handles.get(input_num)
        if not handle:
            return None
        
        try:
            handle.seek(0)  # Reset to beginning
            value = int(handle.read().strip())
            return min(max(value, 0), 1)
        except Exception:
            return None
    
    def write_output_fast(self, output_num: int, value: int) -> bool:
        """Ultra-fast output write using cached handle."""
        handle = self.output_handles.get(output_num)
        if not handle:
            return False
        
        try:
            handle.seek(0)
            handle.write(str(min(max(value, 0), 1)))
            handle.flush()  # Ensure immediate write
            return True
        except Exception:
            return False
    
    def read_all_inputs_fast(self) -> List[int]:
        """Read all inputs with maximum speed."""
        states = [0] * 12
        
        for input_num in self.input_handles.keys():
            value = self.read_input_fast(input_num)
            if value is not None:
                states[input_num - 1] = value
        
        return states
    
    def get_available_inputs(self) -> List[int]:
        """Get list of available input numbers."""
        return list(self.input_handles.keys())
    
    def get_available_outputs(self) -> List[int]:
        """Get list of available output numbers."""
        return list(self.output_handles.keys())
    
    def close_handles(self):
        """Close all cached file handles."""
        with self._handles_lock:
            for handle in self.input_handles.values():
                try:
                    handle.close()
                except Exception:
                    pass
            
            for handle in self.output_handles.values():
                try:
                    handle.close()
                except Exception:
                    pass
            
            self.input_handles.clear()
            self.output_handles.clear()

class SyslogHandler:
    """Handles syslog message transmission."""
    
    def __init__(self, syslog_address: str, enabled: bool = False):
        self.enabled = False
        self.syslog_address = syslog_address
        self.logger = logging.getLogger(__name__)
        self.sock = None
        self.syslog_target = None
        self._lock = threading.Lock()
        
        if enabled and syslog_address and syslog_address.strip():
            self._setup_syslog()
    
    def _setup_syslog(self):
        """Setup UDP socket for syslog transmission."""
        try:
            if not self.syslog_address.strip():
                self.logger.warning("Syslog address is empty, syslog disabled")
                return
            
            # Parse address and port
            if ':' in self.syslog_address:
                host, port_str = self.syslog_address.rsplit(':', 1)
                if not host.strip() or not port_str.strip():
                    self.logger.error("Invalid syslog address format, syslog disabled")
                    return
                
                try:
                    port = int(port_str)
                    if not (1 <= port <= 65535):
                        self.logger.error(f"Invalid syslog port {port}, syslog disabled")
                        return
                except ValueError:
                    self.logger.error(f"Invalid syslog port '{port_str}', syslog disabled")
                    return
                
                self.syslog_target = (host.strip(), port)
            else:
                host = self.syslog_address.strip()
                if not host:
                    self.logger.error("Invalid syslog address, syslog disabled")
                    return
                self.syslog_target = (host, 514)
            
            # Create and bind socket
            self.sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            self.sock.bind(('0.0.0.0', 0))
            bound_port = self.sock.getsockname()[1]
            
            self.logger.info(f"Syslog enabled: port {bound_port}, target: {self.syslog_target}")
            self.enabled = True
            
        except Exception as e:
            self.logger.error(f"Failed to setup syslog: {e}, syslog disabled")
            self.enabled = False
            if self.sock:
                self.sock.close()
                self.sock = None
    
    def send_syslog(self, message: str, facility: int = 16, severity: int = 6):
        """Send syslog message with proper format."""
        if not self.enabled or not self.sock or not self.syslog_target:
            return
        
        try:
            with self._lock:
                priority = facility * 8 + severity
                timestamp = time.strftime('%b %d %H:%M:%S')
                hostname = socket.gethostname()
                tag = 'io_tunnel'
                
                syslog_msg = f"<{priority}>{timestamp} {hostname} {tag}: {message}"
                self.sock.sendto(syslog_msg.encode('utf-8'), self.syslog_target)
                
        except Exception as e:
            self.logger.error(f"Failed to send syslog message: {e}")
    
    def close(self):
        """Close syslog socket."""
        with self._lock:
            if self.sock:
                self.sock.close()
                self.sock = None
            self.enabled = False


class CommWatchdog:
    """Communication loss detection and handling."""
    
    def __init__(self, comm_loss_trigger: int, gpio_handler: HighSpeedGPIOHandler, syslog_handler: SyslogHandler):
        self.comm_loss_trigger = comm_loss_trigger
        self.gpio_handler = gpio_handler
        self.syslog_handler = syslog_handler
        self.logger = logging.getLogger(__name__)
        self.timer = None
        self.lock = threading.RLock()
        self.comm_lost = False
        self.relay_triggered = False
        self.startup_watchdog_active = False
    
    def start_startup_watchdog(self, startup_interval: int):
        """Start watchdog using startup_interval at application startup."""
        with self.lock:
            if startup_interval > 0:
                timeout = (startup_interval * 2) + 1
                self.timer = threading.Timer(timeout, self._on_timeout)
                self.timer.start()
                self.startup_watchdog_active = True
                self.logger.info(f"Started startup watchdog: {timeout}s timeout (startup_interval={startup_interval})")
            else:
                self.logger.info("Startup watchdog disabled (startup_interval=0)")
    
    def reset_watchdog(self, timeout_seconds: int):
        """Reset the communication watchdog timer with peer's send_interval."""
        with self.lock:
            # Handle transition from startup to peer-based watchdog
            if self.startup_watchdog_active:
                self.startup_watchdog_active = False
                self.logger.info("Switched from startup to peer-based watchdog")
            
            if self.timer:
                self.timer.cancel()
            
            if timeout_seconds > 0:
                # Check if communication was previously lost and restore it
                if self.comm_lost:
                    self._on_communication_restored()
                
                self.timer = threading.Timer(timeout_seconds, self._on_timeout)
                self.timer.start()
                self.logger.debug(f"Watchdog reset: {timeout_seconds}s timeout")
            else:
                # Peer disabled periodic transmission, restore any loss state
                if self.comm_lost or self.relay_triggered:
                    self.logger.info("Peer disabled transmission, restoring communication state")
                    self._on_communication_restored()
                
                self.logger.debug("Watchdog disabled (peer send_interval=0)")
    
    def stop_watchdog(self):
        """Stop the watchdog timer."""
        with self.lock:
            if self.timer:
                self.timer.cancel()
                self.timer = None
            self.startup_watchdog_active = False
    
    def _on_timeout(self):
        """Handle communication timeout."""
        with self.lock:
            if self.comm_lost:
                return  # Already handled
            
            self.comm_lost = True
            
            if self.startup_watchdog_active:
                msg = "Startup communication timeout (no peer contact)"
            else:
                msg = "Communication timeout detected"
            
            self.logger.warning(msg)
            self.syslog_handler.send_syslog(msg, severity=4)
            
            # Trigger communication loss relay if configured
            if (1 <= self.comm_loss_trigger <= 4) or (5 <= self.comm_loss_trigger <= 12):
                if self.gpio_handler.write_output_fast(self.comm_loss_trigger, 1):
                    self.relay_triggered = True
                    self.logger.info(f"Triggered comm loss relay: {self.comm_loss_trigger}")
                    self.syslog_handler.send_syslog(f"Communication loss triggered relay {self.comm_loss_trigger}")
                else:
                    self.logger.error(f"Failed to trigger comm loss relay: {self.comm_loss_trigger}")
    
    def _on_communication_restored(self):
        """Handle communication restoration after a loss."""
        if not self.comm_lost:
            return
        
        self.logger.info("Communication restored")
        self.syslog_handler.send_syslog("Communication restored")
        
        # Reset the comm loss relay if it was triggered
        if self.relay_triggered and ((1 <= self.comm_loss_trigger <= 4) or (5 <= self.comm_loss_trigger <= 12)):
            if self.gpio_handler.write_output_fast(self.comm_loss_trigger, 0):
                self.logger.info(f"Reset comm loss relay: {self.comm_loss_trigger}")
                self.syslog_handler.send_syslog(f"Communication restored - reset relay {self.comm_loss_trigger}")
            else:
                self.logger.error(f"Failed to reset comm loss relay: {self.comm_loss_trigger}")
        
        # Reset state flags
        self.comm_lost = False
        self.relay_triggered = False


class ThreadManager:
    """Manages thread lifecycle and exception handling."""
    
    def __init__(self, logger):
        self.logger = logger
        self.threads: List[threading.Thread] = []
        self.shutdown_event = threading.Event()
    
    def start_thread(self, target, name: str, daemon: bool = True):
        """Start a managed thread with exception handling."""
        def wrapped_target():
            try:
                target()
            except Exception as e:
                self.logger.error(f"Thread {name} crashed: {e}", exc_info=True)
                # Signal shutdown on critical thread failure
                self.shutdown_event.set()
        
        thread = threading.Thread(target=wrapped_target, name=name, daemon=daemon)
        thread.start()
        self.threads.append(thread)
        self.logger.info(f"Started thread: {name}")
        return thread
    
    def stop_all(self, timeout: float = 2.0):
        """Stop all managed threads."""
        self.shutdown_event.set()
        
        for thread in self.threads:
            if thread.is_alive():
                thread.join(timeout=timeout)
                if thread.is_alive():
                    self.logger.warning(f"Thread {thread.name} did not stop within {timeout}s")
    
    def is_shutdown_requested(self) -> bool:
        """Check if shutdown has been requested."""
        return self.shutdown_event.is_set()


class IOTunnelServer:
    """Main application class."""
    
    def __init__(self):
        self.config = None
        self.gpio_handler = None
        self.udp_socket = None
        self.syslog_handler = None
        self.watchdog = None
        self.thread_manager = None
        self.running = False
        self.logger = logging.getLogger(__name__)
        
        # Input state tracking with improved thread safety
        self.current_input_states = [0] * 12
        self.remote_relay_states = [0] * 12
        self.state_lock = threading.RLock()
        
        # Setup signal handlers
        signal.signal(signal.SIGINT, self._signal_handler)
        signal.signal(signal.SIGTERM, self._signal_handler)
    
    def _signal_handler(self, signum, frame):
        """Handle shutdown signals."""
        self.logger.info(f"Received signal {signum}, shutting down...")
        self.stop()
    
    def start(self):
        """Start the IO tunnel service."""
        try:
            # Load configuration
            config_loader = ConfigLoader()
            self.config = config_loader.load_config()
            
            if not self.config.io_tunnel_enable:
                self.logger.info("IO tunnel disabled in configuration")
                return
            
            # Initialize components
            self._setup_logging()
            self._setup_syslog()
            self._setup_gpio_handler()
            self._setup_udp_socket()
            self._setup_watchdog()
            self._setup_thread_manager()
            
            self.running = True
            
            # Start threads
            self._start_threads()
            
            # Start startup watchdog
            self.watchdog.start_startup_watchdog(self.config.startup_interval)
            
            self.logger.info("IO Tunnel service started successfully")
            self.syslog_handler.send_syslog("IO Tunnel service started")
            
            # Main loop with proper shutdown handling
            try:
                while self.running and not self.thread_manager.is_shutdown_requested():
                    time.sleep(1)
            except KeyboardInterrupt:
                pass
            
        except Exception as e:
            self.logger.error(f"Failed to start service: {e}", exc_info=True)
            raise
        finally:
            self.stop()
    
    def _setup_logging(self):
        """Setup application logging."""
        handlers = []
        
        # Only log to console if we have a terminal attached
        if os.isatty(sys.stdout.fileno()):
            handlers.append(logging.StreamHandler(sys.stdout))
        else:
            handlers.append(logging.NullHandler())
        
        logging.basicConfig(
            level=logging.INFO,  # Changed back to INFO for production
            format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
            handlers=handlers
        )
    
    def _setup_syslog(self):
        """Setup syslog handler."""
        self.syslog_handler = SyslogHandler(
            self.config.syslog_address,
            self.config.enable_syslog
        )
    
    def _setup_gpio_handler(self):
        """Setup GPIO handler and initialize outputs."""
        try:
            self.gpio_handler = HighSpeedGPIOHandler()
            
            # Initialize all available outputs to 0
            available_outputs = self.gpio_handler.get_available_outputs()
            initialized_count = 0
            
            for output_num in available_outputs:
                if self.gpio_handler.write_output_fast(output_num, 0):
                    initialized_count += 1
                else:
                    self.logger.warning(f"Could not initialize output {output_num}")
            
            self.logger.info(f"GPIO initialized: {initialized_count} outputs reset")
            self.syslog_handler.send_syslog(f"GPIO initialized: {initialized_count} outputs reset")
            
            # Read initial input states
            with self.state_lock:
                self.current_input_states = self.gpio_handler.read_all_inputs_fast()
            
            self.logger.info(f"Initial input states: {self.current_input_states}")
            
        except Exception as e:
            self.logger.error(f"Failed to initialize GPIO: {e}")
            raise
    
    def _setup_udp_socket(self):
        """Setup UDP socket for communication."""
        try:
            self.udp_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            self.udp_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            self.udp_socket.setblocking(False)
            self.udp_socket.bind(('0.0.0.0', self.config.tunneling_port))
            self.udp_socket.settimeout(1.0)
            
            self.logger.info(f"UDP socket bound to port {self.config.tunneling_port}")
            
        except Exception as e:
            self.logger.error(f"Failed to setup UDP socket: {e}")
            raise
    
    def _setup_watchdog(self):
        """Setup communication watchdog."""
        self.watchdog = CommWatchdog(
            self.config.comm_loss_trigger,
            self.gpio_handler,
            self.syslog_handler
        )
    
    def _setup_thread_manager(self):
        """Setup thread manager."""
        self.thread_manager = ThreadManager(self.logger)
    
    def _start_threads(self):
        """Start worker threads."""
        # Start UDP receive thread
        self.thread_manager.start_thread(
            target=self._receive_loop, 
            name="UDP-Receiver"
        )
        
        # Start input monitoring thread
        self.thread_manager.start_thread(
            target=self._input_monitoring_loop, 
            name="Input-Monitor"
        )
        
        # Start periodic send thread if enabled
        if self.config.send_interval > 0:
            self.thread_manager.start_thread(
                target=self._periodic_send_loop, 
                name="Periodic-Sender"
            )
    
    def _input_monitoring_loop(self):
        """Ultra-fast input monitoring with 10ms polling."""
        self.logger.info("Started ultra-fast input monitoring (10ms polling)")
        
        last_states = [0] * 12
        consecutive_errors = 0
        max_errors = 100
        
        while self.running and not self.thread_manager.is_shutdown_requested():
            try:
                # Read all inputs with maximum speed
                current_states = self.gpio_handler.read_all_inputs_fast()
                
                # Detect changes efficiently
                changes = []
                for i in range(12):
                    if current_states[i] != last_states[i]:
                        changes.append((i, last_states[i], current_states[i]))
                
                if changes:
                    # Update state atomically
                    with self.state_lock:
                        self.current_input_states = current_states.copy()
                    
                    # Send state update immediately for any change
                    self._send_state_update()
                    
                    # Log changes efficiently
                    for i, old_val, new_val in changes:
                        if i < 4:
                            input_type, input_num = "built-in", i + 1
                        else:
                            input_type, input_num = "UX8", i + 1
                        
                        self.logger.info(f"{input_type} input {input_num} changed: {old_val} -> {new_val}")
                        self.syslog_handler.send_syslog(f"{input_type} input {input_num} changed to {new_val}")
                    
                    # Update last known states
                    last_states = current_states.copy()
                
                consecutive_errors = 0
                
                # Ultra-fast polling - 10ms for rapid detection
                time.sleep(0.01)
                
            except Exception as e:
                consecutive_errors += 1
                if consecutive_errors <= max_errors and consecutive_errors % 10 == 1:
                    self.logger.error(f"Input monitor error (#{consecutive_errors}): {e}")
                
                if consecutive_errors >= max_errors:
                    self.logger.error("Too many input monitor errors, stopping")
                    break
                
                time.sleep(0.1)  # Longer sleep on error
    
    def _send_state_update(self):
        """Send current input state to remote target."""
        if not self.config.remote_target_address:
            return
        
        try:
            with self.state_lock:
                states = self.current_input_states.copy()
            
            message = MessageProtocol.format_message(self.config.send_interval, states)
            
            # Parse target address with validation
            target = self._parse_target_address(self.config.remote_target_address)
            if not target:
                self.logger.error("Invalid remote target address")
                return
            
            self.udp_socket.sendto(message.encode('utf-8'), target)
            self.logger.debug(f"Sent state update to {target}: {message}")
            
        except Exception as e:
            self.logger.error(f"Failed to send state update: {e}")
    
    def _parse_target_address(self, address: str) -> Optional[Tuple[str, int]]:
        """Parse target address with proper validation."""
        try:
            if ':' in address:
                host, port_str = address.rsplit(':', 1)
                port = int(port_str)
                if not (1 <= port <= 65535):
                    return None
                return (host.strip(), port)
            else:
                # Default to tunneling port
                return (address.strip(), self.config.tunneling_port)
        except (ValueError, AttributeError):
            return None
    
    def _receive_loop(self):
        """High-speed non-blocking UDP receiver."""
        self.logger.info("Started high-speed UDP receiver")
        consecutive_errors = 0
        max_errors = 50
        
        while self.running and not self.thread_manager.is_shutdown_requested():
            try:
                # Use select for non-blocking receive with 1ms timeout
                ready, _, _ = select.select([self.udp_socket], [], [], 0.001)
                
                if ready:
                    try:
                        data, addr = self.udp_socket.recvfrom(1024)
                        message = data.decode('utf-8').strip()
                        
                        self.logger.debug(f"Received from {addr}: {message}")
                        # Process message immediately for minimum latency
                        self._process_incoming_message_fast(message)
                        
                    except socket.error:
                        pass  # Ignore socket errors in non-blocking mode
                
                consecutive_errors = 0
                
            except Exception as e:
                consecutive_errors += 1
                if consecutive_errors <= max_errors and consecutive_errors % 10 == 1:
                    self.logger.error(f"UDP receiver error (#{consecutive_errors}): {e}")
                
                if consecutive_errors >= max_errors:
                    self.logger.error("Too many UDP receiver errors, stopping")
                    break
    
    def _process_incoming_message(self, message: str):
        """Process incoming UDP message."""
        try:
            parsed = MessageProtocol.parse_message(message)
            if parsed is None:
                self.logger.warning(f"Invalid message format: {message}")
                return
            
            send_interval, remote_input_states = parsed
            
            self.logger.info(f"Processed message: interval={send_interval}, states={remote_input_states}")
            self.syslog_handler.send_syslog(f"Received remote data: {message}")
            
            # Handle watchdog reset
            if send_interval > 0:
                timeout = (send_interval * 2) + 1
                self.watchdog.reset_watchdog(timeout)
            else:
                self.watchdog.reset_watchdog(0)
            
            # Process remote input triggers
            self._process_remote_triggers(remote_input_states)
            
        except Exception as e:
            self.logger.error(f"Error processing message: {e}")
    
    def _process_remote_triggers(self, remote_states: List[int]):
        """Process remote input states and trigger local relays."""
        try:
            # Process original inputs (positions 0-3)
            for i, remote_state in enumerate(remote_states[:4]):
                input_num = i + 1
                trigger_address = self.config.input_triggers.get(input_num, 0)
                
                if trigger_address == 0:
                    continue
                
                # Track state changes
                prev_state = self.remote_relay_states[i]
                self.remote_relay_states[i] = remote_state
                
                if remote_state != prev_state:
                    if self.gpio_handler.write_output_fast(trigger_address, remote_state):
                        action = "activated" if remote_state else "deactivated"
                        self.logger.info(f"Remote input {input_num} {action} -> relay {trigger_address}")
                        self.syslog_handler.send_syslog(f"Relay {trigger_address} {action} by remote input {input_num}")
                    else:
                        self.logger.error(f"Failed to control relay {trigger_address} for remote input {input_num}")
            
            # Process UX8 inputs (positions 4-11)
            if len(remote_states) >= 12:
                for i, remote_state in enumerate(remote_states[4:12]):
                    ux8_input_num = i + 1  # UX8 inputs 1-8
                    trigger_address = self.config.ux8_input_triggers.get(ux8_input_num, 0)
                    
                    if trigger_address == 0:
                        continue
                    
                    # Track state changes
                    array_index = i + 4  # Position in remote_relay_states array
                    prev_state = self.remote_relay_states[array_index]
                    self.remote_relay_states[array_index] = remote_state
                    
                    if remote_state != prev_state:
                        if self.gpio_handler.write_output_fast(trigger_address, remote_state):
                            action = "activated" if remote_state else "deactivated"
                            self.logger.info(f"Remote UX8 input {ux8_input_num} {action} -> relay {trigger_address}")
                            self.syslog_handler.send_syslog(f"Relay {trigger_address} {action} by remote UX8 input {ux8_input_num}")
                        else:
                            self.logger.error(f"Failed to control relay {trigger_address} for remote UX8 input {ux8_input_num}")
            
        except Exception as e:
            self.logger.error(f"Error processing remote triggers: {e}")

    def _process_incoming_message_fast(self, message: str):
        """Process incoming message with minimal latency."""
        try:
            parsed = MessageProtocol.parse_message(message)
            if parsed is None:
                return
            
            send_interval, remote_input_states = parsed
            
            # Handle watchdog immediately
            if send_interval > 0:
                timeout = (send_interval * 2) + 1
                self.watchdog.reset_watchdog(timeout)
            else:
                self.watchdog.reset_watchdog(0)
            
            # Process GPIO writes immediately for minimum latency
            self._process_remote_triggers_fast(remote_input_states)
            
        except Exception as e:
            self.logger.error(f"Error processing message: {e}")
    
    def _process_remote_triggers_fast(self, remote_states: List[int]):
        """Process remote input states with minimum latency."""
        try:
            # Process original inputs (positions 0-3) with immediate GPIO writes
            for i, remote_state in enumerate(remote_states[:4]):
                input_num = i + 1
                trigger_address = self.config.input_triggers.get(input_num, 0)
                
                if trigger_address == 0:
                    continue
                
                # Check if state changed
                prev_state = self.remote_relay_states[i]
                if remote_state != prev_state:
                    self.remote_relay_states[i] = remote_state
                    
                    # Immediate GPIO write for minimum latency
                    if self.gpio_handler.write_output_fast(trigger_address, remote_state):
                        action = "activated" if remote_state else "deactivated"
                        self.logger.info(f"Remote input {input_num} {action} -> relay {trigger_address}")
                        self.syslog_handler.send_syslog(f"Relay {trigger_address} {action} by remote input {input_num}")
                    else:
                        self.logger.error(f"Failed to control relay {trigger_address} for remote input {input_num}")
            
            # Process UX8 inputs (positions 4-11)
            if len(remote_states) >= 12:
                for i, remote_state in enumerate(remote_states[4:12]):
                    ux8_input_num = i + 1
                    trigger_address = self.config.ux8_input_triggers.get(ux8_input_num, 0)
                    
                    if trigger_address == 0:
                        continue
                    
                    # Check if state changed
                    array_index = i + 4
                    prev_state = self.remote_relay_states[array_index]
                    if remote_state != prev_state:
                        self.remote_relay_states[array_index] = remote_state
                        
                        # Immediate GPIO write for minimum latency
                        if self.gpio_handler.write_output_fast(trigger_address, remote_state):
                            action = "activated" if remote_state else "deactivated"
                            self.logger.info(f"Remote UX8 input {ux8_input_num} {action} -> relay {trigger_address}")
                            self.syslog_handler.send_syslog(f"Relay {trigger_address} {action} by remote UX8 input {ux8_input_num}")
                        else:
                            self.logger.error(f"Failed to control relay {trigger_address} for remote UX8 input {ux8_input_num}")
            
        except Exception as e:
            self.logger.error(f"Error processing remote triggers: {e}")    
    
    def _periodic_send_loop(self):
        """Periodic state transmission loop."""
        self.logger.info(f"Started periodic send loop: {self.config.send_interval}s interval")
        
        while self.running and not self.thread_manager.is_shutdown_requested():
            try:
                # Sleep with interrupt checking
                for _ in range(self.config.send_interval * 10):  # 100ms increments
                    if not self.running or self.thread_manager.is_shutdown_requested():
                        return
                    time.sleep(0.1)
                
                if self.running:
                    self._send_state_update()
                    
            except Exception as e:
                self.logger.error(f"Error in periodic send loop: {e}")
                time.sleep(1)
    
    def stop(self):
        """Stop the IO tunnel service."""
        if not self.running:
            return
            
        self.logger.info("Stopping IO Tunnel service...")
        self.running = False
        
        # Stop watchdog
        if self.watchdog:
            self.watchdog.stop_watchdog()

        # Close GPIO handles 
        if self.gpio_handler:
            self.gpio_handler.close_handles()
        
        # Stop all threads
        if self.thread_manager:
            self.thread_manager.stop_all(timeout=3.0)
        
        # Close sockets
        if self.udp_socket:
            try:
                self.udp_socket.close()
            except Exception:
                pass
        
        # Send final syslog message and close
        if self.syslog_handler:
            self.syslog_handler.send_syslog("IO Tunnel service stopped")
            self.syslog_handler.close()
        
        self.logger.info("IO Tunnel service stopped")


def main():
    """Main entry point."""
    app = IOTunnelServer()
    try:
        app.start()
    except KeyboardInterrupt:
        print("\nShutdown requested by user")
    except Exception as e:
        logging.error(f"Application error: {e}", exc_info=True)
        sys.exit(1)
    finally:
        app.stop()


if __name__ == '__main__':
    main()